//
// Pie.cs
//
// Defines the Pie tool plug-in to create a plate made up of pie slices
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
// texAlignTop - Flag to indicate that the top faces should align
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Revision History:
// July 13, 2005		David Wyand		Created script file
// October 24, 2015	HiGuy Smith		Added "center top" field
// December 5, 2015	HiGuy Smith		Fixed texture points on the inside hollowed face
//

package Pie
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Pie: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Pie_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %Pie = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %Pie.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%Pie, %static);

      //*** Setup some additional attributes for the cube instance
      %Pie.dirty = tool.DIRTY_NONE();
      %Pie.active = false;
      %Pie.update = tool.EDIT_DONOTHING();
      %Pie.changeCenter = false;
      %Pie.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %Pie;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Pie: Done(" @ %inst @ "," @ %static @ ")");

      %Pie = %inst.instance;
      
      if(%Pie)
      {         
         //*** Delete our instance
         %Pie.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Pie: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Pie: HandleCount(" @ %inst @ "," @ %event @ ")");

      %Pie = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %Pie.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Pie: HandleInit(" @ %inst @ "," @ %event @ ")");

      %Pie = %inst.instance;

      //*** Make the tool active
      if(!%Pie.active)
      {
         %Pie.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%Pie, %Pie.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Pie: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %Pie = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%Pie, %Pie.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Pie: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %Pie = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%Pie, %Pie.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %Pie.dirty = tool.DIRTY_APPEARANCE();
      %Pie.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Pie: Dirty(" @ %inst @ ")");

      %Pie = %inst.instance;
      
      return %Pie.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Pie: Draw(" @ %inst @ "," @ %draw @ ")");

      %Pie = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%Pie.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%Pie, %Pie.static, %draw);

      //*** Indicate that we've drawn the tool
      %Pie.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Pie: CheckEditAction(" @ %inst @ ")");

      %Pie = %inst.instance;
      
      return %Pie.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Pie: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %Pie = %inst.instance;
      
      %Pie.update = tool.EDIT_DONOTHING();
      %Pie.active = false;
      %Pie.changeSize = false;
      %Pie.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Pie: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %Pie = %inst.instance;
      
      // Work on the actual geometry.
      Pie_MakeGeometry(%Pie, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %Pie.update = tool.EDIT_DONOTHING();
      %Pie.changeSize = false;
      %Pie.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Pie: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %Pie = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %Pie.update = tool.EDIT_UPDATE();
            %Pie.active = true;
            %Pie.dirty = tool.DIRTY_APPEARANCE();
            %Pie.changeSize = true;
            %Pie.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Pie_DefaultValues(%Pie.static);
            ToolBB::rebuildBoundingBox(%Pie, %Pie.static);
            %Pie.update = tool.EDIT_UPDATE();
            %Pie.active = true;
            %Pie.dirty = tool.DIRTY_APPEARANCE();
            %Pie.changeSize = true;
            %Pie.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%Pie.active)
            {
               %Pie.update = tool.EDIT_REJECT();
            }
            %Pie.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%Pie.active)
            {
               %Pie.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Pie: Interface(" @ %inst @ "," @ %form @ ")");

      %Pie = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Simple Box");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      
      %form.addField(-1,  "Geometry" ,"divider");
      %form.addField( 2,  "Segments", "numericinteger");
      %form.addField( 10, "Start",    "numericinteger");
      %form.addField( 11, "End",      "numericinteger");
      %form.addField( 12, "Hollow center", "checkbox");
      %form.addField( 14, "Center Size X",   "numeric");
      %form.addField( 18, "Center Size Y",   "numeric");
      %form.addField( 16, "Center Top Adjust", "numeric");
      %form.addField( 17, "Center Bottom Adjust", "numeric");
      %form.addField( 19, "Outside slope", "numeric");
      %form.addField( 20, "Inside slope", "numeric");
      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField(15, "Segment",  "checkbox");
      %form.addField(21, "Align Top", "checkbox");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      %form.setFieldMinLimit(2, 3);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 0);
      %form.setFieldMinLimit(11, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Pie: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %Pie = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %Pie.static.center[0] SPC %Pie.static.center[1] SPC %Pie.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %Pie.static.size[0] SPC %Pie.static.size[1] SPC %Pie.static.size[2];
            return %value;
      
         //*** Handle the 'Segments' field
         case 2:
            %value = %Pie.static.segments;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %Pie.static.sides;
            return %value;
      
         //*** Handle the 'Hollow' field
         case 12:
            %value = %Pie.static.hollow;
            return %value;
            
         //*** Handle the 'center size' field
         case 14:
            %value = %Pie.static.centersizeX;
            return %value;
         //*** Handle the 'center size' field
         case 18:
            %value = %Pie.static.centersizeY;
            return %value;
                        
         //*** Handle the 'U Scale' field
         case 5:
            %value = %Pie.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %Pie.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %Pie.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %Pie.static.voffset;
            return %value;            
         case 10:
            %value = %Pie.static.startseg;
            return %value;
         case 11:
            %value = %Pie.static.endseg;
            return %value;
         case 15:
            %value = %Pie.static.segmenttexture;
            return %value;
         case 16:
            %value = %Pie.static.centertopadjust;
            return %value;
         case 17:
            %value = %Pie.static.centerbottomadjust;
            return %value;
         case 19:
            %value = %Pie.static.outsideslope;
            return %value;
         case 20:
            %value = %Pie.static.insideslope;
            return %value;
         case 21:
            %value = %Pie.static.texAlignTop;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Pie: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %Pie = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %Pie.static.center[%i] = getWord(%value, %i);
            }
            %Pie.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %Pie.static.size[%i] = getWord(%value, %i);
            }
            %Pie.changeSize = true;
      
         //*** Handle the Segments field
         case 2:
            %Pie.static.segments = %value;
            %Pie.static.endseg = %value - 1;
            %Pie.changeSize = true;
      
         //*** Handle the Sides field
         case 3:
            %Pie.static.sides = %value;
            %Pie.changeSize = true;
      
         //*** Handle the constrain field
         case 4:
            %Pie.static.constrain = %value;
            %Pie.changeSize = true;
            
         //*** Handle the hollow field
         case 12:
            %Pie.static.hollow = %value;
            %Pie.changeSize = true;
            
         //*** Handle the 'center size' field
         case 14:
            %Pie.static.centersizeX = %value;
            %Pie.changeSize = true;
                //*** Handle the 'center size' field
         case 18:
            %Pie.static.centersizeY = %value;
            %Pie.changeSize = true;     
         //*** Handle the 'U Scale' field
         case 5:
            %Pie.static.uscale = %value;
            %Pie.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %Pie.static.vscale = %value;
            %Pie.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %Pie.static.uoffset = %value;
            %Pie.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %Pie.static.voffset = %value;
            %Pie.changeSize = true;            
         case 10:
            %Pie.static.startseg = %value;
            %Pie.changeSize = true;            
         case 11:
            %Pie.static.endseg = %value;
            %Pie.changeSize = true;            
         case 15:
            %Pie.static.forceclear = 1;
            %Pie.static.segmenttexture = %value;
            %Pie.changeSize = true;            
            
         case 16:
            %Pie.static.centertopadjust = %value;
            %Pie.changeSize = true;            
         case 17:
            %Pie.static.centerbottomadjust = %value;
            %Pie.changeSize = true;            
         case 19:
            %Pie.static.outsideslope = %value;
            %Pie.changeSize = true;            
         case 20:
            %Pie.static.insideslope = %value;
            %Pie.changeSize = true;            
         case 21:
            %Pie.static.texAlignTop = %value;
            %Pie.changeSize = true;
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%Pie.active)
      {
         %Pie.active = true;
         %Pie.changeSize = true;
         %Pie.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %Pie.update = tool.EDIT_UPDATE();
      %Pie.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%Pie, %Pie.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function Pie_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;      
      
      %obj.segments  = 16;
      %obj.sides     = 8;
      %obj.constrain = 1;
      %obj.centersizeX = 0.1;
      %obj.centersizeY = 0.1;
      %obj.hollow   = 0;
      %obj.startseg  = 0;
      %obj.endseg    = %obj.segments -1;
      %obj.segmenttexture = 0;
      %obj.forceclear = false;
      %obj.centertopadjust = 0.0;
      %obj.centerbottomadjust = 0.0;
      %obj.outsideslope = 0.0;
      %obj.insideslope = 0.0;

      %obj.texAlignTop = false;
   }

   //*** Init the static object
   function Pie_InitStatic(%static)
   {
      //*** Setup default values
      Pie_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Pie_ResetCube(%Pie)
   {
      //*** Retrieve a pointer to the static data
      %static = %Pie.static;
      
      //*** Setup default values
      Pie_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%Pie, %Pie.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }


   function GetPiePoint(%Pie, %SliceNumber, %XRadius, %YRadius, %TZ)
   {
      %radians = 2.0 * 3.14159265358979323846;

      %angle = %SliceNumber / %Pie.static.segments * %radians;
      
      //%rx = %XRadius*mCos(-%angle) - %YRadius*mSin(-%angle); // Rotate
      //%ry = %XRadius*mSin(-%angle) + %YRadius*mCos(-%angle);
      
      %rx = %XRadius*mCos(-%angle);
      %ry = %YRadius*mSin(-%angle);

      %point = NewVector(%rx,%ry,%TZ);
      return %point;      
   }
      
      
   //*** Build/modify the actual geometry
   function Pie_MakeGeometry(%Pie, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%Pie.active)
         return;

      if (%Pie.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %Pie.static.forceclear = 0;
      }
      // Clear brushes if the number has changed.
      %count = %edit.getEditBrushCount();
      %brushcount = (%Pie.static.endseg - %Pie.static.startseg) + 1;
      if(%count != %brushcount)
      {
         %edit.clearAllNewBrushes();
      } 

      //*** Work on the sizing of the brush, if appropriate
      //if(%Pie.changeSize == true)
      //{
         %center = %Pie.static.center[0] SPC %Pie.static.center[1] SPC %Pie.static.center[2];
         
         for (%inx = %Pie.static.startseg; %inx <= %Pie.static.endseg; %inx++)
         {
            %tx = %Pie.static.size[0] * 0.5;
            %ty = %Pie.static.size[1] * 0.5;
            %tz = %Pie.static.size[2] * 0.5;
            
            %BrushNumber = %inx - %Pie.static.startseg;
            
            %brush = GetBrush(%edit,%BrushNumber);
            
            //*** Setup texture parameters
            %brush.setTextureOffset(%Pie.static.uoffset, %Pie.static.voffset);
            %brush.setTextureScale(%Pie.static.uscale, %Pie.static.vscale);
            
            %brush.clearAllPlanes();
            %brush.setOrigin(%center);
            
            %nextinx = %inx+1;
            if (%nextinx >= %Pie.static.segments)
            {
               %nextinx = 0;
            }
            

            // TODO: Adjust center points
            // based on offset 
            %ctz = %tz + %Pie.static.centertopadjust;
            %cbz = (0-%tz) + %Pie.static.centerbottomadjust;
            
            %oz = (%inx * %Pie.static.outsideslope);
            %nextoz = %oz + %Pie.static.outsideslope; // Don't wrap sloping!
            
            %CenterTop    = NewVector(0,0,%ctz+%oz);
            %CenterBottom = NewVector(0,0,%cbz+%nextoz);
            
            // TODO: Adjust outside point based on offset
            //       per index. 
            %ppt1 = GetPiePoint(%Pie,%inx,%tx,%ty,%tz+%oz);
            %ppb1 = GetPiePoint(%Pie,%inx,%tx,%ty,(-%tz)+%oz);
            %ppt2 = GetPiePoint(%Pie,%nextinx,%tx,%ty,%tz+%nextoz);
            %ppb2 = GetPiePoint(%Pie,%nextinx,%tx,%ty,(-%tz)+%nextoz);

            // Add top
            if (%Pie.static.texAlignTop) {
               %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
            } else {
               %brush.setTexturePlanesByPoints(%CenterTop,%ppt1,%ppt2,0.0,0.0);
            }
            %brush.addPlaneByPoints(%CenterTop,%ppt1,%ppt2);
            
            // Add bottom
            if (%Pie.static.texAlignTop) {
               %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
            } else {
               %brush.setTexturePlanesByPoints(%CenterTop,%ppb1,%ppb2,0.0,0.0);
            }
            %brush.addPlaneByPoints(%CenterBottom,%ppb2,%ppb1);

            // Add far plane
            %brush.setTexturePlanesByPoints(%ppt2,%ppt1,%ppb1,0.0,0.0);
            %brush.addPlaneByPoints(%ppt2,%ppt1,%ppb1);

            // Add near Segment plane
            %brush.setTexturePlanesByPoints(%ppt1,%CenterTop,%CenterBottom,0.0,0.0);
            %brush.addPlaneByPoints(%ppt1,%CenterTop,%CenterBottom);
            
            // Add far Segment plane
            %brush.setTexturePlanesByPoints(%ppt1,%CenterTop,%CenterBottom,0.0,0.0);
            %brush.addPlaneByPoints(%CenterBottom,%CenterTop, %ppt2);
            
            // Add the near plane (if there is one...)
            if (%Pie.static.hollow)
            {
               %ptx = %Pie.static.centersizeX;
               %pty = %Pie.static.centersizeY;
               
               %iz = (%inx * %Pie.static.insideslope);
               %nextiz = %iz + %Pie.static.insideslope; // Don't wrap sloping!               
               
               %pct1 = GetPiePoint(%Pie,%inx,%ptx,%pty,%ctz+%iz);
               %pct2 = GetPiePoint(%Pie,%nextinx,%ptx,%pty,%ctz+%nextiz);
               %pcb1 = GetPiePoint(%Pie,%inx,%ptx,%pty,%cbz+%iz);
               
               %brush.setTexturePlanesByPoints(%pct2,%pct1,%pcb1,0.0,0.0);
               %brush.addPlaneByPoints(%pct1,%pct2,%pcb1);
               
            }
            
            
            %edit.updateBrush(%brush);
         }
   }
   
};

tool.register("Pie", tool.typeInteractive(), tool.RFLAG_NONE(), "Create Pie" );
tool.setToolProperty("Pie", "Icon", "icons/Pie");
tool.setToolProperty("Pie", "Group", "Create");
