//
// Ziggurat.cs
//
// Defines the Ziggurat tool plug-in to create a Ziggurat:
//
//         XXX
//       XXXXXXX
//    XXXXXXXXXXXXX
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Sections
// Sides (3+)
// top percentage (10 - 100)
//
// 


package Ziggurat
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Ziggurat: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Ziggurat_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %Ziggurat = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %Ziggurat.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%Ziggurat, %static);

      //*** Setup some additional attributes for the cube instance
      %Ziggurat.dirty = tool.DIRTY_NONE();
      %Ziggurat.active = false;
      %Ziggurat.update = tool.EDIT_DONOTHING();
      %Ziggurat.changeCenter = false;
      %Ziggurat.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %Ziggurat;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Ziggurat: Done(" @ %inst @ "," @ %static @ ")");

      %Ziggurat = %inst.instance;
      
      if(%Ziggurat)
      {         
         //*** Delete our instance
         %Ziggurat.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Ziggurat: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Ziggurat: HandleCount(" @ %inst @ "," @ %event @ ")");

      %Ziggurat = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %Ziggurat.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Ziggurat: HandleInit(" @ %inst @ "," @ %event @ ")");

      %Ziggurat = %inst.instance;

      //*** Make the tool active
      if(!%Ziggurat.active)
      {
         %Ziggurat.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%Ziggurat, %Ziggurat.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Ziggurat: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %Ziggurat = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%Ziggurat, %Ziggurat.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Ziggurat: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %Ziggurat = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%Ziggurat, %Ziggurat.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %Ziggurat.dirty = tool.DIRTY_APPEARANCE();
      %Ziggurat.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Ziggurat: Dirty(" @ %inst @ ")");

      %Ziggurat = %inst.instance;
      
      return %Ziggurat.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Ziggurat: Draw(" @ %inst @ "," @ %draw @ ")");

      %Ziggurat = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%Ziggurat.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%Ziggurat, %Ziggurat.static, %draw);

      //*** Indicate that we've drawn the tool
      %Ziggurat.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Ziggurat: CheckEditAction(" @ %inst @ ")");

      %Ziggurat = %inst.instance;
      
      return %Ziggurat.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Ziggurat: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %Ziggurat = %inst.instance;
      
      %Ziggurat.update = tool.EDIT_DONOTHING();
      %Ziggurat.active = false;
      %Ziggurat.changeSize = false;
      %Ziggurat.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Ziggurat: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %Ziggurat = %inst.instance;
      
      // Work on the actual geometry.
      Ziggurat_MakeGeometry(%Ziggurat, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %Ziggurat.update = tool.EDIT_DONOTHING();
      %Ziggurat.changeSize = false;
      %Ziggurat.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Ziggurat: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %Ziggurat = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %Ziggurat.update = tool.EDIT_UPDATE();
            %Ziggurat.active = true;
            %Ziggurat.dirty = tool.DIRTY_APPEARANCE();
            %Ziggurat.changeSize = true;
            %Ziggurat.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Ziggurat_DefaultValues(%Ziggurat.static);
            ToolBB::rebuildBoundingBox(%Ziggurat, %Ziggurat.static);
            %Ziggurat.update = tool.EDIT_UPDATE();
            %Ziggurat.active = true;
            %Ziggurat.dirty = tool.DIRTY_APPEARANCE();
            %Ziggurat.changeSize = true;
            %Ziggurat.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%Ziggurat.active)
            {
               %Ziggurat.update = tool.EDIT_REJECT();
            }
            %Ziggurat.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%Ziggurat.active)
            {
               %Ziggurat.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Ziggurat: Interface(" @ %inst @ "," @ %form @ ")");

      %Ziggurat = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Create Ziggurat");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      
      %form.addField(-1,  "Geometry" ,"divider");
      %form.addField( 2,  "Segments", "numericinteger");
      %form.addField( 9,  "Rectangular", "checkbox");
      %form.addField( 3,  "Sides",    "numericinteger");
      %form.addField( 4,  "Top %",    "numericinteger");
      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      %form.setFieldMinLimit(2, 2);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 0);
      %form.setFieldMinLimit(11, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Ziggurat: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %Ziggurat = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %Ziggurat.static.center[0] SPC %Ziggurat.static.center[1] SPC %Ziggurat.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %Ziggurat.static.size[0] SPC %Ziggurat.static.size[1] SPC %Ziggurat.static.size[2];
            return %value;
      
         //*** Handle the 'Segments' field
         case 2:
            %value = %Ziggurat.static.segments;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %Ziggurat.static.sides;
            return %value;
      
         //*** Handle the 'Constrain' field
         case 4:
            %value = %Ziggurat.static.percent;
            return %value;
            
         //*** Handle the 'U Scale' field
         case 5:
            %value = %Ziggurat.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %Ziggurat.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %Ziggurat.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %Ziggurat.static.voffset;
            return %value;            
         case 9:
            %value = %Ziggurat.static.rectangular;
            return %value;            
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Ziggurat: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %Ziggurat = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %Ziggurat.static.center[%i] = getWord(%value, %i);
            }
            %Ziggurat.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %Ziggurat.static.size[%i] = getWord(%value, %i);
            }
            %Ziggurat.changeSize = true;
      
         //*** Handle the Segments field
         case 2:
            %Ziggurat.static.segments = %value;
            %Ziggurat.static.forceclear = 1;
            %Ziggurat.changeSize = true;
      
         //*** Handle the Sides field
         case 3:
            %Ziggurat.static.sides = %value;
            %Ziggurat.changeSize = true;
      
         //*** Handle the constrain field
         case 4:
            %Ziggurat.static.percent = %value;
            %Ziggurat.changeSize = true;
            
         //*** Handle the 'U Scale' field
         case 5:
            %Ziggurat.static.uscale = %value;
            %Ziggurat.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %Ziggurat.static.vscale = %value;
            %Ziggurat.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %Ziggurat.static.uoffset = %value;
            %Ziggurat.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %Ziggurat.static.voffset = %value;
            %Ziggurat.changeSize = true;      
         case 9:
            %Ziggurat.static.rectangular = %value;
            %Ziggurat.changeSize = true;      
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%Ziggurat.active)
      {
         %Ziggurat.active = true;
         %Ziggurat.changeSize = true;
         %Ziggurat.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %Ziggurat.update = tool.EDIT_UPDATE();
      %Ziggurat.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%Ziggurat, %Ziggurat.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function Ziggurat_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;      
      
      %obj.segments  = 4;
      %obj.sides     = 4;
      %obj.percent   = 50;
      %obj.rectangular = 1;
      %obj.forceclear = false;
   }

   //*** Init the static object
   function Ziggurat_InitStatic(%static)
   {
      //*** Setup default values
      Ziggurat_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Ziggurat_ResetCube(%Ziggurat)
   {
      //*** Retrieve a pointer to the static data
      %static = %Ziggurat.static;
      
      //*** Setup default values
      Ziggurat_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%Ziggurat, %Ziggurat.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   function GetFaceCenter(%Ziggurat,%slicenumber)
   {
      %radians = 6.283185307;
      %tx = %Ziggurat.static.size[0] * 0.5;
      %ty = %Ziggurat.static.size[1] * 0.5;
      %tz = %Ziggurat.static.size[2] * 0.5;      
      %tz = 50;
      %ti = (%slicenumber / %Ziggurat.static.segments) * %radians;
      %center = NewVector(mCos(%ti) * %tx, mSin(%ti) * %ty, %tz);
      return %center;
   }
   
   function GetFacePoint(%Ziggurat,%edgenumber,%rx, %ry, %zpos)
   {
      %radians = 6.283185307; // close enough.
      %angle = (%edgenumber / %Ziggurat.static.sides) * %radians;
      %point = NewVector(mCos(%angle) * %rx, mSin(%angle) * %ry, %zpos);
      return %point;
   }      
      
   function MakeSolid(%edit,%Ziggurat, %mins, %maxs, %tx,%ty,%tz)
   {
      %brush = GetBrush(%edit,%Ziggurat.currentbrush);
      %Ziggurat.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      %rx = (%maxx - %minx) * 0.5;
      %ry = (%maxy - %miny) * 0.5;
      %center = %Ziggurat.static.center[0] SPC %Ziggurat.static.center[1] SPC %Ziggurat.static.center[2];
      
      for (%face = 0; %face < %Ziggurat.static.sides; %face++)
      {
         if(%face == %Ziggurat.static.sides-1)
         {
            %nextface = 0;
         }
         else
         {
            %nextface = %face+1;
         }
         // calc the points for the face.
         %pt1 = VectorAdd(%center,GetFacePoint(%Ziggurat,%face,%rx,%ry,%minz));
         %pt2 = VectorAdd(%center,GetFacePoint(%Ziggurat,%nextface,%rx,%ry,%minz));
         %pt3 = VectorAdd(%center,GetFacePoint(%Ziggurat,%face,%rx,%ry,%maxz));
         
         %brush.setTexturePlanesByPoints(%pt2,%pt1,%pt3, 0.0, 0.0);
         %brush.addPlaneByPoints(%pt2,%pt1,%pt3);
         // Set the texture parms.
         // Add the face to the brush.
      }
      
      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
      
   }
   // MakeCuboid
   function MakeCuboid(%edit,%Ziggurat, %mins, %maxs, %tx,%ty,%tz)
   {
      %brush = GetBrush(%edit,%Ziggurat.currentbrush);
      %Ziggurat.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      
      %brush.setTextureOffset(%Ziggurat.static.uoffset, %Ziggurat.static.voffset);
      %brush.setTextureScale(%Ziggurat.static.uscale, %Ziggurat.static.vscale);     
        
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%maxx));
      
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ 1.0*(%minx));

      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%maxy));
         
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);   
      %brush.addPlane("0 -1 0 " @ 1.0*(%miny));

      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
   }
      
      
   //*** Build/modify the actual geometry
   function Ziggurat_MakeGeometry(%Ziggurat, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%Ziggurat.active)
         return;

      %Ziggurat.currentbrush = 0;
      if (%Ziggurat.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %Ziggurat.static.forceclear = 0;
      }
   
      //*** Work on the sizing of the brush, if appropriate
      %center = %Ziggurat.static.center[0] SPC %Ziggurat.static.center[1] SPC %Ziggurat.static.center[2];
      %tx = %Ziggurat.static.size[0] * 0.5;
      %ty = %Ziggurat.static.size[1] * 0.5;
      %tz = %Ziggurat.static.size[2] * 0.5;

      %range = 100-%Ziggurat.static.percent;
      %shrink = %range / (%Ziggurat.static.segments-1);
      %currentsize = 100;
      
      %mins = NewVector(%Ziggurat.static.center[0]-%tx,%Ziggurat.static.center[1]-%ty,%Ziggurat.static.center[2]-%tz);
      %maxs = NewVector(%Ziggurat.static.center[0]+(%tx),%Ziggurat.static.center[1]+%ty,%Ziggurat.static.center[2]+%tz);
      %sz = %Ziggurat.static.size[2];
      %ht =  %sz / %Ziggurat.static.segments;
      %cmins = %mins;
      %cmaxs = setWord(%maxs,2,getWord(%mins,2)+%ht);
      %lastx = %tx;
      %lasty = %ty;
      
      for (%inx = 0; %inx < %Ziggurat.static.segments; %inx++)
      {
         if (%Ziggurat.static.rectangular)
         {
            // make a cuboid
            MakeCuboid(%edit,%Ziggurat,%cmins,%cmaxs,%tx,%ty,%tz);
            
            // get the size.
            // shrink it.
            // create new minx, miny, maxx, maxy.
            
         }
         else
         {
            // Make a flat solid
            MakeSolid(%edit,%Ziggurat,%cmins,%cmaxs,%tx,%ty,%tz);
         }
         %currentsize = %currentsize - %shrink;
         %sizex = (%lastx * %currentsize) / 100.0;
         %sizey = (%lasty * %currentsize) / 100.0;
         %minx = %Ziggurat.static.center[0]-%sizex;
         %maxx = %Ziggurat.static.center[0]+%sizex;
         %miny = %Ziggurat.static.center[1]-%sizey;
         %maxy = %Ziggurat.static.center[1]+%sizey;
         %minz = getWord(%cmaxs,2);
         %maxz = %minz + %ht;
         %cmins = NewVector(%minx,%miny,%minz);
         %cmaxs = NewVector(%maxx,%maxy,%maxz);
      }
   }
};

tool.register("Ziggurat", tool.typeInteractive(), tool.RFLAG_NONE(), "Create Ziggurat" );
tool.setToolProperty("Ziggurat", "Icon", "icons/Ziggurat");
tool.setToolProperty("Ziggurat", "Group", "Create");
