//
// Torus.cs
//
// Defines the Torus tool plug-in to create a bare-bones cube.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Revision History:
// July 13, 2005		David Wyand		Created script file
//

package Torus
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Torus: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Torus_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %torus = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %torus.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%torus, %static);

      //*** Setup some additional attributes for the cube instance
      %torus.dirty = tool.DIRTY_NONE();
      %torus.active = false;
      %torus.update = tool.EDIT_DONOTHING();
      %torus.changeCenter = false;
      %torus.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %torus;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Torus: Done(" @ %inst @ "," @ %static @ ")");

      %torus = %inst.instance;
      
      if(%torus)
      {         
         //*** Delete our instance
         %torus.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Torus: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Torus: HandleCount(" @ %inst @ "," @ %event @ ")");

      %torus = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %torus.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Torus: HandleInit(" @ %inst @ "," @ %event @ ")");

      %torus = %inst.instance;

      //*** Make the tool active
      if(!%torus.active)
      {
         %torus.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%torus, %torus.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Torus: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %torus = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%torus, %torus.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Torus: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %torus = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%torus, %torus.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %torus.dirty = tool.DIRTY_APPEARANCE();
      %torus.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Torus: Dirty(" @ %inst @ ")");

      %torus = %inst.instance;
      
      return %torus.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Torus: Draw(" @ %inst @ "," @ %draw @ ")");

      %torus = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%torus.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%torus, %torus.static, %draw);

      //*** Indicate that we've drawn the tool
      %torus.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Torus: CheckEditAction(" @ %inst @ ")");

      %torus = %inst.instance;
      
      return %torus.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Torus: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %torus = %inst.instance;
      
      %torus.update = tool.EDIT_DONOTHING();
      %torus.active = false;
      %torus.changeSize = false;
      %torus.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Torus: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %torus = %inst.instance;
      
      // Work on the actual geometry.
      Torus_MakeGeometry(%torus, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %torus.update = tool.EDIT_DONOTHING();
      %torus.changeSize = false;
      %torus.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Torus: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %torus = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %torus.update = tool.EDIT_UPDATE();
            %torus.active = true;
            %torus.dirty = tool.DIRTY_APPEARANCE();
            %torus.changeSize = true;
            %torus.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Torus_DefaultValues(%torus.static);
            ToolBB::rebuildBoundingBox(%torus, %torus.static);
            %torus.update = tool.EDIT_UPDATE();
            %torus.active = true;
            %torus.dirty = tool.DIRTY_APPEARANCE();
            %torus.changeSize = true;
            %torus.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%torus.active)
            {
               %torus.update = tool.EDIT_REJECT();
            }
            %torus.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%torus.active)
            {
               %torus.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Torus: Interface(" @ %inst @ "," @ %form @ ")");

      %torus = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Simple Box");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      
      %form.addField(-1,  "Geometry" ,"divider");
      %form.addField( 2,  "Segments", "numericinteger");
      %form.addField( 3,  "Sides",    "numericinteger");
      %form.addField( 10, "Start",    "numericinteger");
      %form.addField( 11, "End",      "numericinteger");
      %form.addField( 13, "Options",  "divider");
      %form.addField(  4, "Constrain","checkbox");
      %form.addField( 12, "Tubular",  "checkbox");
      %form.addField( 14, "2P/Face",  "checkbox");
      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField(15, "Segment",  "checkbox");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      %form.setFieldMinLimit(2, 3);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 0);
      %form.setFieldMinLimit(11, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Torus: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %torus = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %torus.static.center[0] SPC %torus.static.center[1] SPC %torus.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %torus.static.size[0] SPC %torus.static.size[1] SPC %torus.static.size[2];
            return %value;
      
         //*** Handle the 'Segments' field
         case 2:
            %value = %torus.static.segments;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %torus.static.sides;
            return %value;
      
         //*** Handle the 'Constrain' field
         case 4:
            %value = %torus.static.constrain;
            return %value;
            
         //*** Handle the 'Tubular' field
         case 12:
            %value = %torus.static.tubular;
            return %value;
         case 14:
            %value = %torus.static.multiplane;
            return %value;
                        
         //*** Handle the 'U Scale' field
         case 5:
            %value = %torus.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %torus.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %torus.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %torus.static.voffset;
            return %value;            
         case 10:
            %value = %torus.static.startseg;
            return %value;
         case 11:
            %value = %torus.static.endseg;
            return %value;
         case 15:
            %value = %torus.static.segmenttexture;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Torus: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %torus = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %torus.static.center[%i] = getWord(%value, %i);
            }
            %torus.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %torus.static.size[%i] = getWord(%value, %i);
            }
            %torus.changeSize = true;
      
         //*** Handle the Segments field
         case 2:
            %torus.static.segments = %value;
            %torus.static.endseg = %value - 1;
            %torus.changeSize = true;
      
         //*** Handle the Sides field
         case 3:
            %torus.static.sides = %value;
            %torus.changeSize = true;
      
         //*** Handle the constrain field
         case 4:
            %torus.static.constrain = %value;
            %torus.changeSize = true;
            
         //*** Handle the tubular field
         case 12:
            %torus.static.tubular = %value;
            %torus.changeSize = true;
            
         case 14:
            %torus.static.multiplane = %value;
            %torus.changeSize = true;
            
         //*** Handle the 'U Scale' field
         case 5:
            %torus.static.uscale = %value;
            %torus.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %torus.static.vscale = %value;
            %torus.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %torus.static.uoffset = %value;
            %torus.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %torus.static.voffset = %value;
            %torus.changeSize = true;            
         case 10:
            %torus.static.startseg = %value;
            %torus.changeSize = true;            
         case 11:
            %torus.static.endseg = %value;
            %torus.changeSize = true;            
         case 15:
            %torus.static.forceclear = 1;
            %torus.static.segmenttexture = %value;
            %torus.changeSize = true;            
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%torus.active)
      {
         %torus.active = true;
         %torus.changeSize = true;
         %torus.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %torus.update = tool.EDIT_UPDATE();
      %torus.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%torus, %torus.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function Torus_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;      
      
      %obj.segments  = 16;
      %obj.sides     = 8;
      %obj.constrain = 1;
      %obj.multiplane= 0;
      %obj.tubular   = 0;
      %obj.startseg  = 0;
      %obj.endseg    = %obj.segments -1;
      %obj.segmenttexture = 0;
      %obj.forceclear = false;
   }

   //*** Init the static object
   function Torus_InitStatic(%static)
   {
      //*** Setup default values
      Torus_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Torus_ResetCube(%torus)
   {
      //*** Retrieve a pointer to the static data
      %static = %torus.static;
      
      //*** Setup default values
      Torus_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%torus, %torus.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   function GetFaceCenter(%torus,%slicenumber)
   {
      %radians = 6.283185307;
      %tx = %torus.static.size[0] * 0.5;
      %ty = %torus.static.size[1] * 0.5;
      %tz = %torus.static.size[2] * 0.5;      
      %tz = 50;
      %ti = (%slicenumber / %torus.static.segments) * %radians;
      %center = NewVector(mCos(%ti) * %tx, mSin(%ti) * %ty, %tz);
      return %center;
   }
   
   function GetFacePoint_Unit(%torus, %slicenumber, %pointnumber)
   {
      %radians = 6.283185307;

      %tx = %torus.static.size[0] * 0.5;
      %ty = %torus.static.size[1] * 0.5;
      %tz = %torus.static.size[2] * 0.5;
      //%tz = 0;
      
      %ti = (%slicenumber / %torus.static.segments) * %radians;
      %tj = (%pointnumber / %torus.static.sides) * %radians;

     // %center = GetFaceCenter(%torus,%slicenumber);
      
      %point = NewVector((mCos(%tj) + %tx) * (mCos(%ti)),(mCos(%tj) + %ty) * (mSin(%ti)),%tz * mSin(%tj));
      return %point;
   }
   
   function GetFacePoint_Tube(%torus, %slicenumber, %pointnumber)
   {
      %radians = 6.283185307;

      %tx = %torus.static.size[0] * 0.5;
      %ty = %torus.static.size[1] * 0.5;
      %tz = %torus.static.size[2] * 0.5;
      
      %ti = (%slicenumber / %torus.static.segments) * %radians;
      %tj = (%pointnumber / %torus.static.sides) * %radians;
      
      if (%torus.static.constrain)
      {
         if (%tx > %ty)
           %tx = %ty;
         else
           %ty = %tx;
      }

      
      %point = NewVector((mCos(%tj) + %tx) * (mCos(%ti)),(mCos(%tj) + %ty) * (mSin(%ti)),%tz * mSin(%tj));
      return %point;
   }   

   function GetFacePoint(%torus, %slicenumber, %pointnumber)
   {
      
      if (%torus.static.tubular == 1) 
         return GetFacePoint_Tube(%torus,%slicenumber,%pointnumber);
         
      %radians = 6.283185307;

      %tx = %torus.static.size[0] * 0.5;
      %ty = %torus.static.size[1] * 0.5;
      %tz = %torus.static.size[2] * 0.5;
      
      if (%torus.static.constrain)
      {
         if (%tx > %ty)
           %con = %ty;
         else
           %con = %tx;
           
         if (%tz > (%con / 2.1)) %tz = (%con / 2.1);
         %rox = %con - (%tz);
         %roy = %rox;
      }
      else
      {
         if (%tz > (%tx / 2.1)) %tz = (%tx / 2.1);
         if (%tz > (%ty / 2.1)) %tz = (%ty / 2.1);
         %rox = %tx - (%tz);
         %roy = %ty - (%tz);
      }
     
      %ti = %slicenumber / %torus.static.segments * %radians;
      %tj = %pointnumber / %torus.static.sides  * %radians;

      %x = mCos(%ti) * ( %rox + %tz * mCos(%tj) );
      %y = mSin(%ti) * ( %roy + %tz * mCos(%tj) );
      %z = %tz * mSin(%tj);

      //%x = (mCos(%tj) + %tx) * (mCos(%ti));
      //%y = (mCos(%tj) + %ty) * (mSin(%ti));
      //%z = %tz * mSin(%tj);
      
      //%x = (mCos(%tj) + %tx) * (mCos(%ti));
      //%y = (mCos(%tj) + %ty) * (mSin(%ti));
      //%z = %tz * mSin(%tj);
      
      %point = NewVector(%x,%y,%z);
      return %point;
   }
      
   //*** Build/modify the actual geometry
   function Torus_MakeGeometry(%torus, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%torus.active)
         return;

      if (%torus.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %torus.static.forceclear = 0;
      }
      // Clear brushes if the number has changed.
      %count = %edit.getEditBrushCount();
      %brushcount = (%torus.static.endseg - %torus.static.startseg) + 1;
      if(%count != %brushcount)
      {
         %edit.clearAllNewBrushes();
      } 
   
      //*** Work on the sizing of the brush, if appropriate
      //if(%torus.changeSize == true)
      //{
         %center = %torus.static.center[0] SPC %torus.static.center[1] SPC %torus.static.center[2];
         %tx = %torus.static.size[0] * 0.5;
         %ty = %torus.static.size[1] * 0.5;
         %tz = %torus.static.size[2] * 0.5;
         
         for (%inx = %torus.static.startseg; %inx <= %torus.static.endseg; %inx++)
         {
            %brush = GetBrush(%edit,%inx);
            //*** Setup texture parameters
            %brush.setTextureOffset(%torus.static.uoffset, %torus.static.voffset);
            %brush.setTextureScale(%torus.static.uscale, %torus.static.vscale);
            
            %brush.clearAllPlanes();
            %brush.setOrigin(%center);
            
            %nextinx = %inx+1;
            if (%nextinx >= %torus.static.segments)
            {
               %nextinx = 0;
            }
            
            // Add the plane at the start of the segment
            %pnt1 = GetFacePoint(%torus,%inx,2);
            %pnt2 = GetFacePoint(%torus,%inx,1);
            %pnt3 = GetFacePoint(%torus,%inx,0);
            %brush.addPlaneByPoints(%pnt1,%pnt2,%pnt3);
           
            // Add the plane at the end of the segment (reverse winding)
            %pnt1 = GetFacePoint(%torus,%nextinx,0);
            %pnt2 = GetFacePoint(%torus,%nextinx,1);
            %pnt3 = GetFacePoint(%torus,%nextinx,2);
            %brush.addPlaneByPoints(%pnt1,%pnt2,%pnt3);
            
            for(%pnt = 0; %pnt < %torus.static.sides; %pnt++)
            {
               %nextpnt = %pnt+1;
               if (%nextpnt >= %torus.static.sides)
               {
                  %nextpnt = 0;
               }
               %pnt1 = GetFacePoint(%torus,%inx,%pnt);      // Current point on the slice
               %pnt2 = GetFacePoint(%torus,%inx,%nextpnt);  // Next point on the slice
               %pnt3 = GetFacePoint(%torus,%nextinx,%pnt);  // Make the plane by getting the opposite point on the next slice.
               
               // echo("Face" SPC %pnt SPC "   " SPC %pnt1 SPC "   " SPC %pnt2 SPC "   " SPC %pnt3);
               if (%pnt == 0)
               {
                 if (%torus.static.segmenttexture)
                 {
                   %brush.setTexturePlanesByPoints(%pnt1,%pnt2,%pnt3, 0.0, 0.0);
                 }
                // else
                // {
                //    if(%inx == %torus.static.startseg)
               //     {
                //       %brush.setTexturePlanesByPoints(%pnt1,%pnt2,%pnt3, 0.0, 0.0);
                //    }
                // }
               }
               %brush.addPlaneByPoints(%pnt1,%pnt2,%pnt3);
               
               if (%torus.static.multiplane == 1)
               {
                  %pnt4 = GetFacePoint(%torus,%nextinx,%nextpnt);
                  %brush.addPlaneByPoints(%pnt3,%pnt2,%pnt4);
               }
            }
            %edit.updateBrush(%brush);
         }
      //}
  
   }
   
};

tool.register("Torus", tool.typeInteractive(), tool.RFLAG_NONE(), "Create Torus" );
tool.setToolProperty("Torus", "Icon", "icons/torus");
tool.setToolProperty("Torus", "Group", "Create");
