//
// SimpleStairs.cs
//
// Defines the SimpleStairs tool plug-in to create a set of stairs.
// 
// You can specify which direction the stair faces.

package SimpleStairs
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("SimpleStairs: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         SimpleStairs_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %SimpleStairs = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %SimpleStairs.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%SimpleStairs, %static);

      //*** Setup some additional attributes for the cube instance
      %SimpleStairs.dirty = tool.DIRTY_NONE();
      %SimpleStairs.active = false;
      %SimpleStairs.update = tool.EDIT_DONOTHING();
      %SimpleStairs.changeCenter = false;
      %SimpleStairs.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %SimpleStairs;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("SimpleStairs: Done(" @ %inst @ "," @ %static @ ")");

      %SimpleStairs = %inst.instance;
      
      if(%SimpleStairs)
      {         
         //*** Delete our instance
         %SimpleStairs.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("SimpleStairs: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("SimpleStairs: HandleCount(" @ %inst @ "," @ %event @ ")");

      %SimpleStairs = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %SimpleStairs.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("SimpleStairs: HandleInit(" @ %inst @ "," @ %event @ ")");

      %SimpleStairs = %inst.instance;

      //*** Make the tool active
      if(!%SimpleStairs.active)
      {
         %SimpleStairs.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%SimpleStairs, %SimpleStairs.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("SimpleStairs: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %SimpleStairs = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%SimpleStairs, %SimpleStairs.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("SimpleStairs: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %SimpleStairs = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%SimpleStairs, %SimpleStairs.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %SimpleStairs.dirty = tool.DIRTY_APPEARANCE();
      %SimpleStairs.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("SimpleStairs: Dirty(" @ %inst @ ")");

      %SimpleStairs = %inst.instance;
      
      return %SimpleStairs.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("SimpleStairs: Draw(" @ %inst @ "," @ %draw @ ")");

      %SimpleStairs = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%SimpleStairs.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%SimpleStairs, %SimpleStairs.static, %draw);

      //*** Indicate that we've drawn the tool
      %SimpleStairs.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("SimpleStairs: CheckEditAction(" @ %inst @ ")");

      %SimpleStairs = %inst.instance;
      
      return %SimpleStairs.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("SimpleStairs: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %SimpleStairs = %inst.instance;
      
      %SimpleStairs.update = tool.EDIT_DONOTHING();
      %SimpleStairs.active = false;
      %SimpleStairs.changeSize = false;
      %SimpleStairs.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("SimpleStairs: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %SimpleStairs = %inst.instance;
      
      // Work on the actual geometry.
      SimpleStairs_MakeGeometry(%SimpleStairs, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %SimpleStairs.update = tool.EDIT_DONOTHING();
      %SimpleStairs.changeSize = false;
      %SimpleStairs.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("SimpleStairs: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %SimpleStairs = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %SimpleStairs.update = tool.EDIT_UPDATE();
            %SimpleStairs.active = true;
            %SimpleStairs.dirty = tool.DIRTY_APPEARANCE();
            %SimpleStairs.changeSize = true;
            %SimpleStairs.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            SimpleStairs_DefaultValues(%SimpleStairs.static);
            ToolBB::rebuildBoundingBox(%SimpleStairs, %SimpleStairs.static);
            %SimpleStairs.update = tool.EDIT_UPDATE();
            %SimpleStairs.active = true;
            %SimpleStairs.dirty = tool.DIRTY_APPEARANCE();
            %SimpleStairs.changeSize = true;
            %SimpleStairs.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%SimpleStairs.active)
            {
               %SimpleStairs.update = tool.EDIT_REJECT();
            }
            %SimpleStairs.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%SimpleStairs.active)
            {
               %SimpleStairs.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("SimpleStairs: Interface(" @ %inst @ "," @ %form @ ")");

      %SimpleStairs = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Simple Box");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      
      %form.addField(-1,  "Architectual" , "divider");
      %form.addField( 6,  "Step Count", "numericinteger");
      %form.addField(25,  "Step Thickness", "numeric");
      %form.addField(26,  "Carriage", "checkbox");
      %form.addField(27,  "Carriage Width", "numeric");
      
      %form.addField(-1,  "Support",        "divider");
      %form.addField(57,  "Supports",       "checkbox");
      %form.addField(58,  "Support Width",  "numeric");
      %form.addField(59,  "Support Sides",  "numericinteger"); 
      %form.addField(60,  "On Carriage",    "checkbox");
      %form.addField(61,  "Conform",        "checkbox");
      
      %form.addField(-1,  "Balustrade", "divider");
      %form.addField(50,  "Right Balustrade", "checkbox");
      %form.addField(51,  "Left Balustrade", "checkbox");
      %form.addField(52,  "Baluster Height", "numeric");
      %form.addField(53,  "Baluster Width",  "numeric");
      %form.addField(54,  "Baluster inset",  "numeric");
      %form.addField(55,  "Rail width",      "numeric");
      %form.addField(56,  "Rail Height",     "numeric");
      
      %form.addField(-1,  "Stair Type" , "divider");
      %form.addField( 2,  "Stair Type",  "radio");
      %form.addField(-1,  "Step Facing", "divider");
      %form.addField( 4,  "Step Facing","radio"); 
      
      %form.addField( -1, "Position" ,"divider");
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");

      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 102, "U Scale"  ,"numeric");
      %form.addField( 103, "V Scale"  ,"numeric");
      %form.addField( 104, "U Offset" ,"numeric");
      %form.addField( 105, "V Offset" ,"numeric");
      
      %form.setFieldMinLimit(59, 3);
      %form.setFieldMaxLimit(59, 64);

      %form.addFieldListItem(2,"Open"  );
      %form.addFieldListItem(2,"Closed");
      %form.addFieldListItem(2,"Boxed" );
      %form.addFieldListItem(2,"Full"  );
      
      %form.addFieldListItem(4, "North");
      %form.addFieldListItem(4, "East" );
      %form.addFieldListItem(4, "South");
      %form.addFieldListItem(4, "West" )  ;
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("SimpleStairs: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %SimpleStairs = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %SimpleStairs.static.center[0] SPC %SimpleStairs.static.center[1] SPC %SimpleStairs.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %SimpleStairs.static.size[0] SPC %SimpleStairs.static.size[1] SPC %SimpleStairs.static.size[2];
            return %value;
      
         case 2:
            %value = %SimpleStairs.static.StairType;
            return %value;
      
         case 4:
            %value = %SimpleStairs.static.StepDirection;
            return %value;
            
         case 6:
            %value = %SimpleStairs.static.StepCount;
            return %value;
            
         case 25:
            %value = %SimpleStairs.static.StepThickness;
            return %value;
            
         case 26:
            %value = %SimpleStairs.static.Carriage;
            return %value;
            
         case 27:
            %value = %SimpleStairs.static.CarriageWidth;
            return %value;
        
         case 50:
            %value = %SimpleStairs.static.RightBalustrade;
            return %value;
            
         case 51:
            %value = %SimpleStairs.static.LeftBalustrade;
            return %value;
            
         case 52:
            %value = %SimpleStairs.static.BalusterHeight;
            return %value;
            
         case 53:
            %value = %SimpleStairs.static.BalusterWidth;
            return %value;
            
         case 54:
            %value = %SimpleStairs.static.BalusterInset;
            return %value;
            
         case 55:
            %value = %SimpleStairs.static.RailWidth;
            return %value;
            
         case 56:
            %value = %SimpleStairs.static.RailHeight;
            return %value;
            
         case 57:
            %value = %SimpleStairs.static.Supports;
            return %value;
            
         case 58:
            %value = %SimpleStairs.static.SupportWidth;
            return %value;
            
         case 59:    
            %value = %SimpleStairs.static.SupportSides;
            return %value;
            
         case 60:
            %value = %SimpleStairs.static.SupportsOnCarriage;
            return %value;
            
         case 61:
            %value = %SimpleStairs.static.ConformToCarriage;
            return %value;
            
      //%form.addField(57,  "Supports",       "checkbox");
      //%form.addField(58,  "Support Width",  "numeric");
      //%form.addField(59,  "Support Sides",  "numericinteger"); 
      
      
         //*** Handle the 'U Scale' field
         case 102:
            %value = %SimpleStairs.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 103:
            %value = %SimpleStairs.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 104:
            %value = %SimpleStairs.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 105:
            %value = %SimpleStairs.static.voffset;
            return %value;            
            
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("SimpleStairs: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %SimpleStairs = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %SimpleStairs.static.center[%i] = getWord(%value, %i);
            }
            %SimpleStairs.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %SimpleStairs.static.size[%i] = getWord(%value, %i);
            }
            %SimpleStairs.changeSize = true;
      
         case 2:
            %SimpleStairs.static.StairType = %value;
            %SimpleStairs.changeSize = true;
      
         case 4:
            %SimpleStairs.static.StepDirection = %value;
            %SimpleStairs.changeSize = true;
            
         case 6:
            %SimpleStairs.static.StepCount = %value;
            %SimpleStairs.static.forceclear = 1;
            %SimpleStairs.changeSize = true;
                        
         case 25:
            %SimpleStairs.static.StepThickness = %value;
            %SimpleStairs.changeSize = true;
            
         case 26:
            %SimpleStairs.static.Carriage = %value;
            %SimpleStairs.static.forceclear = 1;
            %SimpleStairs.changeSize = true;
            
         case 27:
            %SimpleStairs.static.CarriageWidth = %value;
            %SimpleStairs.changeSize = true;
            
         case 50:
            %SimpleStairs.static.RightBalustrade = %value;
            %SimpleStairs.static.forceclear = 1;
            %SimpleStairs.changeSize = true;
            
         case 51:
            %SimpleStairs.static.LeftBalustrade = %value;
            %SimpleStairs.static.forceclear = 1;
            %SimpleStairs.changeSize = true;
            
         case 52:
            %SimpleStairs.static.BalusterHeight  = %value;
            %SimpleStairs.changeSize = true;
            
         case 53:
            %SimpleStairs.static.BalusterWidth  = %value;
            %SimpleStairs.changeSize = true;
            
         case 54:
            %SimpleStairs.static.BalusterInset = %value;
            %SimpleStairs.changeSize = true;
            
         case 55:
            %SimpleStairs.static.RailWidth = %value;
            %SimpleStairs.changeSize = true;
            
         case 56:
            %SimpleStairs.static.RailHeight = %value;
            %SimpleStairs.changeSize = true;
            
         case 57:
            %SimpleStairs.static.Supports = %value;
            %SimpleStairs.static.forceclear = 1;
            %SimpleStairs.changeSize = true;
            
         case 58:
            %SimpleStairs.static.SupportWidth = %value;
            %SimpleStairs.changeSize = true;
            
         case 59:    
            %SimpleStairs.static.SupportSides = %value;
            %SimpleStairs.changeSize = true;
            
         case 60:
            %SimpleStairs.static.SupportsOnCarriage = %value;
            %SimpleStairs.changeSize = true;

         case 61:
            %SimpleStairs.static.ConformToCarriage = %value;
            %SimpleStairs.changeSize = true;
            
         //*** Handle the 'U Scale' field
         case 102:
            %SimpleStairs.static.uscale = %value;
            %SimpleStairs.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 103:
            %SimpleStairs.static.vscale = %value;
            %SimpleStairs.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 104:
            %SimpleStairs.static.uoffset = %value;
            %SimpleStairs.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 105:
            %SimpleStairs.static.voffset = %value;
            %SimpleStairs.changeSize = true;            
            
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%SimpleStairs.active)
      {
         %SimpleStairs.active = true;
         %SimpleStairs.changeSize = true;
         %SimpleStairs.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %SimpleStairs.update = tool.EDIT_UPDATE();
      %SimpleStairs.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%SimpleStairs, %SimpleStairs.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function SimpleStairs_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;      
      
      %obj.StairType = 0;
      %obj.StepDirection = 0;
      %obj.StepCount = 8;
      %obj.StepThickness = 0.1;
      %obj.Carriage = 0;
      %obj.CarriageWidth = 0.1;
      %obj.ConformToCarriage = false;
      
      %obj.Supports = false;
      %obj.SupportWidth = 0.5;
      %obj.SupportSides = 4;
      
      %obj.RightBalustrade = false;
      %obj.LefttBalustrade = false;
      %obj.BalusterHeight = 1.5;
      %obj.BalusterWidth = 0.1;
      %obj.BalusterInset = 0.0;
      %obj.RailWidth = 0.2;
      %obj.RailHeight = 0.1;
      
      //%obj.Nosing        = 0.05;
      //%obj.OuterRail     = 0;
      //%obj.InnerRail     = 0;
      //%obj.Balusters     = 0;
   }

   //*** Init the static object
   function SimpleStairs_InitStatic(%static)
   {
      //*** Setup default values
      SimpleStairs_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function SimpleStairs_ResetCube(%SimpleStairs)
   {
      //*** Retrieve a pointer to the static data
      %static = %SimpleStairs.static;
      
      //*** Setup default values
      SimpleStairs_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%SimpleStairs, %SimpleStairs.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = -VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   
   // MakeCuboid
   // Makes a solid at the position specified.
   // Creates a new brush if one was not available.
   // can pass in up to 2 additional planes that are added to the cube.
   function MakeCuboid(%edit,%SimpleStairs, %brushnum, %mins, %maxs, %tx,%ty,%tz, %topplane, %bottomplane)
   {
      // build the eight vertexes
      // Get the brush from the scene.
      // Build the planes.
      // Add each plane, and set each texture.
      
      %brush = GetBrush(%edit,%brushnum);
      
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      
      %veca = NewVector(%minx,%miny,%minz);
      %vecb = NewVector(%maxx,%miny,%minz);
      %vecc = NewVector(%minx,%miny,%maxz);
      %vecd = NewVector(%maxx,%miny,%maxz);
      %vece = NewVector(%maxx,%maxy,%minz);
      %vecf = NewVector(%minx,%maxy,%minz);
      %vecg = NewVector(%maxx,%maxy,%maxz);
      %vech = NewVector(%minx,%maxy,%maxz);
      
      %brush.setTextureOffset(%SimpleStairs.static.uoffset, %SimpleStairs.static.voffset);
      %brush.setTextureScale(%SimpleStairs.static.uscale, %SimpleStairs.static.vscale);     
        
      // 
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%maxx));
      
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ 1.0*(%minx));

      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%maxy));
         
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);   
      %brush.addPlane("0 -1 0 " @ 1.0*(%miny));

      // Top faces
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      
      if (strlen(%topplane) > 0)
      {
         %brush.addPlane(%topplane);
      }
      else
      {
         %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      }
            
      // Bottom faces
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);

      if (strlen(%bottomplane) > 0)
      {
         %brush.addPlane(%bottomplane);
      }
      else
      {
         %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  
      }
        
      %edit.updateBrush(%brush);
   }
       
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = -VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
      
   function GetMins(%point,%size)
   {
      return VectorSub(%point,NewVector(%size,%size,%size));
   }
   
   function GetMaxs(%point,%size)
   {
      return VectorAdd(%point,NewVector(%size,%size,%size));
   }

   function CreateMarker(%edit, %SimpleStairs, %point)
   {
      %mins = GetMins(%point,0.5);
      %maxs = GetMaxs(%point,0.5);
      MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%maxs,.5,.5,.5,"","");
      %SimpleStairs.CurrentBrush++;
   }
   
   function GetBoxedPlane(%edit, %SimpleStairs, %stepheight, %stepwidth, %steplength, %mins, %maxs)
   {
      %center = NewVector(%SimpleStairs.static.center[0],%SimpleStairs.static.center[1],%SimpleStairs.static.center[2]);
      //%mins = VectorSub(%mins,%center);
      //%maxs = VectorSub(%maxs,%center);
      
      //if (%SimpleStairs.static.StairType == 2)
      //{
         // min/max coordinates.
         %minx = getWord(%mins,0);
         %maxx = getWord(%maxs,0);
         %miny = getWord(%mins,1);
         %maxy = getWord(%maxs,1);
         // Top always starts down 
         %topz = getWord(%maxs,2)-%stepheight;
         %bottomz = getWord(%mins,2);
         
         switch(%SimpleStairs.static.StepDirection)
         {
            case 0: // North
               %pt1 = NewVector(%minx,%maxy,%topz);
               %pt2 = NewVector(%maxx,%maxy,%topz);
               %pt3 = NewVector(%minx,%miny+%stepwidth,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 1: // East
               %pt1 = NewVector(%maxx,%maxy,%topz);
               %pt2 = NewVector(%maxx,%miny,%topz);
               %pt3 = NewVector(%minx+%steplength,%miny,%bottomz);
               //CreateMarker(%edit,%SimpleStairs,%pt1);
               //CreateMarker(%edit,%SimpleStairs,%pt2);
               //CreateMarker(%edit,%SimpleStairs,%pt3);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 2: // South
               %pt1 = NewVector(%minx,%miny,%topz);
               %pt2 = NewVector(%maxx,%miny,%topz);
               %pt3 = NewVector(%minx,%maxy-%stepwidth,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 3: // West
               %pt1 = NewVector(%minx,%maxy,%topz);
               %pt2 = NewVector(%minx,%miny,%topz);
               %pt3 = NewVector(%maxx-%steplength,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
         }
      //}
      return "";
   }
   
   function InvertPlane(%plane)
   {
     %nx = 0-(getWord(%plane,0));      
     %ny = 0-(getWord(%plane,1));      
     %nz = 0-(getWord(%plane,2));      
     %d  = 0-(getWord(%plane,3));      
     return (%nx SPC %ny SPC %nz SPC %d);
   }
   
   function GetTopCarriage(%edit, %SimpleStairs, %stepheight, %stepwidth, %steplength, %mins, %maxs)
   {
      //%stepspace = %stepheight - %SimpleStairs.static.StepThickness;
      %minx = getWord(%mins,0);
      %maxx = getWord(%maxs,0);
      %miny = getWord(%mins,1);
      %maxy = getWord(%maxs,1);

      %topz = getWord(%maxs,2);
      %bottomz = getWord(%mins,2)+%stepheight;
      
         switch(%SimpleStairs.static.StepDirection)
         {
            case 0: // North
               %pt1 = NewVector(%minx,%maxy-%stepwidth,%topz);
               %pt2 = NewVector(%maxx,%maxy-%stepwidth,%topz);
               %pt3 = NewVector(%minx,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 1: // East
               %pt1 = NewVector(%maxx-%steplength,%maxy,%topz);
               %pt2 = NewVector(%maxx-%steplength,%miny,%topz);
               %pt3 = NewVector(%minx,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 2: // South
               %pt1 = NewVector(%minx,%miny+%stepwidth,%topz);
               %pt2 = NewVector(%maxx,%miny+%stepwidth,%topz);
               %pt3 = NewVector(%minx,%maxy,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 3: // West
               %pt1 = NewVector(%minx+%steplength,%maxy,%topz);
               %pt2 = NewVector(%minx+%steplength,%miny,%topz);
               %pt3 = NewVector(%maxx,%miny,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
         }
      return "";
   }
   
   function GetRailPlane(%edit, %SimpleStairs, %stepheight, %stepwidth, %steplength, %mins, %maxs,%offset)
   {
      %minx = getWord(%mins,0);
      %maxx = getWord(%maxs,0);
      %miny = getWord(%mins,1);
      %maxy = getWord(%maxs,1);

      
      %topz = getWord(%maxs,2) + %offset;
      %bottomz = getWord(%mins,2)+%stepheight + %offset;
      
         switch(%SimpleStairs.static.StepDirection)
         {
            case 0: // North
               %pt1 = NewVector(%minx,%maxy-%stepwidth,%topz);
               %pt2 = NewVector(%maxx,%maxy-%stepwidth,%topz);
               %pt3 = NewVector(%minx,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 1: // East
               %pt1 = NewVector(%maxx-%steplength,%maxy,%topz);
               %pt2 = NewVector(%maxx-%steplength,%miny,%topz);
               %pt3 = NewVector(%minx,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 2: // South
               %pt1 = NewVector(%minx,%miny+%stepwidth,%topz);
               %pt2 = NewVector(%maxx,%miny+%stepwidth,%topz);
               %pt3 = NewVector(%minx,%maxy,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 3: // West
               %pt1 = NewVector(%minx+%steplength,%maxy,%topz);
               %pt2 = NewVector(%minx+%steplength,%miny,%topz);
               %pt3 = NewVector(%maxx,%miny,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
         }
      return "";
   }

   function GetBottomCarriage(%edit, %SimpleStairs, %stepheight, %stepwidth, %steplength, %mins, %maxs)
   {
      %minx = getWord(%mins,0);
      %maxx = getWord(%maxs,0);
      %miny = getWord(%mins,1);
      %maxy = getWord(%maxs,1);

      %topz = getWord(%maxs,2)-%SimpleStairs.static.StepThickness;
      %bottomz = (getWord(%mins,2)+%stepheight)-%SimpleStairs.static.StepThickness;
      
      switch (%SimpleStairs.static.StairType)
      {
        // case 0: // Open
        //    return GetBoxedPlane(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
         case 1: // Closed
            return GetBoxedPlane(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
         case 2: // Boxed
            return GetBoxedPlane(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
         case 3: // Full
            return "";
      }
      
       
         switch(%SimpleStairs.static.StepDirection)
         {
            case 0: // North
               %pt1 = NewVector(%minx,%maxy,%topz);
               %pt2 = NewVector(%maxx,%maxy,%topz);
               %pt3 = NewVector(%minx,%miny+%stepwidth,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 1: // East
               %pt1 = NewVector(%maxx,%maxy,%topz);
               %pt2 = NewVector(%maxx,%miny,%topz);
               %pt3 = NewVector(%minx+%steplength,%miny,%bottomz);
               return PlaneFromPoints(%pt2,%pt1,%pt3);
            case 2: // South
               %pt1 = NewVector(%minx,%miny,%topz);
               %pt2 = NewVector(%maxx,%miny,%topz);
               %pt3 = NewVector(%minx,%maxy-%stepwidth,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
            case 3: // West
               %pt1 = NewVector(%minx,%maxy,%topz);
               %pt2 = NewVector(%minx,%miny,%topz);
               %pt3 = NewVector(%maxx-%steplength,%miny,%bottomz);
               return PlaneFromPoints(%pt1,%pt2,%pt3);
         }
      //}
      return "";
   }

   function CreateBaluster(%edit,%SimpleStairs,%tx,%ty,%tz,%stepmin,%stepmax, %direction,%balustertop)
   {
         // Z is constant
         %minz = getWord(%stepmax,2);
         %maxz = %minz + %SimpleStairs.static.BalusterHeight;
         
         switch (%direction)
         {
            case 0: // north
               // offset from right edge
               %minx = getWord(%stepmax,0) - (%SimpleStairs.static.BalusterWidth+%SimpleStairs.static.BalusterInset);
               %maxx = %minx +%SimpleStairs.static.BalusterWidth;
               // Y is centered
               %ht = getWord(%stepmax,1) - getWord(%stepmin,1);
               %centery = getWord(%stepmin,1)+(%ht/2);
               %miny = %centery - (%SimpleStairs.static.BalusterWidth / 2);
               %maxy = %miny + %SimpleStairs.static.BalusterWidth;
               
            case 1: // east
               // offset from right edge
               %miny = getWord(%stepmin,1) + %SimpleStairs.static.BalusterInset;
               %maxy = %miny +%SimpleStairs.static.BalusterWidth;
               
               // x is centered
               %wd = getWord(%stepmax,0) - getWord(%stepmin,0);
               %centerx = getWord(%stepmin,0)+(%wd/2);
               %minx = %centerx - (%SimpleStairs.static.BalusterWidth / 2);
               %maxx = %minx + %SimpleStairs.static.BalusterWidth;

            
            case 2: // south
               // offset from right edge
               %minx = getWord(%stepmin,0)+%SimpleStairs.static.BalusterInset;
               %maxx = %minx +%SimpleStairs.static.BalusterWidth;
               // Y is centered
               %ht = getWord(%stepmax,1) - getWord(%stepmin,1);
               %centery = getWord(%stepmin,1)+(%ht/2);
               %miny = %centery - (%SimpleStairs.static.BalusterWidth / 2);
               %maxy = %miny + %SimpleStairs.static.BalusterWidth;
               
            case 3: // west
            
            
               // offset from right edge
               %miny = getWord(%stepmax,1) - (%SimpleStairs.static.BalusterWidth+%SimpleStairs.static.BalusterInset);
               %maxy = %miny +%SimpleStairs.static.BalusterWidth;
               
               // x is centered
               %wd = getWord(%stepmax,0) - getWord(%stepmin,0);
               %centerx = getWord(%stepmin,0)+(%wd/2);
               %minx = %centerx - (%SimpleStairs.static.BalusterWidth / 2);
               %maxx = %minx + %SimpleStairs.static.BalusterWidth;            
            
         }
      %mins = NewVector(%minx, %miny, %minz);
      %maxs = NewVector(%maxx, %maxy, %maxz);
      MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%maxs,%tx,%ty,%tz,%balustertop,"");
      %SimpleStairs.CurrentBrush++;
   }
   
   function BuildStep(%edit,%SimpleStairs,%tx,%ty,%tz,%stepmin,%stepmax, %bottomplane, %balustertop)
   {
      MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%stepmin,%stepmax,%tx,%ty,%tz,"",%bottomplane);
      %SimpleStairs.CurrentBrush++;
      
      if (%SimpleStairs.static.RightBalustrade)
      {
         %direction = %SimpleStairs.static.StepDirection;
         CreateBaluster(%edit,%SimpleStairs,%tx,%ty,%tz,%stepmin,%stepmax,%direction, %balustertop);
      }
      
      if (%SimpleStairs.static.LeftBalustrade)
      {
         // invert direction for left 
         %direction = %SimpleStairs.static.StepDirection + 2;
         if (%direction >= 4) %direction -= 4;
         CreateBaluster(%edit,%SimpleStairs,%tx,%ty,%tz,%stepmin,%stepmax,%direction,%balustertop);
      }
   }

   function BuildRails(%edit,%SimpleStairs,%tx,%ty,%tx,%mins,%maxs,%stepheight,%stepwidth,%steplength)
   {
      // Build the rails. Very similar to the carriage...
      
      %topplane    = GetRailPlane(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs,%SimpleStairs.static.RailHeight + %SimpleStairs.static.BalusterHeight );
      %balustertop = GetRailPlane(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs,%SimpleStairs.static.BalusterHeight );
      %bottomplane = InvertPlane(%balustertop);

      
      if (%SimpleStairs.static.StepDirection == 1 || %SimpleStairs.static.StepDirection == 3)
      {
        // East/West
        %first = false;
        %second = false;
        if (%SimpleStairs.static.StepDirection == 3)
        {
           if (%SimpleStairs.static.LeftBalustrade) %first  = true;
           if (%SimpleStairs.static.RightBalustrade) %second = true;
        }
        if (%SimpleStairs.static.StepDirection == 1)
        {
           if (%SimpleStairs.static.LeftBalustrade) %second  = true;
           if (%SimpleStairs.static.RightBalustrade) %first = true;
        }
        
        %miny = getWord(%mins,1);
        %maxy = getWord(%maxs,1);
        %min1 = setWord(%maxs,1,%miny+%SimpleStairs.static.RailWidth);
        %max1 = setWord(%mins,1,%maxy-%SimpleStairs.static.RailWidth);
        
        if (%first)
        {
          MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%min1,%tx,%ty,%tz,%topplane,%bottomplane);
          %SimpleStairs.CurrentBrush++;
        }
        if (%second)
        {
          MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%max1,%maxs,%tx,%ty,%tz,%topplane,%bottomplane);
          %SimpleStairs.CurrentBrush++;
        }
      }
      else
      {
        // North/South
        %first = false;
        %second = false;
        %minx = getWord(%mins,0);
        %maxx = getWord(%maxs,0);
        
        %min1 = setWord(%maxs,0,%minx+%SimpleStairs.static.RailWidth);
        %max1 = setWord(%mins,0,%maxx-%SimpleStairs.static.RailWidth);
        if (%SimpleStairs.static.StepDirection == 0)
        {
           if (%SimpleStairs.static.LeftBalustrade) %first  = true;
           if (%SimpleStairs.static.RightBalustrade) %second = true;
        }
        if (%SimpleStairs.static.StepDirection == 2)
        {
           if (%SimpleStairs.static.LeftBalustrade) %second  = true;
           if (%SimpleStairs.static.RightBalustrade) %first = true;
        }        
        
        if (%first)
        {
          MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%min1,%tx,%ty,%tz,%topplane,%bottomplane);
          %SimpleStairs.CurrentBrush++;
        }
        if (%second)
        {
          MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%max1,%maxs,%tx,%ty,%tz,%topplane,%bottomplane);
          %SimpleStairs.CurrentBrush++;        
        }
      }     
      return %balustertop; 
   }

   
   function BuildCarriage(%edit,%SimpleStairs,%tx,%ty,%tz,%mins,%maxs,%stepheight,%stepwidth,%steplength)
   { 
      // MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%stepmin,%stepmax,%tx,%ty,%tz,%topplane,%bottomplane);
      // %SimpleStairs.CurrentBrush++;
      %topplane    = GetTopCarriage(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
      %bottomplane = GetBottomCarriage(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
      if (%SimpleStairs.static.StepDirection == 1 || %SimpleStairs.static.StepDirection == 3)
      {
        %miny = getWord(%mins,1);
        %maxy = getWord(%maxs,1);
        // East/West
        %min1 = setWord(%maxs,1,%miny+%SimpleStairs.static.CarriageWidth);
        %max1 = setWord(%mins,1,%maxy-%SimpleStairs.static.CarriageWidth);
        MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%min1,%tx,%ty,%tz,%topplane,%bottomplane);
        %SimpleStairs.CurrentBrush++;
        MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%max1,%maxs,%tx,%ty,%tz,%topplane,%bottomplane);
        %SimpleStairs.CurrentBrush++;
      }
      else
      {
        // North/South
        %minx = getWord(%mins,0);
        %maxx = getWord(%maxs,0);
        
        %min1 = setWord(%maxs,0,%minx+%SimpleStairs.static.CarriageWidth);
        %max1 = setWord(%mins,0,%maxx-%SimpleStairs.static.CarriageWidth);
        MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%mins,%min1,%tx,%ty,%tz,%topplane,%bottomplane);
        %SimpleStairs.CurrentBrush++;
        MakeCuboid(%edit,%SimpleStairs,%SimpleStairs.CurrentBrush,%max1,%maxs,%tx,%ty,%tz,%topplane,%bottomplane);
        %SimpleStairs.CurrentBrush++;        
      }
   }
   
   function BuildStairs(%edit,%SimpleStairs,%tx,%ty,%tz,%mins, %maxs)
   {
      %cmins = %mins;
      %cmaxs = %maxs;      
      // if carriage, then we must make it first, and then readjust the bounding box.
      if (%SimpleStairs.static.Carriage)
      {
         // build the carriage on both sides.
         // Adjust the mins, maxs.

         if (%SimpleStairs.static.StepDirection == 0 || %SimpleStairs.static.StepDirection == 2)
         {
            // North/South
            %minx = getWord(%mins,0);
            %maxx = getWord(%maxs,0);
            %mins = setWord(%mins,0,%minx+%SimpleStairs.static.CarriageWidth);
            %maxs = setWord(%maxs,0,%maxx-%SimpleStairs.static.CarriageWidth);
         }
         else
         {
            // East/West
            %miny = getWord(%mins,1);
            %maxy = getWord(%maxs,1);
            %mins = setWord(%mins,1,%miny+%SimpleStairs.static.CarriageWidth);
            %maxs = setWord(%maxs,1,%maxy-%SimpleStairs.static.CarriageWidth);
         }
      }

      %stepmin = %mins;
      %stepbounds = VectorSub(%maxs,%mins);

      
      %stepheight = getWord(%stepbounds,2) / %SimpleStairs.static.StepCount;
      if (%SimpleStairs.static.StepDirection == 1 || %SimpleStairs.static.StepDirection == 3)
      {
        // East/West
        %stepwidth  = getWord(%stepbounds,1);
        %steplength = getWord(%stepbounds,0) / %SimpleStairs.static.StepCount;
        %stepdelta  = NewVector(%steplength,0,%stepheight);
      }
      else
      {
        // North/South
        %stepwidth  = getWord(%stepbounds,1) / %SimpleStairs.static.StepCount;
        %steplength = getWord(%stepbounds,0);
        %stepdelta  = NewVector(0,%stepwidth,%stepheight);
      }
      
      %stepsize   = NewVector(%steplength,%stepwidth,%stepheight);
      %minh = getWord(%mins,2);

      if (%SimpleStairs.static.StairType == 2)
      {
        %bottomplane = GetBoxedPlane(%edit, %SimpleStairs,%stepheight,%stepwidth, %steplength, %mins,%maxs);
        %invertedbottom = InvertPlane(%bottomplane);
      }
      else
      {
          if (%SimpleStairs.static.ConformToCarriage)
          {
             %bottomplane = GetBottomCarriage(%edit,%SimpleStairs,%stepheight,%stepwidth,%steplength,%mins,%maxs);
             %invertedbottom = InvertPlane(%bottomplane);
          }
          else
          {
            %bottomplane = "";
            %invertedbottom = "";
          }
      }
      
      
      if (%SimpleStairs.static.Carriage)
      {
         // Now that all the needed vars are created, we can actually
         // make the carriage.
         BuildCarriage(%edit,%SimpleStairs,%tx,%ty,%tx,%cmins,%cmaxs,%stepheight,%stepwidth,%steplength);
      }
      
      %balustertop = "";
      if (%SimpleStairs.static.LeftBalustrade || %SimpleStairs.static.RightBalustrade)
      {
         %balustertop = BuildRails(%edit,%SimpleStairs,%tx,%ty,%tx,%mins,%maxs,%stepheight,%stepwidth,%steplength);
      }

      switch(%SimpleStairs.static.StairType)
      {
         case 0:
           // Open Stairs.
           %newmin = %minh + (%stepheight - %SimpleStairs.static.StepThickness);
           %stepmin = setWord(%stepmin,2,%newmin);
           %stepsize   = NewVector(%steplength,%stepwidth,%SimpleStairs.static.StepThickness);
         case 1:
           // Closed Stairs.
           %stepsize   = NewVector(%steplength,%stepwidth,%stepheight);
         case 2:
           // Boxed Stairs.
           %stepsize   = NewVector(%steplength,%stepwidth,%stepheight);
      }

      // Invert if south or west.      
      if (%SimpleStairs.static.StepDirection >= 2)
      {
         %stepmin = setWord(%stepmin,2,getWord(%maxs,2)-getWord(%stepsize,2));
         %stepdelta = setWord(%stepdelta,2,0-%stepheight);
      }
      
    
      // Create the supports.
      if (%SimpleStairs.static.Supports && (%SimpleStairs.static.StairType != 3))
      {
          
         
          %swidth = %SimpleStairs.static.SupportWidth;
          %ywidth = %swidth;
          %xwidth = %swidth;
          %minx = getWord(%mins,0);
          %maxx = getWord(%maxs,0);
          %miny = getWord(%mins,1);
          %maxy = getWord(%maxs,1);         
          %minz = getWord(%mins,2);
          
          if (%SimpleStairs.static.StairType == 1)
          {
            %maxz = getWord(%maxs,2)-%stepheight;
          }
          else
          {
            %maxz = getWord(%maxs,2)-%SimpleStairs.static.StepThickness;
          }
          
         if (%SimpleStairs.static.SupportsOnCarriage)
         {
            %minx = getWord(%cmins,0);
            %maxx = getWord(%cmaxs,0);
            %miny = getWord(%cmins,1);
            %maxy = getWord(%cmaxs,1);         
            %minz = getWord(%cmins,2);
            if (%SimpleStairs.static.ConformToCarriage)
            {
               if(%SimpleStairs.static.StepDirection == 0 || %SimpleStairs.static.StepDirection == 2)
               {
               %ywidth = %swidth;
               %xwidth = %SimpleStairs.static.CarriageWidth;
               }
               else
               {
               %ywidth = %SimpleStairs.static.CarriageWidth;
               %xwidth = %swidth;
               }
            }
         }
  
         // calc bounding box for supports.
         switch(%SimpleStairs.static.StepDirection)
         {
            case 0: // north
               %smin1 = NewVector(%minx,%maxy-%ywidth,%minz);
               %smax1 = NewVector(%minx+%xwidth,%maxy,%maxz);
               %smin2 = NewVector(%maxx-%xwidth,%maxy-%ywidth,%minz);
               %smax2 = NewVector(%maxx,%maxy,%maxz);
            case 1: // east
               %smin1 = NewVector(%maxx-%xwidth,%maxy-%ywidth,%minz);
               %smax1 = NewVector(%maxx,%maxy,%maxz);
               %smin2 = NewVector(%maxx-%xwidth,%miny,%minz);
               %smax2 = NewVector(%maxx,%miny+%ywidth,%maxz);
               
            case 2: // south
               %smin1 = NewVector(%maxx-%xwidth,%miny,%minz);
               %smax1 = NewVector(%maxx,%miny+%ywidth,%maxz);
               // 
               %smin2 = NewVector(%minx,%miny,%minz);
               %smax2 = NewVector(%minx+%xwidth,%miny+%ywidth,%maxz);

            case 3: // west
               %smin1 = NewVector(%minx,%maxy-%ywidth,%minz);
               %smax1 = NewVector(%minx+%xwidth,%maxy,%maxz);
               
               %smin2 = NewVector(%minx,%miny,%minz);
               %smax2 = NewVector(%minx+%xwidth,%miny+%ywidth,%maxz);
               //%smin2 = NewVector(%maxx-%swidth,%maxy-%swidth,%minz);
               //%smax2 = NewVector(%maxx,%maxy,%maxz);
         }

         MakeCuboid(%edit,%SimpleStairs, %SimpleStairs.CurrentBrush, %smin1, %smax1, %tx,%ty,%tz, %invertedbottom, "");
         %SimpleStairs.CurrentBrush++;
         
         MakeCuboid(%edit,%SimpleStairs, %SimpleStairs.CurrentBrush, %smin2, %smax2, %tx,%ty,%tz, %invertedbottom, "");
         %SimpleStairs.CurrentBrush++;
      }
      
      // Build each step
      for (%i = 0; %i < %SimpleStairs.static.StepCount; %i++)
      {
         %stepmax = VectorAdd(%stepmin,%stepsize);
         %steppos = %stepmin;
         if (%SimpleStairs.static.StairType == 3)
         {
            %steppos = setWord(%stepmin,2,%minh);
         }
         if (%SimpleStairs.static.StairType == 2)
         {
            %steppos = setWord(%stepmin,2,%minh);
            //%newh = getWord(%steppos,2);
            //%newh = %newh - %stepheight;
            //if (%newh <= %minh) %newh = %minh;
            //%steppos = setWord(%steppos,2,%newh);
         }
         BuildStep(%edit,%SimpleStairs,%tx,%ty,%tz,%steppos,%stepmax,%bottomplane,%balustertop);
         %stepmin = VectorAdd(%stepmin,%stepdelta);
      }
   }
   
   //*** Build/modify the actual geometry
   function SimpleStairs_MakeGeometry(%SimpleStairs, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%SimpleStairs.active)
         return;

      if (%SimpleStairs.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %SimpleStairs.static.forceclear = 0;
      }
      
      %center = %SimpleStairs.static.center[0] SPC %SimpleStairs.static.center[1] SPC %SimpleStairs.static.center[2];
      %tx = %SimpleStairs.static.size[0] * 0.5;
      %ty = %SimpleStairs.static.size[1] * 0.5;
      %tz = %SimpleStairs.static.size[2] * 0.5;
      %mins = NewVector(%SimpleStairs.static.center[0]-%tx,%SimpleStairs.static.center[1]-%ty,%SimpleStairs.static.center[2]-%tz);
      %maxs = NewVector(%SimpleStairs.static.center[0]+(%tx),%SimpleStairs.static.center[1]+%ty,%SimpleStairs.static.center[2]+%tz);
      %SimpleStairs.CurrentBrush = 0;
      BuildStairs(%edit,%SimpleStairs,%tx,%ty,%tz,%mins,%maxs);
   }
};

tool.register("SimpleStairs", tool.typeInteractive(), tool.RFLAG_NONE(), "Simple Stairs" );
tool.setToolProperty("SimpleStairs", "Icon", "icons/stair");
tool.setToolProperty("SimpleStairs", "Group", "Create");
