//
// Obelisk.cs
//
// Defines the Obelisk tool plug-in to create a Obelisk:
//
//     
//     CC
//     CC
//     CC
//    XXXX
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Sides (3+)
// Base Sides  (Default 4. Min 3).
// Base lip    (default 0.1);
// Base Height (default 0.2).
// Array
// X count (Default 1)
// Y count (Default 1)
// X Spacing (Default 4)
// Y Spacing (Default 4).


package Obelisk
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Obelisk: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Obelisk_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %Obelisk = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %Obelisk.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%Obelisk, %static);

      //*** Setup some additional attributes for the cube instance
      %Obelisk.dirty = tool.DIRTY_NONE();
      %Obelisk.active = false;
      %Obelisk.update = tool.EDIT_DONOTHING();
      %Obelisk.changeCenter = false;
      %Obelisk.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %Obelisk;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Obelisk: Done(" @ %inst @ "," @ %static @ ")");

      %Obelisk = %inst.instance;
      
      if(%Obelisk)
      {         
         //*** Delete our instance
         %Obelisk.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Obelisk: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Obelisk: HandleCount(" @ %inst @ "," @ %event @ ")");

      %Obelisk = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %Obelisk.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Obelisk: HandleInit(" @ %inst @ "," @ %event @ ")");

      %Obelisk = %inst.instance;

      //*** Make the tool active
      if(!%Obelisk.active)
      {
         %Obelisk.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%Obelisk, %Obelisk.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Obelisk: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %Obelisk = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%Obelisk, %Obelisk.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Obelisk: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %Obelisk = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%Obelisk, %Obelisk.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %Obelisk.dirty = tool.DIRTY_APPEARANCE();
      %Obelisk.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Obelisk: Dirty(" @ %inst @ ")");

      %Obelisk = %inst.instance;
      
      return %Obelisk.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Obelisk: Draw(" @ %inst @ "," @ %draw @ ")");

      %Obelisk = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%Obelisk.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%Obelisk, %Obelisk.static, %draw);

      //*** Indicate that we've drawn the tool
      %Obelisk.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Obelisk: CheckEditAction(" @ %inst @ ")");

      %Obelisk = %inst.instance;
      
      return %Obelisk.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Obelisk: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %Obelisk = %inst.instance;
      
      %Obelisk.update = tool.EDIT_DONOTHING();
      %Obelisk.active = false;
      %Obelisk.changeSize = false;
      %Obelisk.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Obelisk: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %Obelisk = %inst.instance;
      
      // Work on the actual geometry.
      Obelisk_MakeGeometry(%Obelisk, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %Obelisk.update = tool.EDIT_DONOTHING();
      %Obelisk.changeSize = false;
      %Obelisk.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Obelisk: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %Obelisk = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %Obelisk.update = tool.EDIT_UPDATE();
            %Obelisk.active = true;
            %Obelisk.dirty = tool.DIRTY_APPEARANCE();
            %Obelisk.changeSize = true;
            %Obelisk.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Obelisk_DefaultValues(%Obelisk.static);
            ToolBB::rebuildBoundingBox(%Obelisk, %Obelisk.static);
            %Obelisk.update = tool.EDIT_UPDATE();
            %Obelisk.active = true;
            %Obelisk.dirty = tool.DIRTY_APPEARANCE();
            %Obelisk.changeSize = true;
            %Obelisk.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%Obelisk.active)
            {
               %Obelisk.update = tool.EDIT_REJECT();
            }
            %Obelisk.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%Obelisk.active)
            {
               %Obelisk.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Obelisk: Interface(" @ %inst @ "," @ %form @ ")");

      %Obelisk = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Create Obelisks");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");

      // TODO: Add some kid of slope, so we can have curves, etc.      
      %form.addField(-1,  "Obelisk" ,"divider");
      %form.addField(14,  "Autocalc Cap",   "checkbox");
      %form.addField( 2,  "Cap Percentage", "numericinteger");
      %form.addField( 9,  "Taper Percent", "numericinteger");
      %form.addField( 3,  "Sides",       "numericinteger");
      %form.addField( 4,  "Rectangular", "checkbox");
      
      %form.addField(-1,  "Base" ,"divider");
      %form.addField( 10,  "Sides",        "numericinteger");
      %form.addField( 11,  "Side Spacing", "numeric");
      %form.addField( 12,  "Height",       "numeric");
      %form.addField( 13,  "Rectangular",  "checkbox");
      
      %form.addField(-1,  "Array" ,"divider");
      %form.addField(20,   "X Count", "numericinteger");
      %form.addField(21,   "Y Count", "numericinteger");
      %form.addField(22,   "X Spacing", "numeric");
      %form.addField(23,   "Y Spacing", "numeric");
/*      
      %obj.segments  = 1;     // number of segments in main Obelisk.
      %obj.sides     = 6;     // sides of the Obelisk
      %obj.basesides = 4;     // sides of the bases (top and bottom)
      %obj.baselip   = 0.1;   // spacing of base from Obelisk side
      %obj.baseheight = 0.2;  // height of the base (if 0, then no base)
      %obj.xcount     = 1;    // count along the X axis.
      %obj.ycount     = 1;    // count along the Y axis.
      %obj.xspace     = 4;    // Spacing
      %obj.yspace     = 4;    // Spacing
*/      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      //********************************************
      %form.setFieldMinLimit(2, 1);
      %form.setFieldMaxLimit(2, 99);
      %form.setFieldMinLimit(9, 1);
      %form.setFieldMaxLimit(9, 99);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 3);
      //********************************************
      %form.setFieldMinLimit(20, 0);
      %form.setFieldMinLimit(21, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Obelisk: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %Obelisk = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %Obelisk.static.center[0] SPC %Obelisk.static.center[1] SPC %Obelisk.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %Obelisk.static.size[0] SPC %Obelisk.static.size[1] SPC %Obelisk.static.size[2];
            return %value;
      
         //*** Handle the 'Segments' field
         case 2:
            %value = %Obelisk.static.capperc;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %Obelisk.static.sides;
            return %value;
            
         case 4:
            %value = %Obelisk.static.rectangular;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 5:
            %value = %Obelisk.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %Obelisk.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %Obelisk.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %Obelisk.static.voffset;
            return %value;            
            
            // Taper
         case 9:
            %value = %Obelisk.static.taper;
            return %value;
            
         case 10:
            %value = %Obelisk.static.basesides;
            return %value;            
            
         case 11:
            %value = %Obelisk.static.baselip;
            return %value;            
            
         case 12:
            %value = %Obelisk.static.baseheight;
            return %value;            
         case 13:
            %value = %Obelisk.static.baserectangular;
            return %value;
                        
         case 14:
            %value = %Obelisk.static.autocalccap;
            return %value;
            
         case 20:
            %value = %Obelisk.static.xcount;
            return %value;            
         case 21:
            %value = %Obelisk.static.ycount;
            return %value;            
         case 22:
            %value = %Obelisk.static.xspace;
            return %value;            
         case 23:
            %value = %Obelisk.static.yspace;
            return %value;            
         
      }
      
      return 0;
   }



   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Obelisk: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %Obelisk = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %Obelisk.static.center[%i] = getWord(%value, %i);
            }
            %Obelisk.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %Obelisk.static.size[%i] = getWord(%value, %i);
            }
            %Obelisk.changeSize = true;
      
         //*** Handle the Cap Percentage field
         case 2:
            %Obelisk.static.capperc = %value;
            %Obelisk.changeSize = true;
            
         case 9:
            %Obelisk.static.taper = %value;
            %Obelisk.changeSize = true;
      
         //*** Handle the Sides field
         case 3:
            %Obelisk.static.sides = %value;
            %Obelisk.changeSize = true;
         case 4:
            %Obelisk.static.rectangular = %value;
            %Obelisk.changeSize = true;

         //*** Handle the 'U Scale' field
         case 5:
            %Obelisk.static.uscale = %value;
            %Obelisk.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %Obelisk.static.vscale = %value;
            %Obelisk.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %Obelisk.static.uoffset = %value;
            %Obelisk.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %Obelisk.static.voffset = %value;
            %Obelisk.changeSize = true;   
               
         case 10:
            %Obelisk.static.basesides =%value;
            %Obelisk.changeSize = true;   
            
         case 11:
            %Obelisk.static.baselip = %value;
            %Obelisk.changeSize = true;   
            
         case 12:
            %Obelisk.static.baseheight=%value;
            %Obelisk.changeSize = true;   
            %Obelisk.static.forceclear = true;

         case 13:
            %Obelisk.static.baserectangular=%value;
            %Obelisk.changeSize = true;   
            
         case 14:
            %Obelisk.static.autocalccap = %value;
            %Obelisk.changeSize = true;
               
         case 20:
            %Obelisk.static.xcount=%value;
            %Obelisk.static.forceclear = true;
            %Obelisk.changeSize = true;   
            
         case 21:
            %Obelisk.static.ycount=%value;
            %Obelisk.static.forceclear = true;
            %Obelisk.changeSize = true;   
            
         case 22:
            %Obelisk.static.xspace=%value;
            %Obelisk.changeSize = true;   
            
         case 23:
            %Obelisk.static.yspace=%value;
            %Obelisk.changeSize = true;   
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%Obelisk.active)
      {
         %Obelisk.active = true;
         %Obelisk.changeSize = true;
         %Obelisk.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %Obelisk.update = tool.EDIT_UPDATE();
      %Obelisk.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%Obelisk, %Obelisk.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function Obelisk_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;  
      
      %obj.autocalccap = true;
      %obj.capperc   = 10;    // Percentage of height that is the cap.
      %obj.taper     = 30;    // taper of sides.
      %obj.sides     = 4;     // sides of the Obelisk
      %obj.rectangular = false;
      %obj.basesides = 8;     // sides of the bases (top and bottom)
      %obj.baselip   = 1.0;   // spacing of base from Obelisk side
      %obj.baseheight = 0.5;  // height of the base (if 0, then no base)
      %obj.baserectangular = false;
      %obj.xcount     = 1;    // count along the X axis.
      %obj.ycount     = 1;    // count along the Y axis.
      %obj.xspace     = 4;    // Spacing
      %obj.yspace     = 4;    // Spacing
      %obj.forceclear = false;
   }

   //*** Init the static object
   function Obelisk_InitStatic(%static)
   {
      //*** Setup default values
      Obelisk_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Obelisk_ResetCube(%Obelisk)
   {
      //*** Retrieve a pointer to the static data
      %static = %Obelisk.static;
      
      //*** Setup default values
      Obelisk_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%Obelisk, %Obelisk.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   function GetFaceCenter(%Obelisk,%slicenumber)
   {
      %radians = 6.283185307;
      %tx = %Obelisk.static.size[0] * 0.5;
      %ty = %Obelisk.static.size[1] * 0.5;
      %tz = %Obelisk.static.size[2] * 0.5;      
      %tz = 50;
      %ti = %slicenumber * %radians;
      %center = NewVector(mCos(%ti) * %tx, mSin(%ti) * %ty, %tz);
      return %center;
   }
   
   function GetFacePoint(%Obelisk,%edgenumber,%rx, %ry, %zpos, %sides)
   {
      %radians = 6.283185307; // close enough.
      %angle = (%edgenumber / %sides) * %radians;
      %point = NewVector(mCos(%angle) * %rx, mSin(%angle) * %ry, %zpos);
      return %point;
   }      
      
   function MakeSolid(%edit,%Obelisk, %mins, %maxs, %tx,%ty,%tz, %sides, %dotaper)
   {
      %brush = GetBrush(%edit,%Obelisk.currentbrush);
      %Obelisk.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      %rx = (%maxx - %minx) * 0.5;
      %ry = (%maxy - %miny) * 0.5;
      
      %center = NewVector(%minx + %rx,%miny + %ry,%Obelisk.static.center[2]); 
      %toppt  = NewVector(%minx + %rx,%miny + %ry,%maxz); 
      %ht     = %maxz - %minz;
      if (%Obelisk.static.autocalccap)
        %capht = ((%maxx - %minx) * (100-%Obelisk.static.taper)) / 100.0; 
      else
        %capht = %ht * (%Obelisk.static.capperc / 100);
        
      %conestart = %maxz - %capht;
      
      for (%face = 0; %face < %sides; %face++)
      {
         if(%face == %sides-1)
         {
            %nextface = 0;
         }
         else
         {
            %nextface = %face+1;
         }
         // calc the points for the face.
         %pt1 = VectorAdd(%center,GetFacePoint(%Obelisk,%face,%rx,%ry,%minz,%sides));
         %pt2 = VectorAdd(%center,GetFacePoint(%Obelisk,%nextface,%rx,%ry,%minz,%sides));
         // Top point. This needs to be tapered.
         if (%dotaper)
         {
           %taperx = (%rx * (100-%Obelisk.static.taper)) / 100.0;
           %tapery = (%ry * (100-%Obelisk.static.taper)) / 100.0;
         }
         else
         {
           %taperx = %rx;
           %tapery = %ry;
         }
         %pt3 = VectorAdd(%center,GetFacePoint(%Obelisk,%face,%taperx,%tapery,%maxz,%sides));
         
         %brush.setTexturePlanesByPoints(%pt2,%pt1,%pt3, 0.0, 0.0);
         %brush.addPlaneByPoints(%pt2,%pt1,%pt3);

         %center2 = setWord(%center,2,0);
         %pt1 = VectorAdd(%center2,GetFacePoint(%Obelisk,%face,%taperx,%tapery,%conestart,%sides));
         %pt2 = VectorAdd(%center2,GetFacePoint(%Obelisk,%nextface,%taperx,%tapery,%conestart,%sides));
         %pt3 = %toppt;
         
         if (%dotaper)
         {
           %brush.setTexturePlanesByPoints(%pt2,%pt1,%pt3, 0.0, 0.0);
           %brush.addPlaneByPoints(%pt2,%pt1,%pt3);
         }
      }
      
   
      
      // Top face
      if (!%dotaper)
      {
        %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
        %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      }
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
      
   }
   // MakeCuboid
   function MakeCuboid(%edit,%Obelisk, %mins, %maxs, %tx,%ty,%tz)
   {
      %brush = GetBrush(%edit,%Obelisk.currentbrush);
      %Obelisk.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      
      %brush.setTextureOffset(%Obelisk.static.uoffset, %Obelisk.static.voffset);
      %brush.setTextureScale(%Obelisk.static.uscale, %Obelisk.static.vscale);     
        
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%maxx));
      
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ 1.0*(%minx));

      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%maxy));
         
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);   
      %brush.addPlane("0 -1 0 " @ 1.0*(%miny));

      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
   }
      
      
   function CreateObelisk(%edit,%Obelisk,%mins,%maxs,%tx,%ty,%tz)
   {
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);

      if (%Obelisk.static.baseheight > 0)
      {    
        // Make the base. Reset the mins.
        %minz = %minz+%Obelisk.static.baseheight;
        %bmaxs = setWord(%maxs,2,%minz);
        if (%Obelisk.static.baserectangular)
        {
           MakeCuboid(%edit,%Obelisk,%mins,%bmaxs,%tx,%ty,%tz);
        }
        else
        {
           MakeSolid(%edit,%Obelisk,%mins,%bmaxs,%tx,%ty,%tz,%Obelisk.static.basesides,false);
        }
        %mins = setWord(%mins,2,%minz);
      }
      
      // get the segment height.
      // create the Obelisks.
      %minx = %minx + %Obelisk.static.baselip;
      %maxx = %maxx - %Obelisk.static.baselip;
      %miny = %miny + %Obelisk.static.baselip;
      %maxy = %maxy - %Obelisk.static.baselip;
      %sz = %maxz - %minz;
      %ht =  %sz;
      
      %cmins = NewVector(%minx,%miny,%minz);
      %cmaxs = NewVector(%maxx,%maxy,%minz+%ht);      
      
      if ((%minx < %maxx) && (%miny < %maxy) && (%minz < %maxz))
      {
         MakeSolid(%edit,%Obelisk,%cmins,%cmaxs,%tx,%ty,%tz,%Obelisk.static.sides,true);
      }
      // MakeCuboid(%edit,%Obelisk,%mins,%maxs,%tx,%ty,%tz);
      // build base
      // build capital
      // build Obelisk
   }
      
   //*** Build/modify the actual geometry
   function Obelisk_MakeGeometry(%Obelisk, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%Obelisk.active)
         return;

      %Obelisk.currentbrush = 0;
      if (%Obelisk.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %Obelisk.static.forceclear = 0;
      }
   
      //*** Work on the sizing of the brush, if appropriate
      %center = %Obelisk.static.center[0] SPC %Obelisk.static.center[1] SPC %Obelisk.static.center[2];
      %tx = %Obelisk.static.size[0] * 0.5;
      %ty = %Obelisk.static.size[1] * 0.5;
      %tz = %Obelisk.static.size[2] * 0.5;

      %range = 100-%Obelisk.static.percent;
      %currentsize = 100;
      
      %sz = %Obelisk.static.size[2];
      %ht =  %sz;
      %cmins = %mins;
      %cmaxs = setWord(%maxs,2,getWord(%mins,2)+%ht);
      %lastx = %tx;
      %lasty = %ty;
      
      
      for (%xrow = 0; %xrow < %Obelisk.static.xcount; %xrow++)
      {
         for (%yrow = 0; %yrow < %Obelisk.static.ycount; %yrow++)
         {
            %xpos = (%Obelisk.static.center[0]) + (%xrow * %Obelisk.static.xspace);
            %ypos = (%Obelisk.static.center[1]) + (%yrow * %Obelisk.static.yspace);
            %mins = NewVector(%xpos-%tx,%ypos-%ty,%Obelisk.static.center[2]-%tz);
            %maxs = NewVector(%xpos+%tx,%ypos+%ty,%Obelisk.static.center[2]+%tz);
            CreateObelisk(%edit,%Obelisk,%mins,%maxs,%tx,%ty,%tz);
         }
      }
   }
};

tool.register("Obelisk", tool.typeInteractive(), tool.RFLAG_NONE(), "Create Obelisks" );
tool.setToolProperty("Obelisk", "Icon", "icons/Obelisk");
tool.setToolProperty("Obelisk", "Group", "Create");
