//
// MakeHouse.cs
//
// Defines the MakeHouse tool plug-in to create a house shaped primitive
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
// uscale	- Texture scale along the 'u' axis
// vscale	- Texture scale along the 'v' axis
// uoffset	- Texture offset along the 'u' axis
// voffset	- Texture offset along the 'v' axis
// RoofPercentage - what percentage of the cube is the base of the roof.
// 
//
// The house instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
// 
//
// Revision History:
// 5/3/2007       Jaimi McEntire     Created from SimpleBox.cs
//

package MakeHouse
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("MakeHouse: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         MakeHouse_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %house = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %house.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%house, %static);

      //*** Setup some additional attributes for the house instance
      %house.dirty = tool.DIRTY_NONE();
      %house.active = false;
      %house.update = tool.EDIT_DONOTHING();
      %house.changeCenter = false;
      %house.changeSize = false;
      %house.recreatebrushes = true;
      
      //*** Pass along the instance
      %inst.instance = %house;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("MakeHouse: Done(" @ %inst @ "," @ %static @ ")");

      %house = %inst.instance;
      
      if(%house)
      {         
         //*** Delete our instance
         %house.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("MakeHouse: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("MakeHouse: HandleCount(" @ %inst @ "," @ %event @ ")");

      %house = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %house.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("MakeHouse: HandleInit(" @ %inst @ "," @ %event @ ")");

      %house = %inst.instance;

      //*** Make the tool active
      if(!%house.active)
      {
         %house.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%house, %house.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("MakeHouse: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %house = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%house, %house.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("MakeHouse: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %house = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%house, %house.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %house.dirty = tool.DIRTY_APPEARANCE();
      %house.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("MakeHouse: Dirty(" @ %inst @ ")");

      %house = %inst.instance;
      
      return %house.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("MakeHouse: Draw(" @ %inst @ "," @ %draw @ ")");

      %house = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%house.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%house, %house.static, %draw);

      //*** Indicate that we've drawn the tool
      %house.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("MakeHouse: CheckEditAction(" @ %inst @ ")");

      %house = %inst.instance;
      
      return %house.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("MakeHouse: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %house = %inst.instance;
      
      %house.update = tool.EDIT_DONOTHING();
      %house.active = false;
      %house.changeSize = false;
      %house.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("MakeHouse: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %house = %inst.instance;
      
      // Work on the actual geometry.
      MakeHouse_MakeCubeGeometry(%house, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %house.update = tool.EDIT_DONOTHING();
      %house.changeSize = false;
      %house.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("MakeHouse: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %house = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %house.update = tool.EDIT_UPDATE();
            %house.active = true;
            %house.dirty = tool.DIRTY_APPEARANCE();
            %house.changeSize = true;
            %house.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            MakeHouse_DefaultValues(%house.static);
            ToolBB::rebuildBoundingBox(%house, %house.static);
            %house.update = tool.EDIT_UPDATE();
            %house.active = true;
            %house.dirty = tool.DIRTY_APPEARANCE();
            %house.changeSize = true;
            %house.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%house.active)
            {
               %house.update = tool.EDIT_REJECT();
            }
            %house.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%house.active)
            {
               %house.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("MakeHouse: Interface(" @ %inst @ "," @ %form @ ")");

      %house = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Simple Box");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0,  "Center"   ,"distance3");
      %form.addField( 1,  "Size"     ,"distance3");
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 2,  "U Scale"  ,"numeric");
      %form.addField( 3,  "V Scale"  ,"numeric");
      %form.addField( 4,  "U Offset" ,"numeric");
      %form.addField( 5,  "V Offset" ,"numeric");
      %form.addField( -1, "Roof",     "divider");
      %form.addField( 15, "Slope",    "radio");
      %form.addField( 8,  "Height",          "numeric");
      %form.addField( -1, "Structure","divider");
      %form.addField( 9,  "Structure",    "checkbox");
      %form.addField(14,  "Ceiling",      "checkbox");
      %form.addField(10,  "Floor Thickness", "numeric");
      %form.addField(11,  "Wall Thickness", "numeric");
      %form.addField(12,  "Roof Thickness", "numeric");
      %form.addField(13,  "Roof Overhang",  "numeric");
      
      
      %form.addFieldListItem(15,"4 in 10");
      %form.addFieldListItem(15,"6 in 10");
      %form.addFieldListItem(15,"8 in 10");
      %form.addFieldListItem(15,"10 in 10");
      %form.addFieldListItem(15,"Specify Height");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("MakeHouse: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %house = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %house.static.center[0] SPC %house.static.center[1] SPC %house.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %house.static.size[0] SPC %house.static.size[1] SPC %house.static.size[2];
            return %value;
      
         //*** Handle the 'U Scale' field
         case 2:
            %value = %house.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 3:
            %value = %house.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 4:
            %value = %house.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 5:
            %value = %house.static.voffset;
            return %value;

         // Roof            
         case 8:
            %value = %house.static.roofheight;
            return %value;
            
         case 15:
             %value = %house.static.roofmode;
             return %value;

           
         // Structure
         case 9:
            %value = %house.static.createstructure;
            return %value;
            
         case 10:
            %value = %house.static.floorthickness;
            return %value;
            
         case 11:
            %value = %house.static.wallthickness;
            return %value;
            
         case 12:
            %value = %house.static.roofthickness;
            return %value;
            
         case 13:
            %value = %house.static.roofoverhang;
            return %value;
            
         case 14:
            %value = %house.static.ceiling;
            return %value;
            
      //%form.addField( 9, "Create Structure","checkbox");
      //%form.addField(10, "Floor Thickness", "numeric");
      //%form.addField(11, "Wall Thickness", "numeric");
      //%form.addField(12, "Roof Thickness", "numeric");
      //%form.addField(13, "Roof Overhang",  "numeric");
            
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("MakeHouse: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %house = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %house.static.center[%i] = getWord(%value, %i);
            }
            %house.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %house.static.size[%i] = getWord(%value, %i);
            }
            %house.changeSize = true;
      
         //*** Handle the 'U Scale' field
         case 2:
            %house.static.uscale = %value;
            %house.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 3:
            %house.static.vscale = %value;
            %house.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 4:
            %house.static.uoffset = %value;
            %house.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 5:
            %house.static.voffset = %value;
            %house.changeSize = true;
            
         //*** The roof percentage   
         case 14:
            %house.static.ceiling = %value;
            %house.changeSize = true;
            
         case 6:
            %house.static.roofpercentage = %value;
            %house.changeSize = true;

         //*** whether or not to use the height value (if not, then use percentage)            
         case 7:
            %house.static.useheight = %value;
            %house.changeSize = true;
         case 8:
            %house.static.roofheight = %value;
            %house.changeSize = true;
            
            
         // Structure
         case 9:
            %house.static.createstructure = %value;
            %house.changeSize = true;
            
         case 10:
            %house.static.floorthickness = %value;
            %house.changeSize = true;
            
         case 11:
            %house.static.wallthickness = %value;
            %house.changeSize = true;
            
         case 12:
            %house.static.roofthickness = %value;
            %house.changeSize = true;
            
         case 13:
            %house.static.roofoverhang = %value;
            %house.changeSize = true;
            
         case 15:
             echo("Roof value is " SPC %value);
             %house.static.roofmode = %value;
             %house.changeSize = true;
          
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%house.active)
      {
         %house.active = true;
         %house.changeSize = true;
         %house.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %house.update = tool.EDIT_UPDATE();
      %house.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%house, %house.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function MakeHouse_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0] = 1.0; // x
      %obj.size[1] = 1.0; // y
      %obj.size[2] = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;
      // roof angles
      %obj.roofpercentage = 50.0;
      %obj.roofheight = 8;
      %obj.useheight = 0;
      
      // Structure
      %obj.createstructure = 0;
      %obj.floorthickness  = 1.0;
      %obj.wallthickness   = 0.25;
      %obj.roofthickness   = 0.25;
      %obj.roofoverhang    = 0.25;
      %obj.ceiling         = 0;
      
      %obj.roofmode        = 2;
      
   }

   //*** Init the static object
   function MakeHouse_InitStatic(%static)
   {
      //*** Setup default values
      MakeHouse_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function MakeHouse_ResetCube(%house)
   {
      //*** Retrieve a pointer to the static data
      %static = %house.static;
      
      //*** Setup default values
      MakeHouse_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%house, %house.static);
   }
   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   function InvertPlane(%plane)
   {
     %nx = 0-(getWord(%plane,0));      
     %ny = 0-(getWord(%plane,1));      
     %nz = 0-(getWord(%plane,2));      
     %d  = 0-(getWord(%plane,3));      
     return (%nx SPC %ny SPC %nz SPC %d);
   }
   
   function MovePlane(%plane, %length)
   {
     %nx = (getWord(%plane,0));      
     %ny = (getWord(%plane,1));      
     %nz = (getWord(%plane,2));      
     %d  = (getWord(%plane,3)) + %length;      
     return (%nx SPC %ny SPC %nz SPC %d);
   }
   
   function GetBrush(%edit, %center, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin(%center);
      return %tempbrush;
   }
   
   //*** Build/modify the actual geometry
   function MakeHouse_MakeCubeGeometry(%house, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%house.active)
         return;

      if (%house.static.createstructure)
      {
         %brushesneeded = 1;
      }
      else
      {
         if (%house.static.ceiling)
            %brushesneeded = 8;
         else
            %brushesneeded = 7;
      }
      
      %count = %edit.getEditBrushCount();         
      if (%count != %brushesneeded)
      {
         %edit.clearAllNewBrushes();
      }
      
      %brush = GetBrush(%edit,"0 0 0",0);
      
      //*** Clear all planes on the brush to rebuild them.
      %brush.clearAllPlanes();
      
      //*** Define the cube's origin
      %center = %house.static.center[0] SPC %house.static.center[1] SPC %house.static.center[2];
      %brush.setOrigin(%center);

      //*** Setup texture parameters
      %brush.setTextureOffset(%house.static.uoffset, %house.static.voffset);
      %brush.setTextureScale(%house.static.uscale, %house.static.vscale);
      
      //*** Prepare for texturing
      %tx = %house.static.size[0] * 0.5;
      %ty = %house.static.size[1] * 0.5;
      %tz = %house.static.size[2] * 0.5;

      %perc = 40;
      switch(%house.static.roofmode)
      {
         case 0:
            %perc = 40;
            %roofbase = %tz - (((%ty * %perc))/100.0);
         case 1:
            %perc = 60;
            %roofbase = %tz - (((%ty * %perc))/100.0);
         case 2:
            %perc = 80;
            %roofbase = %tz - (((%ty * %perc))/100.0);
         case 3:
            %perc = 100;
            %roofbase = %tz - (((%ty * %perc))/100.0);
         case 4:
           %roofbase = %tz - %house.static.roofheight;
      }
      
      %roofcenterleft  = NewVector( %tx,0, %tz);
      %roofcenterright = NewVector(-%tx,0, %tz);
      
      %roofleft2       = NewVector( -%tx, -%ty, %roofbase);
      %roofleft1       = NewVector( -%tx,  %ty, %roofbase);
      
      %roofright1      = NewVector(  %tx, -%ty, %roofbase);
      %roofright2      = NewVector(  %tx,  %ty, %roofbase);

      //echo ("roofbase " SPC %roofbase);
      //*** Now build the main six planes.
      // left wall
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%tx));
      %leftplane = %brush.getLastPlane();

      // right wall
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ -1.0*(%tx));
      %rightplane = %brush.getLastPlane();

      // front wall
      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%ty));
      %frontplane = %brush.getLastPlane();
         
      // back wall
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 -1 0 " @ -1.0*(%ty));
      %backplane = %brush.getLastPlane();

      //%brush.setTexturePlanesByPoints(%roofcenterright,%roofcenterleft,%roofleft1, 0.0, 0.0);
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlaneByPoints(%roofcenterright,%roofcenterleft,%roofleft2);
      %roofleftplane = %brush.getLastPlane();
      
      //%brush.setTexturePlanesByPoints(%roofcenterleft,%roofcenterright,%roofright1, 0.0, 0.0);
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlaneByPoints(%roofcenterleft,%roofcenterright,%roofright2);
      %roofrightplane = %brush.getLastPlane();
      
      // floor
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ -1.0*(%tz));
      %floorplane = %brush.getLastPlane();
      
      
      %edit.updateBrush(%brush);
      
      if (%house.static.createstructure)
      {
         // Now create the structure.
         // Front Roof
         %brush = GetBrush(%edit,"0 0 0",0);
         %brush.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush.setTextureScale(%house.static.uscale, %house.static.vscale);
         
         %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush.addPlane(%rightplane);
         %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush.addPlane(%leftplane);
         %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush.addPlane(%frontplane);
         %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush.addPlane("0 -1 0" SPC %house.static.center[1]); // slice it right up the middle.
         %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush.addPlane(%roofrightplane);
         %tempplane1 = MovePlane(%roofrightplane,%house.static.roofthickness);
         %tempplane2 = InvertPlane(%tempplane1);
         %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush.addPlane(%tempplane2);
         %edit.updateBrush(%brush);

       
      
         // **************************
         // Back Roof
         %brush2 = GetBrush(%edit,"0 0 0",1);
         %brush2.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush2.setTextureScale(%house.static.uscale, %house.static.vscale);
         %brush2.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane(%rightplane);
         %brush2.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane(%leftplane);
         %brush2.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane("0 1 0" SPC 0-%house.static.center[1]); // slice it right up the middle.
         %brush2.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane(%backplane);
         %brush2.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane(%roofleftplane);
         %tempplane1 = MovePlane(%roofleftplane,%house.static.roofthickness);
         %tempplane2 = InvertPlane(%tempplane1);
         %brush2.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush2.addPlane(%tempplane2);
         %edit.updateBrush(%brush2); 
         
        
         
         // Make the front wall
         %brush3 = GetBrush(%edit,"0 0 0",2);
         %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
         
         %topplanefront = MovePlane(%roofleftplane,%house.static.roofthickness);
         %topplaneback  = MovePlane(%roofrightplane,%house.static.roofthickness);
         %inset = %house.static.roofoverhang;
         %farplane   = MovePlane(%frontplane,%inset);
         %nearplane  = InvertPlane(MovePlane(%farplane,%house.static.wallthickness));
         %topplane1  = MovePlane(%roofrightplane,%house.static.roofthickness);
         %topplane2  = MovePlane(%roofleftplane,%house.static.roofthickness);
         
         %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%farplane);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane1);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%floorplane);
         %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%nearplane);
         %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%rightplane,%inset));
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%leftplane,%inset));
         %edit.updateBrush(%brush3); 
         
         // Make the back wall
         %brush3 = GetBrush(%edit,"0 0 0",3);
         %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
         %farplane   = MovePlane(%backplane,%inset);
         %nearplane  = InvertPlane(MovePlane(%farplane,%house.static.wallthickness));
         %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%farplane);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane2);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%floorplane);
         %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%nearplane);
         %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%rightplane,%inset));
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%leftplane,%inset));
         %edit.updateBrush(%brush3); 
                  
         // Make the left wall
         %inset2 = %inset + %house.static.wallthickness;
         %brush3 = GetBrush(%edit,"0 0 0",4);
         %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
         %farplane   = MovePlane(%rightplane,%inset);
         %nearplane  = InvertPlane(MovePlane(%farplane,%house.static.wallthickness));
         
         %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%farplane);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane1);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane2);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%floorplane);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%nearplane);
         %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%frontplane,%inset2));
         %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%backplane,%inset2));
         %edit.updateBrush(%brush3);          
         
         // Make the right wall
         %brush3 = GetBrush(%edit,"0 0 0",5);
         %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
         %farplane   = MovePlane(%leftplane,%inset);
         %nearplane  = InvertPlane(MovePlane(%farplane,%house.static.wallthickness));
         
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%farplane);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane1);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%topplane2);
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%floorplane);
         %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%nearplane);
         %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%frontplane,%inset2));
         %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%backplane,%inset2));
         %edit.updateBrush(%brush3);          
         
         // Make the floor
         %brush3 = GetBrush(%edit,"0 0 0",6);
         %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
         %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
         
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(InvertPlane(%floorplane),-%house.static.floorthickness));
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(%floorplane);
          
         %floorinset = %inset + %house.static.wallthickness;
        
         %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%frontplane,%floorinset));
         %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%rightplane,%floorinset));
         %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%backplane,%floorinset));
         %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
         %brush3.addPlane(MovePlane(%leftplane,%floorinset));
         %edit.updateBrush(%brush3);  
         
         if (%house.static.ceiling)
         {
            // Make the ceiling.
            %brush3 = GetBrush(%edit,"0 0 0",7);
            %brush3.setTextureOffset(%house.static.uoffset, %house.static.voffset);
            %brush3.setTextureScale(%house.static.uscale, %house.static.vscale);
            
            %r1 = %roofbase+%house.static.center[2];
            %r2 = (%roofbase - %house.static.wallthickness)+%house.static.center[2];
            
            %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
            %brush3.addPlane("0 0 1 " SPC (-1.0 * %r1));
            
            %brush3.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
            %brush3.addPlane("0 0 -1 " SPC(%r2));
            
            // the sides - same as the floor
            %brush3.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
            %brush3.addPlane(MovePlane(%frontplane,%floorinset));
            %brush3.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
            %brush3.addPlane(MovePlane(%rightplane,%floorinset));
            %brush3.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);
            %brush3.addPlane(MovePlane(%backplane,%floorinset));
            %brush3.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
            %brush3.addPlane(MovePlane(%leftplane,%floorinset));
            %edit.updateBrush(%brush3);
         }
      }
   }
};

tool.register("MakeHouse", tool.typeInteractive(), tool.RFLAG_NONE(), "Build House" );
tool.setToolProperty("MakeHouse", "Icon", "icons/house");
tool.setToolProperty("MakeHouse", "Group", "Create");

