//
// Extrude.cs
//
// Defines the Extrude tool plug-in to create a bare-bones cube.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// 
//
// The house instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeSize   - Flag to indicate that the tool's bounding box size has changed
// length       - The length to extrude the face
// 
//
// Revision History:
// 5/3/2007       Jaimi McEntire     Created from SimpleBox.cs
//

package Extrude
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Extrude: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      
      echo ("Selection type: " SPC select.getSelectionType());
      //*** Check that we're the Brush selection mode is current -- the only one
      //*** that is supported at this time.
      if(select.getSelectionType() !$= "SelectFaces")
      {
         tool.activateErrorMsg = "This tool only works in the 'Faces' selection mode.";
         return tool.FUNC_BADSELECTMODE();
      }

      //*** With the appropriate preference set, make sure there is a selected brush or shape
      if(pref.getBool("ToolOperateSelectionOnly"))
      {
         if(select.count() < 1)
         {
            tool.activateErrorMsg = "Please select at least one face for this tool to operate on.";
            return tool.FUNC_BADSELECTMODE();
         }
      }
            
      //*** Has the static been set up?
      if(!%static.init)
      {
         Extrude_InitStatic(%static);
      }
      
      
      %map = scene.getCurrentMap();
      %static.faces = %map.getSelectedFaces();
      %static.facecount = select.count();

       echo ("Faces" SPC %static.faces);
      %brushid = getWord(%static.faces, 0);
      %faceid = getWord(%static.faces, 1);
      %faceplane = %map.getFacePlane(%brushid,%faceid);
      %brushpos  = %map.getBrushPosition(%brushid);

      echo ("Face plane" SPC %faceplane);
      %static.Length  = 1.0;
      %static.SelectedBrushID = %brushid;
      %static.SelectedFaceID = %faceid;
      %static.Normal[0] = 0;
      %static.Normal[1] = 0;
      %static.Normal[2] = 0;
      %static.center[0] = getWord(%brushpos,0);
      %static.center[1] = getWord(%brushpos,1);
      %static.center[2] = getWord(%brushpos,2);
      %static.transform = %map.getBrushTransformMatrix(%brushid);
      
      // storeSelectedBrushTransform();
      echo("Transform" SPC %static.transform);
      
      echo ("Brush Pos" SPC %brushpos);
      
      
      //*** Build the tool's instance
      %extrude = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %extrude.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%extrude, %static);

      //*** Setup some additional attributes for the house instance
      %extrude.dirty = tool.DIRTY_APPEARANCE();
      %extrude.active = false;
      %extrude.update = tool.EDIT_UPDATE();
      %extrude.changeCenter = false;
      %extrude.changeSize = true;
      
      //*** Pass along the instance
      %inst.instance = %extrude;
      %inst.flagsInterface = tool.IFLAG_RESETBUTTON() + tool.IFLAG_MAKEBUTTON() + tool.IFLAG_DRAWALLSAME();
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Extrude: Done(" @ %inst @ "," @ %static @ ")");

      %extrude = %inst.instance;
      
      if(%extrude)
      {         
         //*** Delete our instance
         %extrude.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Extrude: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Extrude: HandleCount(" @ %inst @ "," @ %event @ ")");

      %extrude = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %extrude.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Extrude: HandleInit(" @ %inst @ "," @ %event @ ")");

      %extrude = %inst.instance;

      //*** Make the tool active
      if(!%extrude.active)
      {
         %extrude.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%extrude, %extrude.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Extrude: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %extrude = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%extrude, %extrude.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Extrude: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %extrude = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%extrude, %extrude.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %extrude.dirty = tool.DIRTY_APPEARANCE();
      %extrude.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Extrude: Dirty(" @ %inst @ ")");

      %extrude = %inst.instance;
      
      return %extrude.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Extrude: Draw(" @ %inst @ "," @ %draw @ ")");

      %extrude = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%extrude.active)
         return;

      //*** Draw the standard bounding box
      //ToolBB::draw(%extrude, %extrude.static, %draw);

      //*** Indicate that we've drawn the tool
      %extrude.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Extrude: CheckEditAction(" @ %inst @ ")");

      %extrude = %inst.instance;
      
      return %extrude.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Extrude: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %extrude = %inst.instance;
      
      %extrude.update = tool.EDIT_DONOTHING();
      %extrude.active = false;
      %extrude.changeSize = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Extrude: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %extrude = %inst.instance;
      
      // Work on the actual geometry.
      Extrude_MakeGeometry(%extrude, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %extrude.update = tool.EDIT_DONOTHING();
      %extrude.changeSize = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Extrude: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %extrude = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %extrude.update = tool.EDIT_UPDATE();
            %extrude.active = true;
            %extrude.dirty = tool.DIRTY_APPEARANCE();
            %extrude.changeSize = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Extrude_DefaultValues(%extrude.static);
            ToolBB::rebuildBoundingBox(%extrude, %extrude.static);
            %extrude.update = tool.EDIT_UPDATE();
            %extrude.active = true;
            %extrude.dirty = tool.DIRTY_APPEARANCE();
            %extrude.changeSize = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%extrude.active)
            {
               %extrude.update = tool.EDIT_REJECT();
            }
            %extrude.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%extrude.active)
            {
               %extrude.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Extrude: Interface(" @ %inst @ "," @ %form @ ")");

      %extrude = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Extrude Face");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0,  "Length"    ,"numeric");
      if (%extrude.static.facecount < 2)
      {
         %form.addField( 1,  "Normal"    ,"distance3");
      }
      %form.addField( 2, "Conform"    ,"checkbox");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Extrude: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %extrude = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %extrude.static.Length;
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %extrude.static.Normal[0] SPC %extrude.static.Normal[1] SPC %extrude.static.Normal[2];
            return %value;
         case 2:
            %value = %extrude.static.conform;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   //************************************************************************************
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Extrude: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %extrude = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %extrude.static.Length = %value;
            %extrude.changeSize = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %extrude.static.Normal[%i] = getWord(%value, %i);
            }
            %extrude.changeSize = true;
            
         case 2:
            %extrude.static.conform = %value;
            %extrude.changeSize = true;
            
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%extrude.active)
      {
         %extrude.active = true;
         %extrude.changeSize = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %extrude.update = tool.EDIT_UPDATE();
      %extrude.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%extrude, %extrude.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function Extrude_DefaultValues(%obj)
   {
      %obj.Length = 1.0;
      %obj.conform = 0;
      
      %obj.Normal[0] = 0.0; // x
      %obj.Normal[1] = 1.0; // y
      %obj.Normal[2] = 0.0; // z
   }

   //*** Init the static object
   function Extrude_InitStatic(%static)
   {
      //*** Setup default values
      Extrude_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Extrude_ResetCube(%extrude)
   {
      //*** Retrieve a pointer to the static data
      %static = %extrude.static;
      
      //*** Setup default values
      Extrude_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%extrude, %extrude.static);
   }
   
   // *************************************************************************
   // *************************************************************************
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }

   // *************************************************************************
   // *************************************************************************
   function fabs(%val)
   {
      if (%val > 0) return %val;
      return 0-%val;
   }

   // *************************************************************************
   // The built in Torque matrix functions don't
   // actually take matrixes. This takes a 4x4 matrix
   // and a point, and returns the transformed point.
   // *************************************************************************
   function MatrixMultiplyPoint(%mat, %vsrc)
   {
      %vsrc0 = getWord(%vsrc,0);
      %vsrc1 = getWord(%vsrc,1);
      %vsrc2 = getWord(%vsrc,2);
      
      %mat11 = getWord(%mat,0);
      %mat21 = getWord(%mat,1);
      %mat31 = getWord(%mat,2);
      %mat41 = getWord(%mat,3);
      
      %mat12 = getWord(%mat,4);
      %mat22 = getWord(%mat,5);
      %mat32 = getWord(%mat,6);
      %mat42 = getWord(%mat,7);
      
      %mat13 = getWord(%mat,8);
      %mat23 = getWord(%mat,9);
      %mat33 = getWord(%mat,10);
      %mat43 = getWord(%mat,11);
      
      %mat14 = getWord(%mat,12);
      %mat24 = getWord(%mat,13);
      %mat34 = getWord(%mat,14);
      %mat44 = getWord(%mat,15);
      
      %x = (%vsrc0 * %mat11) + (%vsrc1 * %mat21) + (%vsrc2 * %mat31) + %mat41;
      %y = (%vsrc0 * %mat12) + (%vsrc1 * %mat22) + (%vsrc2 * %mat32) + %mat42;
      %z = (%vsrc0 * %mat13) + (%vsrc1 * %mat23) + (%vsrc2 * %mat33) + %mat43;
      %w = (%vsrc0 * %mat14) + (%vsrc1 * %mat24) + (%vsrc2 * %mat34) + %mat44;
      
      if (fabs(%w) < 0.000000000001) %w = 0.000000000001;
      %x = %x / %w;
      %y = %y / %w;
      %z = %z / %w;
      return NewVector(%x,%y,%z);
   }
   
   // *************************************************************************
   // Inverts the passed plane.
   // *************************************************************************
   function InvertPlane(%plane)
   {
     %nx = 0-(getWord(%plane,0));      
     %ny = 0-(getWord(%plane,1));      
     %nz = 0-(getWord(%plane,2));      
     %d  = 0-(getWord(%plane,3));      
     return (%nx SPC %ny SPC %nz SPC %d);
   }
   
   // *************************************************************************
   // Moves a plane by the distance provided.
   // *************************************************************************
   function MovePlane(%plane, %length)
   {
     %nx = (getWord(%plane,0));      
     %ny = (getWord(%plane,1));      
     %nz = (getWord(%plane,2));      
     %d  = (getWord(%plane,3)) + %length;      
     return (%nx SPC %ny SPC %nz SPC %d);
   }
   
   // *************************************************************************
   // Gets a brush from the edit buffer. If it doesn't exist, it creates it.
   // *************************************************************************
   function GetBrush(%edit, %center, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin(%center);
      return %tempbrush;
   }   

   // ***********************************************************************
   // Returns a vector that is is the cross of the edge and the adjoining face.
   // this gives us a vector on the plane of the adjoining face that points up
   // into the selected edge. Used when conforming the extruded brush to the
   // adjoining face.
   // ***********************************************************************
   function FindVector(%extrude,%vector, %brushid, %faceid, %v, %nv)
   {
      // find the face that connects to this edge.
      // 
      // if the plane is too similar, then just return the passed in vector (90deg).
      // otherwise, return a vector along the slope of the plane.
      %map = scene.getCurrentMap();
      %facecount = %map.getNumBrushFaces(%brushid);
 
      %v1id = %map.getFaceVertexID(%brushid,%faceid,%v);
      %v2id = %map.getFaceVertexID(%brushid,%faceid,%nv);
      
      // Find the face that contains both v1id and v2id.
      
      %facecount1 = %map.getNumVertexFaces(%brushid,%v1id);
      %facecount2 = %map.getNumVertexFaces(%brushid,%v2id);
      
      for (%i = 0;%i < %facecount; %i++)
      {
         %matches = 0;
         // find the face for this edge.
         %tfaceid = %map.getFaceID(%brushid,%i);
         if (%tfaceid != %faceid)
         {
            %vertexcount = %map.getNumFaceVertices(%brushid,%tfaceid);
            
            for (%fv=0;%fv < %vertexcount;%fv++)
            {
               %nexfv = %fv+1;
               if (%nexfv >= %vertexcount) 
                 %nexfv = 0;
                 
               %tv1 = %map.getFaceVertexID(%brushid,%tfaceid,%fv);
               %tv2 = %map.getFaceVertexID(%brushid,%tfaceid,%nexfv);
               
               if ((%tv1 == %v2id && %tv2 == %v1id) || (%tv1 == %v1id && %tv2 == %v2id))
               {
                  // Found it!
                  %faceplane = %map.getFacePlane(%brushid,%tfaceid);
                  %facenormal = getWord(%faceplane,0) SPC getWord(%faceplane,1) SPC getWord(%faceplane,2);
                  %dot = VectorDot(%facenormal,VectorNormalize(%vector));
                  // validate it's not too close...
                  // echo("%dot=" SPC %dot);
                  if (%dot < 0.95)
                  {
                     // Get the normal of the plane.
                     // Cross it with the vector that defines the edge
                     %vec1 = %map.getVertexPosition(%brushid,%v1id);
                     %vec2 = %map.getVertexPosition(%brushid,%v2id);
                     %edgevec = VectorSub(%vec2,%vec1);
                     %edgevec = VectorNormalize(%edgevec);
                     %newvector = VectorCross(%edgevec,%facenormal);
                     // Make sure it's a unit vector
                     %newvector = VectorNormalize(%newvector);
                     // The new vector points up the slope of the adjoining face
                     return %newvector;
                  }
               }
            }
         }
      }
      //echo("Connecting face not found");
      return %vector;
   }

   // ***********************************************************************
   // Extrudes the face specified by %faceid of brush %brushid.
   // ***********************************************************************
   function ExtrudeFace(%extrude, %edit, %normal, %index, %brushid, %faceid)
   {
      %center = %extrude.static.center[0] SPC %extrude.static.center[1] SPC %extrude.static.center[2];
      %brush = GetBrush(%edit,"0 0 0",%index);
      %map = scene.getCurrentMap();
     
      %transform = %map.getBrushTransformMatrix(%brushid);      

      %vertpos1 = %map.getFaceVertexPosition(%brushid,%faceid,0);
      %vertpos2 = %map.getFaceVertexPosition(%brushid,%faceid,1);
      %vertpos3 = %map.getFaceVertexPosition(%brushid,%faceid,2);

      %vertpos1 = MatrixMultiplyPoint(%transform,%vertpos1);
      %vertpos2 = MatrixMultiplyPoint(%transform,%vertpos2);
      %vertpos3 = MatrixMultiplyPoint(%transform,%vertpos3);
      
      // Add the connecting plane
      %brush.setTexturePlanesByPoints(%vertpos1, %vertpos2, %vertpos3, 0.0, 0.0);
      %brush.addPlaneByPoints(%vertpos3,%vertpos2,%vertpos1);

      %vector = VectorScale(%normal,%extrude.static.length);
         
      %vertpos4 = VectorAdd(%vertpos3,%vector);
      %vertpos5 = VectorAdd(%vertpos2,%vector);
      %vertpos6 = VectorAdd(%vertpos1,%vector); 
         
      // Add the far plane
      %brush.addPlaneByPoints(%vertpos6,%vertpos5,%vertpos4);
      %vertexcount = %map.getNumFaceVertices(%brushid,%faceid);
         
      // Add a plane for each edge.
      for (%v = 0;%v<%vertexcount; %v++)
      {
         %nv = %v+1;
         if (%nv >= %vertexcount) %nv = 0;
         
         if (%extrude.static.conform)
         {
            %facevector = FindVector(%extrude, %vector, %brushid, %faceid, %v, %nv);
         }
         else
         {
            %facevector = %vector;
         }
         
         %vertpos1 = %map.getFaceVertexPosition(%brushid,%faceid,%v);
         %vertpos2 = %map.getFaceVertexPosition(%brushid,%faceid,%nv);
            
         %vertpos1 = MatrixMultiplyPoint(%transform,%vertpos1);
         %vertpos2 = MatrixMultiplyPoint(%transform,%vertpos2);
            
         %vertpos3 = VectorAdd(%facevector,%vertpos2);
         %brush.setTexturePlanesByPoints(%vertpos1, %vertpos2, %vertpos3, 0.0, 0.0);
         %brush.addPlaneByPoints(%vertpos1,%vertpos2,%vertpos3);
      }
      %edit.updateBrush(%brush);
   }
   
   //************************************************************
   //*** Build/modify the actual geometry
   //************************************************************
   function Extrude_MakeGeometry(%extrude, %edit)
   {
      %map = scene.getCurrentMap();

      // If there is only one face selected, then we can use the modified normal.
      // Note: The normal is by default the face normal. However, the user can 
      // modify it in the edit panel.
      if (%extrude.static.facecount == 1)
      {            
         %brushid = getWord(%extrude.static.faces, 0);
         %faceid = getWord(%extrude.static.faces, 1);
         if (%extrude.static.Normal[0] == 0 && %extrude.static.Normal[1] == 0 && %extrude.static.Normal[2] == 0)
         {
            %plane = %map.getFacePlane(%brushid,%faceid);
            %extrude.static.Normal[0] = getWord(%plane,0);
            %extrude.static.Normal[1] = getWord(%plane,1);
            %extrude.static.Normal[2] = getWord(%plane,2);
         }
         %normal = NewVector(%extrude.static.Normal[0],%extrude.static.Normal[1],%extrude.static.Normal[2]);
         ExtrudeFace(%extrude, %edit, %normal, 0,%brushid,%faceid);
      }
      else
      {
         // Use the face normal to extrude all of the selected faces.
         for (%i = 0; %i < %extrude.static.facecount; %i++)
         {
            %bpos = %i * 2;
            %brushid = getWord(%extrude.static.faces, %bpos);
            %faceid = getWord(%extrude.static.faces, %bpos + 1);
            %plane = %map.getFacePlane(%brushid,%faceid);
            %normal = getWord(%plane,0) SPC getWord(%plane,1) SPC getWord(%plane,2);
            ExtrudeFace(%extrude, %edit, %normal, %i,%brushid,%faceid);
         }
      }
   }
};

tool.register("Extrude", tool.typeInteractive(), tool.RFLAG_NONE(), "Extrude Face" );
tool.setToolProperty("Extrude", "Icon", "icons/extrude");
tool.setToolProperty("Extrude", "Group","Edit");

