//
// Column.cs
//
// Defines the Column tool plug-in to create a Column:
//
//    XXXX
//     CC
//     CC
//     CC
//    XXXX
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Segments    (Default 1 - how many sections to make the column)
// Sides (3+)
// Base Sides  (Default 4. Min 3).
// Base lip    (default 0.1);
// Base Height (default 0.2).
// Array
// X count (Default 1)
// Y count (Default 1)
// X Spacing (Default 4)
// Y Spacing (Default 4).


package Column
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Column: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Column_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %Column = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %Column.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%Column, %static);

      //*** Setup some additional attributes for the cube instance
      %Column.dirty = tool.DIRTY_NONE();
      %Column.active = false;
      %Column.update = tool.EDIT_DONOTHING();
      %Column.changeCenter = false;
      %Column.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %Column;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Column: Done(" @ %inst @ "," @ %static @ ")");

      %Column = %inst.instance;
      
      if(%Column)
      {         
         //*** Delete our instance
         %Column.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Column: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Column: HandleCount(" @ %inst @ "," @ %event @ ")");

      %Column = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %Column.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Column: HandleInit(" @ %inst @ "," @ %event @ ")");

      %Column = %inst.instance;

      //*** Make the tool active
      if(!%Column.active)
      {
         %Column.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%Column, %Column.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Column: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %Column = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%Column, %Column.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Column: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %Column = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%Column, %Column.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %Column.dirty = tool.DIRTY_APPEARANCE();
      %Column.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Column: Dirty(" @ %inst @ ")");

      %Column = %inst.instance;
      
      return %Column.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Column: Draw(" @ %inst @ "," @ %draw @ ")");

      %Column = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%Column.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%Column, %Column.static, %draw);

      //*** Indicate that we've drawn the tool
      %Column.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Column: CheckEditAction(" @ %inst @ ")");

      %Column = %inst.instance;
      
      return %Column.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Column: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %Column = %inst.instance;
      
      %Column.update = tool.EDIT_DONOTHING();
      %Column.active = false;
      %Column.changeSize = false;
      %Column.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Column: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %Column = %inst.instance;
      
      // Work on the actual geometry.
      Column_MakeGeometry(%Column, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %Column.update = tool.EDIT_DONOTHING();
      %Column.changeSize = false;
      %Column.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Column: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %Column = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %Column.update = tool.EDIT_UPDATE();
            %Column.active = true;
            %Column.dirty = tool.DIRTY_APPEARANCE();
            %Column.changeSize = true;
            %Column.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Column_DefaultValues(%Column.static);
            ToolBB::rebuildBoundingBox(%Column, %Column.static);
            %Column.update = tool.EDIT_UPDATE();
            %Column.active = true;
            %Column.dirty = tool.DIRTY_APPEARANCE();
            %Column.changeSize = true;
            %Column.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%Column.active)
            {
               %Column.update = tool.EDIT_REJECT();
            }
            %Column.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%Column.active)
            {
               %Column.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Column: Interface(" @ %inst @ "," @ %form @ ")");

      %Column = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Create Columns");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");

      // TODO: Add some kid of slope, so we can have curves, etc.      
      %form.addField(-1,  "Column" ,"divider");
      %form.addField( 2,  "Segments",    "numericinteger");
      %form.addField( 3,  "Sides",       "numericinteger");
      %form.addField( 4,  "Rectangular", "checkbox");
      
      %form.addField(-1,  "Base" ,"divider");
      %form.addField( 10,  "Sides",        "numericinteger");
      %form.addField( 11,  "Side Spacing", "numeric");
      %form.addField( 12,  "Height",       "numeric");
      %form.addField( 13,  "Rectangular",  "checkbox");
      
      %form.addField(-1,  "Array" ,"divider");
      %form.addField(20,   "X Count", "numericinteger");
      %form.addField(21,   "Y Count", "numericinteger");
      %form.addField(22,   "X Spacing", "numeric");
      %form.addField(23,   "Y Spacing", "numeric");
/*      
      %obj.segments  = 1;     // number of segments in main column.
      %obj.sides     = 6;     // sides of the column
      %obj.basesides = 4;     // sides of the bases (top and bottom)
      %obj.baselip   = 0.1;   // spacing of base from column side
      %obj.baseheight = 0.2;  // height of the base (if 0, then no base)
      %obj.xcount     = 1;    // count along the X axis.
      %obj.ycount     = 1;    // count along the Y axis.
      %obj.xspace     = 4;    // Spacing
      %obj.yspace     = 4;    // Spacing
*/      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      //********************************************
      %form.setFieldMinLimit(2, 1);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 3);
      //********************************************
      %form.setFieldMinLimit(20, 0);
      %form.setFieldMinLimit(21, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Column: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %Column = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %Column.static.center[0] SPC %Column.static.center[1] SPC %Column.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %Column.static.size[0] SPC %Column.static.size[1] SPC %Column.static.size[2];
            return %value;
      
         //*** Handle the 'Segments' field
         case 2:
            %value = %Column.static.segments;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %Column.static.sides;
            return %value;
            
         case 4:
            %value = %Column.static.rectangular;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 5:
            %value = %Column.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %Column.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %Column.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %Column.static.voffset;
            return %value;            
            
         case 10:
            %value = %Column.static.basesides;
            return %value;            
            
         case 11:
            %value = %Column.static.baselip;
            return %value;            
            
         case 12:
            %value = %Column.static.baseheight;
            return %value;            
         case 13:
            %value = %Column.static.baserectangular;
            return %value;            
            
         case 20:
            %value = %Column.static.xcount;
            return %value;            
         case 21:
            %value = %Column.static.ycount;
            return %value;            
         case 22:
            %value = %Column.static.xspace;
            return %value;            
         case 23:
            %value = %Column.static.yspace;
            return %value;            
         
      }
      
      return 0;
   }



   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Column: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %Column = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %Column.static.center[%i] = getWord(%value, %i);
            }
            %Column.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %Column.static.size[%i] = getWord(%value, %i);
            }
            %Column.changeSize = true;
      
         //*** Handle the Segments field
         case 2:
            %Column.static.segments = %value;
            %Column.static.forceclear = 1;
            %Column.changeSize = true;
      
         //*** Handle the Sides field
         case 3:
            %Column.static.sides = %value;
            %Column.changeSize = true;
         case 4:
            %Column.static.rectangular = %value;
            %Column.changeSize = true;

         //*** Handle the 'U Scale' field
         case 5:
            %Column.static.uscale = %value;
            %Column.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %Column.static.vscale = %value;
            %Column.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %Column.static.uoffset = %value;
            %Column.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %Column.static.voffset = %value;
            %Column.changeSize = true;   
               
         case 10:
            %Column.static.basesides =%value;
            %Column.changeSize = true;   
            
         case 11:
            %Column.static.baselip = %value;
            %Column.changeSize = true;   
            
         case 12:
            %Column.static.baseheight=%value;
            %Column.changeSize = true;   

         case 13:
            %Column.static.baserectangular=%value;
            %Column.changeSize = true;   
            
         case 20:
            %Column.static.xcount=%value;
            %Column.static.forceclear = true;
            %Column.changeSize = true;   
            
         case 21:
            %Column.static.ycount=%value;
            %Column.static.forceclear = true;
            %Column.changeSize = true;   
            
         case 22:
            %Column.static.xspace=%value;
            %Column.changeSize = true;   
            
         case 23:
            %Column.static.yspace=%value;
            %Column.changeSize = true;   
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%Column.active)
      {
         %Column.active = true;
         %Column.changeSize = true;
         %Column.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %Column.update = tool.EDIT_UPDATE();
      %Column.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%Column, %Column.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function Column_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;  
      
      %obj.segments  = 1;     // number of segments in main column.
      %obj.sides     = 6;     // sides of the column
      %obj.rectangular = false;
      %obj.basesides = 4;     // sides of the bases (top and bottom)
      %obj.baselip   = 0.0;   // spacing of base from column side
      %obj.baseheight = 0.2;  // height of the base (if 0, then no base)
      %obj.baserectangular = true;
      %obj.xcount     = 1;    // count along the X axis.
      %obj.ycount     = 1;    // count along the Y axis.
      %obj.xspace     = 4;    // Spacing
      %obj.yspace     = 4;    // Spacing
      %obj.forceclear = false;
   }

   //*** Init the static object
   function Column_InitStatic(%static)
   {
      //*** Setup default values
      Column_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function Column_ResetCube(%Column)
   {
      //*** Retrieve a pointer to the static data
      %static = %Column.static;
      
      //*** Setup default values
      Column_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%Column, %Column.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   function GetFaceCenter(%Column,%slicenumber)
   {
      %radians = 6.283185307;
      %tx = %Column.static.size[0] * 0.5;
      %ty = %Column.static.size[1] * 0.5;
      %tz = %Column.static.size[2] * 0.5;      
      %tz = 50;
      %ti = (%slicenumber / %Column.static.segments) * %radians;
      %center = NewVector(mCos(%ti) * %tx, mSin(%ti) * %ty, %tz);
      return %center;
   }
   
   function GetFacePoint(%Column,%edgenumber,%rx, %ry, %zpos, %sides)
   {
      %radians = 6.283185307; // close enough.
      %angle = (%edgenumber / %sides) * %radians;
      %point = NewVector(mCos(%angle) * %rx, mSin(%angle) * %ry, %zpos);
      return %point;
   }      
      
   function MakeSolid(%edit,%Column, %mins, %maxs, %tx,%ty,%tz, %sides)
   {
      %brush = GetBrush(%edit,%Column.currentbrush);
      %Column.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      %rx = (%maxx - %minx) * 0.5;
      %ry = (%maxy - %miny) * 0.5;
      
      %center = NewVector(%minx + %rx,%miny + %ry,%Column.static.center[2]); 
      
      for (%face = 0; %face < %sides; %face++)
      {
         if(%face == %sides-1)
         {
            %nextface = 0;
         }
         else
         {
            %nextface = %face+1;
         }
         // calc the points for the face.
         %pt1 = VectorAdd(%center,GetFacePoint(%Column,%face,%rx,%ry,%minz,%sides));
         %pt2 = VectorAdd(%center,GetFacePoint(%Column,%nextface,%rx,%ry,%minz,%sides));
         %pt3 = VectorAdd(%center,GetFacePoint(%Column,%face,%rx,%ry,%maxz,%sides));
         
         %brush.setTexturePlanesByPoints(%pt2,%pt1,%pt3, 0.0, 0.0);
         %brush.addPlaneByPoints(%pt2,%pt1,%pt3);
         // Set the texture parms.
         // Add the face to the brush.
      }
      
      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
      
   }
   // MakeCuboid
   function MakeCuboid(%edit,%Column, %mins, %maxs, %tx,%ty,%tz)
   {
      %brush = GetBrush(%edit,%Column.currentbrush);
      %Column.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      
      %brush.setTextureOffset(%Column.static.uoffset, %Column.static.voffset);
      %brush.setTextureScale(%Column.static.uscale, %Column.static.vscale);     
        
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%maxx));
      
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ 1.0*(%minx));

      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%maxy));
         
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);   
      %brush.addPlane("0 -1 0 " @ 1.0*(%miny));

      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
   }
      
      
   function CreateColumn(%edit,%Column,%mins,%maxs,%tx,%ty,%tz)
   {
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);

      if (%Column.static.baseheight > 0)
      {    
        // Make the base. Reset the mins.
        %minz = %minz+%Column.static.baseheight;
        %maxz = %maxz-%Column.static.baseheight;
        %bmaxs = setWord(%maxs,2,%minz);
        if (%Column.static.baserectangular)
        {
           MakeCuboid(%edit,%Column,%mins,%bmaxs,%tx,%ty,%tz);
        }
        else
        {
           MakeSolid(%edit,%Column,%mins,%bmaxs,%tx,%ty,%tz,%Column.static.basesides);
        }
        %mins = setWord(%mins,2,%minz);
        // Make the capital. Reset the maxs.
        %bmins = setWord(%mins,2,%maxz);
        if (%Column.static.baserectangular)
        {
           MakeCuboid(%edit,%Column,%bmins,%maxs,%tx,%ty,%tz);
        }
        else
        {
           MakeSolid(%edit,%Column,%bmins,%maxs,%tx,%ty,%tz,%Column.static.basesides);
        }
        %maxs = setWord(%maxs,2,%maxz);
      }
      
      // get the segment height.
      // create the columns.
      %minx = %minx + %Column.static.baselip;
      %maxx = %maxx - %Column.static.baselip;
      %miny = %miny + %Column.static.baselip;
      %maxy = %maxy - %Column.static.baselip;
      %sz = %maxz - %minz;
      %ht =  %sz / %Column.static.segments;
      
      %cmins = NewVector(%minx,%miny,%minz);
      %cmaxs = NewVector(%maxx,%maxy,%minz+%ht);      
      
      if ((%minx < %maxx) && (%miny < %maxy) && (%minz < %maxz))
      {
         for (%inx = 0; %inx < %Column.static.segments;%inx++)
         {
            if (%Column.static.rectangular)
            {
               MakeCuboid(%edit,%Column,%cmins,%cmaxs,%tx,%ty,%tz);
            }
            else
            {
               MakeSolid(%edit,%Column,%cmins,%cmaxs,%tx,%ty,%tz,%Column.static.sides);
            }
            %minz = getWord(%cmaxs,2);
            %maxz = %minz + %ht;
            %cmins = NewVector(%minx,%miny,%minz);
            %cmaxs = NewVector(%maxx,%maxy,%maxz);
         }
      }
      // MakeCuboid(%edit,%Column,%mins,%maxs,%tx,%ty,%tz);
      // build base
      // build capital
      // build column
   }
      
   //*** Build/modify the actual geometry
   function Column_MakeGeometry(%Column, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%Column.active)
         return;

      %Column.currentbrush = 0;
      if (%Column.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %Column.static.forceclear = 0;
      }
   
      //*** Work on the sizing of the brush, if appropriate
      %center = %Column.static.center[0] SPC %Column.static.center[1] SPC %Column.static.center[2];
      %tx = %Column.static.size[0] * 0.5;
      %ty = %Column.static.size[1] * 0.5;
      %tz = %Column.static.size[2] * 0.5;

      %range = 100-%Column.static.percent;
      %shrink = %range / (%Column.static.segments-1);
      %currentsize = 100;
      
      %sz = %Column.static.size[2];
      %ht =  %sz / %Column.static.segments;
      %cmins = %mins;
      %cmaxs = setWord(%maxs,2,getWord(%mins,2)+%ht);
      %lastx = %tx;
      %lasty = %ty;
      
      
      for (%xrow = 0; %xrow < %Column.static.xcount; %xrow++)
      {
         for (%yrow = 0; %yrow < %Column.static.ycount; %yrow++)
         {
            %xpos = (%Column.static.center[0]) + (%xrow * %Column.static.xspace);
            %ypos = (%Column.static.center[1]) + (%yrow * %Column.static.yspace);
            %mins = NewVector(%xpos-%tx,%ypos-%ty,%Column.static.center[2]-%tz);
            %maxs = NewVector(%xpos+%tx,%ypos+%ty,%Column.static.center[2]+%tz);
            CreateColumn(%edit,%Column,%mins,%maxs,%tx,%ty,%tz);
         }
      }
   }
};

tool.register("Column", tool.typeInteractive(), tool.RFLAG_NONE(), "Create Columns" );
tool.setToolProperty("Column", "Icon", "icons/Column");
tool.setToolProperty("Column", "Group", "Create");
