//#region
// BlockStack.cs
//
// Defines the BlockStack tool plug-in to create a BlockStack:
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
//
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// X Segments (Default 2)
// Y Segments (Default 2)
// Z Segments (Default 2)
//
// X Spacing  (Default 0)
// Y Spacing  (Default 0)
// Z Spacing  (Default 0)
//#endregion


package BlockStack
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("BlockStack: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         BlockStack_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %BlockStack = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %BlockStack.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeBox(), true);
      ToolBB::rebuildBoundingBox(%BlockStack, %static);

      //*** Setup some additional attributes for the cube instance
      %BlockStack.dirty = tool.DIRTY_NONE();
      %BlockStack.active = false;
      %BlockStack.update = tool.EDIT_DONOTHING();
      %BlockStack.changeCenter = false;
      %BlockStack.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %BlockStack;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("BlockStack: Done(" @ %inst @ "," @ %static @ ")");

      %BlockStack = %inst.instance;
      
      if(%BlockStack)
      {         
         //*** Delete our instance
         %BlockStack.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("BlockStack: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("BlockStack: HandleCount(" @ %inst @ "," @ %event @ ")");

      %BlockStack = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %BlockStack.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("BlockStack: HandleInit(" @ %inst @ "," @ %event @ ")");

      %BlockStack = %inst.instance;

      //*** Make the tool active
      if(!%BlockStack.active)
      {
         %BlockStack.active = true;
      }
      
      //*** Allow the bounding box helper to set up the handles
      return ToolBB::initHandles(%BlockStack, %BlockStack.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("BlockStack: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %BlockStack = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%BlockStack, %BlockStack.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("BlockStack: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %BlockStack = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%BlockStack, %BlockStack.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %BlockStack.dirty = tool.DIRTY_APPEARANCE();
      %BlockStack.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("BlockStack: Dirty(" @ %inst @ ")");

      %BlockStack = %inst.instance;
      
      return %BlockStack.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("BlockStack: Draw(" @ %inst @ "," @ %draw @ ")");

      %BlockStack = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%BlockStack.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%BlockStack, %BlockStack.static, %draw);

      //*** Indicate that we've drawn the tool
      %BlockStack.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("BlockStack: CheckEditAction(" @ %inst @ ")");

      %BlockStack = %inst.instance;
      
      return %BlockStack.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("BlockStack: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %BlockStack = %inst.instance;
      
      %BlockStack.update = tool.EDIT_DONOTHING();
      %BlockStack.active = false;
      %BlockStack.changeSize = false;
      %BlockStack.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("BlockStack: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %BlockStack = %inst.instance;
      
      // Work on the actual geometry.
      BlockStack_MakeGeometry(%BlockStack, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %BlockStack.update = tool.EDIT_DONOTHING();
      %BlockStack.changeSize = false;
      %BlockStack.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("BlockStack: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %BlockStack = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %BlockStack.update = tool.EDIT_UPDATE();
            %BlockStack.active = true;
            %BlockStack.dirty = tool.DIRTY_APPEARANCE();
            %BlockStack.changeSize = true;
            %BlockStack.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            BlockStack_DefaultValues(%BlockStack.static);
            ToolBB::rebuildBoundingBox(%BlockStack, %BlockStack.static);
            %BlockStack.update = tool.EDIT_UPDATE();
            %BlockStack.active = true;
            %BlockStack.dirty = tool.DIRTY_APPEARANCE();
            %BlockStack.changeSize = true;
            %BlockStack.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%BlockStack.active)
            {
               %BlockStack.update = tool.EDIT_REJECT();
            }
            %BlockStack.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%BlockStack.active)
            {
               %BlockStack.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("BlockStack: Interface(" @ %inst @ "," @ %form @ ")");

      %BlockStack = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Create BlockStacks");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");

      // TODO: Add some kid of slope, so we can have curves, etc.      
      %form.addField(-1,  "BlockStack" ,"divider");
      
      //%form.addField( 2,  "Segments",    "numericinteger");
      %form.addField( 3,  "Sides",       "numericinteger");
      %form.addField( 4,  "Rectangular", "checkbox");
      
      %form.addField(-1,  "Array" ,"divider");
      %form.addField(20,   "X Count", "numericinteger");
      %form.addField(21,   "Y Count", "numericinteger");
      %form.addField(22,   "Z Count", "numericinteger");
      %form.addField(23,   "X Spacing", "numeric");
      %form.addField(24,   "Y Spacing", "numeric");
      %form.addField(25,   "Z Spacing", "numeric");
      
      %form.addField(30,   "X Offset on Z", "numeric");
      %form.addField(32,   "Alternate XOffset", "checkbox");
      %form.addField(31,   "Y Offset on X", "numeric");
      %form.addField(33,   "Alternate YOffset", "checkbox");
      
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 5, "U Scale"  ,"numeric");
      %form.addField( 6, "V Scale"  ,"numeric");
      %form.addField( 7, "U Offset" ,"numeric");
      %form.addField( 8, "V Offset" ,"numeric");      
      
      //********************************************
      %form.setFieldMinLimit(2, 1);
      %form.setFieldMinLimit(3, 3);
      %form.setFieldMinLimit(10, 3);
      //********************************************
      %form.setFieldMinLimit(20, 0);
      %form.setFieldMinLimit(21, 0);
      %form.setFieldMinLimit(22, 0);  
     
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("BlockStack: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %BlockStack = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %BlockStack.static.center[0] SPC %BlockStack.static.center[1] SPC %BlockStack.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %BlockStack.static.size[0] SPC %BlockStack.static.size[1] SPC %BlockStack.static.size[2];
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %BlockStack.static.sides;
            return %value;
            
         case 4:
            %value = %BlockStack.static.rectangular;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 5:
            %value = %BlockStack.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 6:
            %value = %BlockStack.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 7:
            %value = %BlockStack.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 8:
            %value = %BlockStack.static.voffset;
            return %value;            
            
         case 20:
            %value = %BlockStack.static.xcount;
            return %value;            
         case 21:
            %value = %BlockStack.static.ycount;
            return %value;            
         case 22:
            %value = %BlockStack.static.zcount;
            return %value;            
            
         case 23:
            %value = %BlockStack.static.xspace;
            return %value;            
         case 24:
            %value = %BlockStack.static.yspace;
            return %value;            
         case 25:
            %value = %BlockStack.static.zspace;
            return %value;  
       
         case 30:
            %value = %BlockStack.static.xoff;
            return %value;  
            
         case 31:
            %value = %BlockStack.static.yoff;
            return %value;  
            
         case 32:
            %value = %BlockStack.static.alternatex;         
            return %value;  
            
         case 33:
            %value = %BlockStack.static.alternatey;         
            return %value;  
      }
      
      return 0;
   }



   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("BlockStack: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %BlockStack = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %BlockStack.static.center[%i] = getWord(%value, %i);
            }
            %BlockStack.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %BlockStack.static.size[%i] = getWord(%value, %i);
            }
            %BlockStack.changeSize = true;
      
      
         //*** Handle the Sides field
         case 3:
            %BlockStack.static.sides = %value;
            %BlockStack.changeSize = true;
         case 4:
            %BlockStack.static.rectangular = %value;
            %BlockStack.changeSize = true;

         //*** Handle the 'U Scale' field
         case 5:
            %BlockStack.static.uscale = %value;
            %BlockStack.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 6:
            %BlockStack.static.vscale = %value;
            %BlockStack.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 7:
            %BlockStack.static.uoffset = %value;
            %BlockStack.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 8:
            %BlockStack.static.voffset = %value;
            %BlockStack.changeSize = true;   
               
         case 20:
            %BlockStack.static.xcount=%value;
            %BlockStack.static.forceclear = true;
            %BlockStack.changeSize = true;   
            
         case 21:
            %BlockStack.static.ycount=%value;
            %BlockStack.static.forceclear = true;
            %BlockStack.changeSize = true;   
            
         case 22:
            %BlockStack.static.zcount=%value;
            %BlockStack.static.forceclear = true;
            %BlockStack.changeSize = true;   
            
         case 23:
            %BlockStack.static.xspace=%value;
            %BlockStack.changeSize = true;   
            
         case 24:
            %BlockStack.static.yspace=%value;
            %BlockStack.changeSize = true;   
            
         case 25:
            %BlockStack.static.zspace=%value;
            %BlockStack.changeSize = true;   
            
         case 30:
            %BlockStack.static.xoff = %value;
            %BlockStack.changeSize = true;   
            
         case 31:
            %BlockStack.static.yoff = %value;
            %BlockStack.changeSize = true;   
            
         case 32:
            %BlockStack.static.alternatex = %value;         
            %BlockStack.changeSize = true;
               
         case 33:
            %BlockStack.static.alternatey = %value;         
            %BlockStack.changeSize = true;   
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%BlockStack.active)
      {
         %BlockStack.active = true;
         %BlockStack.changeSize = true;
         %BlockStack.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %BlockStack.update = tool.EDIT_UPDATE();
      %BlockStack.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%BlockStack, %BlockStack.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************

   
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   //*** Reset the given object to default values
   function BlockStack_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0]   = 1.0; // x
      %obj.size[1]   = 1.0; // y
      %obj.size[2]   = 1.0; // z
      
      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;  
      
      %obj.segments  = 1;     // number of segments in main BlockStack.
      %obj.sides     = 4;     // sides of the BlockStack
      %obj.rectangular = true;
      
      %obj.xcount     = 2;    // count along the X axis.
      %obj.ycount     = 1;    // count along the Y axis.
      %obj.zcount     = 2;
      
      %obj.xspace     = 0;    // Spacing
      %obj.yspace     = 0;    // Spacing
      %obj.zspace     = 0;    // Spacing
      
      %obj.xoff = 0;
      %obj.yoff = 0;
      %obj.zoff = 0;
      %obj.alternatex = false;
      %obj.alternatey = false;
      
      %obj.forceclear = false;
   }

   //*** Init the static object
   function BlockStack_InitStatic(%static)
   {
      //*** Setup default values
      BlockStack_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function BlockStack_ResetCube(%BlockStack)
   {
      //*** Retrieve a pointer to the static data
      %static = %BlockStack.static;
      
      //*** Setup default values
      BlockStack_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%BlockStack, %BlockStack.static);
   }
      
   function NewVector(%cx, %cy, %cz)
   {
      return (%cx SPC %cy SPC %cz);
   }
   
   // Function
   function PlaneFromPoints(%v0, %v1, %v2)
   {
      %t1 = VectorSub(%v0,%v1);
      %t2 = VectorSub(%v2,%v1);
      %vector = VectorCross(%t1,%t2);
      %normal = VectorNormalize(%vector);
      %dist   = VectorDot(%v0,%normal);
      return %normal SPC %dist;
   }
   
   function GetBrush(%edit, %index)
   {
      %tempbrush = %edit.getEditBrush(%index);
      if(%tempbrush == -1)
      {
         %tempbrush = new MapBrush();
      }
      %tempbrush.clearAllPlanes();
      %tempbrush.setOrigin("0 0 0");
      return %tempbrush;
   }
      
   
   function GetFacePoint(%BlockStack,%edgenumber,%rx, %ry, %zpos, %sides)
   {
      %radians = 6.283185307; // close enough.
      %angle = (%edgenumber / %sides) * %radians;
      %point = NewVector(mCos(%angle) * %rx, mSin(%angle) * %ry, %zpos);
      return %point;
   }      
      
   function MakeSolid(%edit,%BlockStack, %mins, %maxs, %tx,%ty,%tz, %sides)
   {
      %brush = GetBrush(%edit,%BlockStack.currentbrush);
      %BlockStack.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      %rx = (%maxx - %minx) * 0.5;
      %ry = (%maxy - %miny) * 0.5;
      
      %center = NewVector(%minx + %rx,%miny + %ry,%BlockStack.static.center[2]); 
      
      for (%face = 0; %face < %sides; %face++)
      {
         if(%face == %sides-1)
         {
            %nextface = 0;
         }
         else
         {
            %nextface = %face+1;
         }
         // calc the points for the face.
         %pt1 = VectorAdd(%center,GetFacePoint(%BlockStack,%face,%rx,%ry,%minz,%sides));
         %pt2 = VectorAdd(%center,GetFacePoint(%BlockStack,%nextface,%rx,%ry,%minz,%sides));
         %pt3 = VectorAdd(%center,GetFacePoint(%BlockStack,%face,%rx,%ry,%maxz,%sides));
         
         %brush.setTexturePlanesByPoints(%pt2,%pt1,%pt3, 0.0, 0.0);
         %brush.addPlaneByPoints(%pt2,%pt1,%pt3);
         // Set the texture parms.
         // Add the face to the brush.
      }
      
      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
      
   }
   // MakeCuboid
   function MakeCuboid(%edit,%BlockStack, %mins, %maxs, %tx,%ty,%tz)
   {
      %brush = GetBrush(%edit,%BlockStack.currentbrush);
      %BlockStack.currentbrush++;
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
      
      %brush.setTextureOffset(%BlockStack.static.uoffset, %BlockStack.static.voffset);
      %brush.setTextureScale(%BlockStack.static.uscale, %BlockStack.static.vscale);     
        
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty) SPC -(%tz*2), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("1 0 0 " @ -1.0*(%maxx));
      
      %brush.setTexturePlanesByPoints(-(%tx) SPC %ty SPC -(%tz*2), -(%tx) SPC %ty SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("-1 0 0 " @ 1.0*(%minx));

      %brush.setTexturePlanesByPoints(%tx SPC %ty SPC -(%tz*2), %tx SPC %ty SPC -(%tz), -(%tx) SPC %ty SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 1 0 " @ -1.0*(%maxy));
         
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz*2), -(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), 0.0, 0.0);   
      %brush.addPlane("0 -1 0 " @ 1.0*(%miny));

      // Top face
      %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty*2) SPC %tz, -(%tx) SPC -(%ty) SPC %tz, %tx SPC -(%ty) SPC %tz, 0.0, 0.0);
      %brush.addPlane("0 0 1 " @ -1.0*(%maxz));
      
      // Bottom face
      %brush.setTexturePlanesByPoints(%tx SPC -(%ty*2) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), -(%tx) SPC -(%ty) SPC -(%tz), 0.0, 0.0);
      %brush.addPlane("0 0 -1 " @ 1.0*(%minz));  

      %edit.updateBrush(%brush);
   }
      
      
   function CreateBlockStack(%edit,%BlockStack,%mins,%maxs,%tx,%ty,%tz)
   {
      %minx = getWord(%mins,0);
      %miny = getWord(%mins,1);
      %minz = getWord(%mins,2);
      
      %maxx = getWord(%maxs,0);
      %maxy = getWord(%maxs,1);
      %maxz = getWord(%maxs,2);
   
      %cmins = NewVector(%minx,%miny,%minz);
      %cmaxs = NewVector(%maxx,%maxy,%maxz);
      
      if (%BlockStack.static.rectangular)
        {
        MakeCuboid(%edit,%BlockStack,%cmins,%cmaxs,%tx,%ty,%tz);
        }
      else
        {
        MakeSolid(%edit,%BlockStack,%cmins,%cmaxs,%tx,%ty,%tz,%BlockStack.static.sides);
       }
   }
      
   //*** Build/modify the actual geometry
   function BlockStack_MakeGeometry(%BlockStack, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%BlockStack.active)
         return;

      %BlockStack.currentbrush = 0;
      if (%BlockStack.static.forceclear)
      {
         %edit.clearAllNewBrushes();
         %BlockStack.static.forceclear = 0;
      }
   
      //*** Work on the sizing of the brush, if appropriate
      %center = %BlockStack.static.center[0] SPC %BlockStack.static.center[1] SPC %BlockStack.static.center[2];
      %tx = %BlockStack.static.size[0] * 0.5;
      %ty = %BlockStack.static.size[1] * 0.5;
      %tz = %BlockStack.static.size[2] * 0.5;

      %range = 100-%BlockStack.static.percent;
      %currentsize = 100;
    
      %lastx = %tx;
      %lasty = %ty;
            
      %offx = 0;
      for (%xrow = 0; %xrow < %BlockStack.static.xcount; %xrow++)
      {
         %offy = 0;
         for (%yrow = 0; %yrow < %BlockStack.static.ycount; %yrow++)
         {
            %offz = 0;
            for (%zrow = 0;%zrow < %BlockStack.static.zcount; %zrow++)
            {
               %zforx = %zrow;
               %zfory = %zrow;
               
               if (%BlockStack.static.alternatex)
               {
                 if ((%zforx % 2) == 1)
                 {
                    %zforx = 1;
                 }
                 else
                 {
                    %zforx = 0;
                 }
               }
               if (%BlockStack.static.alternatey)
               {
                 if ((%zfory % 2) == 1)
                 {
                    %zfory = 1;
                 }
                 else
                 {
                    %zfory = 0;
                 }
               }               
               %ox = %offx + %zforx * %BlockStack.static.xoff;
               %oy = %offy + %zfory * %BlockStack.static.yoff;
               %oz = %offz;// + %yrow * %BlockStack.static.zoff;
               %xpos = (%BlockStack.static.center[0]) + %ox + (%xrow * %BlockStack.static.xspace);
               %ypos = (%BlockStack.static.center[1]) + %oy + (%yrow * %BlockStack.static.yspace);
               %zpos = (%BlockStack.static.center[2]) + %oz + (%zrow * %BlockStack.static.zspace);
               
               %mins = NewVector(%xpos-%tx,%ypos-%ty,%zpos-%tz);
               %maxs = NewVector(%xpos+%tx,%ypos+%ty,%zpos+%tz);
               CreateBlockStack(%edit,%BlockStack,%mins,%maxs,%tx,%ty,%tz);
               %offz +=  %BlockStack.static.size[2];
            }
            %offy +=  %BlockStack.static.size[1];
         }
         %offx +=  %BlockStack.static.size[0];
      }
   }
};

tool.register("BlockStack", tool.typeInteractive(), tool.RFLAG_NONE(), "Create BlockStacks" );
tool.setToolProperty("BlockStack", "Icon", "icons/BlockStack");
tool.setToolProperty("BlockStack", "Group", "Create");
