//
// transformsize.cs
//
// Defines the TransformSize tool plug-in to size selected geometry using a bounding box.
//
// The static ScriptObject makes use of the following dynamic fields:
// size[]        - Defines the xyz size of the selection as a 3 element array.  Required as static for the bounding box helper.
// center[]      - Defines the xyz center point of the selection as a 3 element array.  Required as static for the bounding box helper.
// startsize[]   - Defines the starting xyz size of the selection as a 3 element array.
// startcenter[] - Defines the starting xyz center point of the selection as a 3 element array.
//
// The plug-in instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
// active       - Flag to indicate that the tool is active.
// modify       - Flag to indicate that it is safe to modify the geometry
//
// Revision History:
// July 13, 2006			David Wyand		Created script file
// February 15, 2007        David Wyand     Added active flag support
// April 12, 2007           David Wyand     Modified to only work in 'Brushes' selection mode and
//                                          specifically ignore the other modes.  Previously strange
//                                          results would occur if the user changed selection modes
//                                          while the tool was active.
//

package TransformSize
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("TransformSize: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Warn the user that this tool only works in 'Brushes' selection mode but
      //*** allow the tool to continue.
      if(select.getSelectionType() !$= "SelectPrimitives")
      {
         MessageBoxOK("TransformSize", "This tool only works in the 'Brushes' selection mode.");
      }
      
      //*** Set up the static data.  We're really treating this like instance data
      //*** but the standard bounding box routines look for it in the static area.
      TransformSize_InitStatic(%static);
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeDynamic(), true, true);
      ToolBB::setCustomColor(%static, pref.getColorI("GeometryOperationLineColor"));
      ToolBB::rebuildBoundingBox(%plugin, %static);
      
      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeCenter = false;
      %plugin.changeSize = false;
      %plugin.modify = true;
      //%plugin.active = true; <-- Actually set by the following function call
      TransformSize_CheckForSelection(%plugin);
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_HIDEMOUSE();        // We don't need any support buttons.
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("TransformSize: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("TransformSize: MouseDown(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      if(%plugin.active)
      {
         %plugin.modify = true;
         return false;

      } else
      {
         return true;
      }
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("TransformSize: HandleCount(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;
      
      if(%plugin.active)
      {
         //*** Return the standard bounding box handles
         return ToolBB::getHandleCount();

      } else
      {
         return 0;
      }
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("TransformSize: HandleInit(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;
      
      //*** There are always active handles so no need to initialize.  In fact,
      //*** this method should never be called.
      
      return -1;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine which handle
   // has been selected.  In both cases, %info is a ScriptObject that contains the
   // .pos[3] fields that are to be filled in with the requested handle's position.
   // This function returns the priority of the handle, the higher the number the higher
   // the priority.  This is used to determine which handle should be selected when two
   // or more handles overlap on the screen.  If -1 is returned, then the handle is
   // considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("TransformSize: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%plugin, %plugin.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("TransformSize: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%plugin, %plugin.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
      //error("TransformSize: MouseUp(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;
      
      //*** Notify that we have performed our action and are now done
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      //%plugin.update = tool.EDIT_ACCEPT();
      %plugin.update = tool.EDIT_UPDATE();
   }
   
   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("TransformSize: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class and is
   // used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("TransformSize: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;

      if(%plugin.active)
      {
         //*** Draw the standard bounding box
         ToolBB::draw(%plugin, %plugin.static, %draw);
      }

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("TransformSize: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("TransformSize: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      //if(%keep)
      //{
      //   %plugin.update = tool.EDIT_UPDATE();
      //   TransformSize_Reset(%plugin);
      //   tool.refresh();
         
      //} else
      //{
      //   %plugin.update = tool.EDIT_DONOTHING();
      //}
      if(!%keep)
      {
         //*** Restore the objects back to their original position
         TransformSize_Restore(%plugin);
         %plugin.update = tool.EDIT_UPDATE();
         %plugin.changeCenter = true;
         %plugin.changeSize = true;
         tool.refresh();
      }
      
      %plugin.changeCenter = false;
      %plugin.changeSize = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("TransformSize: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** Work on a size or center change.
      if(%plugin.modify && (%plugin.changeCenter || %plugin.changeSize) )
      {
         for(%i=0; %i<3; %i++)
         {
            %newsize[%i] = %plugin.static.size[%i] / %plugin.static.startsize[%i];
            %newcenter[%i] = %plugin.static.center[%i] - %plugin.static.startcenter[%i];
         }
         
         //*** Perform the scale first relative to the starting/buffered scale and center
         %scale = %newsize[0] SPC %newsize[1] SPC %newsize[2];
         %scalecenter = %plugin.static.startcenter[0] SPC %plugin.static.startcenter[1] SPC %plugin.static.startcenter[2];
         %edit.scaleItemsCenterRelBuf(%scale, %scalecenter);
         
         //*** Perform a move relative to where the objects are now to add to what has already
         //*** been done with the scale above as the scale may have already moved some objects.
         %centeroffset = %newcenter[0] SPC %newcenter[1] SPC %newcenter[2];
         %edit.moveItemsRel(%newcenter[0] SPC %newcenter[1] SPC %newcenter[2]);

         %edit.updateSelectionBounds();
      }
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeSize = false;
      %plugin.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("TransformSize: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeSize = true;
            %plugin.changeCenter = true;
            %plugin.modify = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            TransformSize_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeSize = true;
            %plugin.changeCenter = true;
            %plugin.modify = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            %plugin.update = tool.EDIT_REJECT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has modified what items are selected.
         case tool.EVENT_SELECTIONCHANGE():
            TransformSize_Reset(%plugin);
            %plugin.update = tool.EDIT_ACCEPT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeSize = true;
            %plugin.changeCenter = true;
            %plugin.modify = false;
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("TransformSize: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Transform: Size");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("TransformSize: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            if(%plugin.active)
            {
               %value = %plugin.static.center[0] SPC %plugin.static.center[1] SPC %plugin.static.center[2];

            } else
            {
               %value = "";
            }
            return %value;
            
         //*** Handle the 'Size' field
         case 1:
            if(%plugin.active)
            {
               %value = %plugin.static.size[0] SPC %plugin.static.size[1] SPC %plugin.static.size[2];

            } else
            {
               %value = "";
            }
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("TransformSize: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;

      if(!%plugin.active)
      {
         return tool.FUNC_BADVALUE();
      }
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %plugin.static.center[%i] = getWord(%value, %i);
            }
            %plugin.changeCenter = true;
            
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %plugin.static.size[%i] = getWord(%value, %i);
            }
            %plugin.changeSize = true;
      }
      
      //*** First do an update and then an accept to allow for
      //*** the selection widgets to update.
      %plugin.modify = true;
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%plugin, %plugin.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function TransformSize_DefaultValues(%static)
   {
      %bounds = select.getSelectionBounds();
      %static.size[0] = getWord(%bounds,3)-getWord(%bounds,0);
      %static.size[1] = getWord(%bounds,4)-getWord(%bounds,1);
      %static.size[2] = getWord(%bounds,5)-getWord(%bounds,2);
      
      %center = select.getSelectionBoundsCenter();
      for(%i=0; %i<3; %i++)
      {
         %static.center[%i] = getWord(%center, %i);
         
         //*** Also store the starting values
         %static.startsize[%i] = %static.size[%i];
         %static.startcenter[%i] = %static.center[%i];
      }
   }

   //*** Init the static object
   function TransformSize_InitStatic(%static)
   {
      //*** Setup default values
      TransformSize_DefaultValues(%static);
   }
   
   //*** Reset the tube instance to default values
   function TransformSize_Reset(%plugin)
   {
      //*** Setup default values
      TransformSize_DefaultValues(%plugin.static);
      ToolBB::rebuildBoundingBox(%plugin, %plugin.static);

      TransformSize_CheckForSelection(%plugin);
   }
   
   //*** Restore the original values
   function TransformSize_Restore(%plugin)
   {
      for(%i=0; %i<3; %i++)
      {
         %plugin.static.size[%i] = %plugin.static.startsize[%i];
         %plugin.static.center[%i] = %plugin.static.startcenter[%i];
      }
   }

   function TransformSize_CheckForSelection(%plugin)
   {
      //*** Only allow operation in the 'Brushes' selection mode
      if(select.getSelectionType() $= "SelectPrimitives" && (select.count() != 0 || !pref.getBool("ToolOperateSelectionOnly")))
      {
         %plugin.active = true;

      } else
      {
         %plugin.active = false;
      }
   }
};

tool.register("TransformSize", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS() + tool.RFLAG_ALLOWNONMODALSELECTION(), "Size Bounds" );
