//
// transformrotate.cs
//
// Defines the TransformRotate tool plug-in to move geometry.
//
// The instance makes use of the following dynamic fields:
// static    - Points to the static ScriptObject
// dirty     - Flag to indicate that the tool needs to be refreshed on screen
// active    - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update    - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// center    - Defines the "x y z" center point of the rotation
// axis      - Defines the axis on which the rotation is performed 0=x, 1=y, 2=z
// angle     - Defines the angle of the rotation in degrees
// angleBuf  - Stores the angle when the mouse button is first down

// Revision History:
// June 29, 2005		David Wyand		Created script file
// February 16, 2007    David Wyand     Added support for non-modal selection
//

package TransformRotate
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("TransformRotate: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }

      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** This plugin doesn't make use of static data but we still need to
      //*** assign it.
      %plugin.static = %static;

      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.active = true;
      %plugin.update = tool.EDIT_DONOTHING();
      TransformRotate_Reset(%plugin);
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_STANDARDTRANSFORM() + tool.IFLAG_DRAWALLSAME(); // Set up the tool with the standard transform tool GUI
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("TransformRotate: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("TransformRotate: MouseDown(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;

      //*** Determine if the mouse is on a handle.  If so then manipulate that handle.
      //*** This is used with non-modal selection.  If not then work as normal.
      if(!pref.getBool("ToolModalSelection") && tool.isMouseOverHandle(%event) != -1)
      {
         return false;
      }

      //*** We're automatically activated when the mouse goes down
      %plugin.active = true;

      //*** Set the center of rotation to where the mouse was pressed and depending on
      //*** the current Action Center.
      if(pref.getBool("ToolModalSelection"))
      {
         if(!TransformRotate_UpdateActionCenter(%plugin))
         {
            //*** Default to the tool's chosen center
            %plugin.center = %event.pos;
         }
      }
      
      //*** Set the axis of rotation based on the view that was clicked in
      %prevaxis = %plugin.axis;
      if(%event.view >= 0 && %event.view <= 2)
      {
         //*** Orthographic views
         %plugin.axis = %event.view;
         
      } else
      {
         //*** Handle the perspective view based on the build axis that
         //*** is perpendicular to the work plane
         %plugin.axis = %event.gridBuildAxis;
      }
      
      if(%prevaxis != %plugin.axis)
      {
         %plugin.active = false; //*** Don't allow the change in axis prematurely modify geometry
         TransformRotate_Reset(%plugin, true);
         %plugin.update = tool.EDIT_ACCEPT();
         %plugin.dirty = tool.DIRTY_APPEARANCE();

      } else
      {
         //*** Notify that we need to redraw the plugin as well as geometry
         %plugin.dirty = tool.DIRTY_APPEARANCE();
         %plugin.update = tool.EDIT_UPDATE();
      }
      
      %plugin.angleBuf = %plugin.angle;
      %plugin.endPos = %event.endPos;
      
      //*** We want to process the mouse drags directly, so return true.
      return true;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("TransformRotate: MouseDrag(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;

      %plugin.active = true; //*** Needs to be reset here in case we had a mouse down on a different axis

      //*** Calculate the rotation amout based on the number of pixels the mouse
      //*** has been dragged.
      %plugin.angle = getWord(%event.screenAbsDelta, 0);
      %plugin.angle *= pref.getFloat("ToolRotationAmount");
      
      //*** Account for the user dragging down the positive or negative end of
      //*** the displayed axis, for orthographic displays
      if(%event.view >= 0 && %event.view <= 2 && %event.viewDir)
      {
         //*** The view is down the -ve axis
         %plugin.angle *= -1.0;
      }
      
      //*** If the user has constrained movement, then we'll rotate in fixed
      //*** 15 degree increments.
      if(%event.FLAG_CONSTRAINED())
      {
         %mod = mAbs(%plugin.static.angle) % 15;
         if(%plugin.angle >= 0)
         {
            %plugin.angle -= %mod;
            
         } else
         {
            %plugin.angle += %mod;
         }
      }

      %plugin.angle += %plugin.angleBuf;
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("TransformRotate: MouseUp(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;
      
      //*** Notify that we have performed our action and are now done
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      //%plugin.update = tool.EDIT_ACCEPT();
      %plugin.update = tool.EDIT_UPDATE();
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("TransformRotate: HandleCount(" @ %inst @ "," @ %event @ ")");

      //*** The handle count depends on if non-modal selection is set.  If it is and
      //*** the action center is tool or selection, then we have a movable handle.
      //*** Otherwise we act just like before under modal selection.
      %selmode = pref.getBool("ToolModalSelection");
      if(!%selmode)
      {
         %ac = pref.getInt("ToolActionCenter");
         //*** We have handles if the Action Center is Mouse/Tool or Selection Center
         if(%ac == 0 || %ac == 1)
         {
            return 1;
         }
      }

      //*** We don't have any handles
      return 0;
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("TransformRotate: HandleInit(" @ %inst @ "," @ %event @ ")");

      //*** We don't have any handles
      return -1;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("TransformRotate: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");
      
      %plugin = %inst.instance;

      //*** This should only be called when operating in non-modal selection mode.  In this
      //*** case the handle is the tool's center
      if(%hindex == 0)
      {
         for(%i=0; %i<3; %i++)
            %info.pos[%i] = getWord(%plugin.center, %i);

         return %hindex;
      }

      //*** We don't have any handles
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("TransformRotate: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");
      
      %plugin = %inst.instance;

      //*** This should only be called when operating in non-modal selection mode.  In this
      //*** case the handle is the tool's center
      if(%hindex == 0)
      {
         %plugin.center = %event.pos;
      
         //*** Notify that we need to redraw the plugin as well as geometry
         %plugin.dirty = tool.DIRTY_APPEARANCE();
         //%plugin.update = tool.EDIT_UPDATE();

         return %hindex;
      }

      //*** We don't have any handles
      return -1;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
      //error("TransformRotate: HandleDone(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");
      
      %plugin = %inst.instance;

      if(%hindex == 0)
      {
         %plugin.active = false; //*** Don't allow the change in center handle prematurely modify geometry
         TransformRotate_Reset(%plugin, true);
         %plugin.update = tool.EDIT_ACCEPT();
         %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("TransformRotate: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("TransformRotate: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** Draw the center of rotation 'handle'.  If we're working under non-modal
      //*** selection and nothing is selected then don't draw the handle
      if(pref.getBool("ToolModalSelection") || select.count() > 0)
      {
         %type = %draw.HANDLE_ACTION_X();
         if(%plugin.axis == 1)
         {
            %type = %draw.HANDLE_ACTION_Y();
         
         } else if(%plugin.axis == 2)
         {
            %type = %draw.HANDLE_ACTION_Z();
         }
         %draw.handle(%type, "255 255 255", %plugin.center);
      }
      
      //*** We're done drawing
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("TransformRotate: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("TransformRotate: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      if(!%keep)
      {
         //*** Restore the objects back to their original position
         TransformRotate_Reset(%plugin);
         %plugin.update = tool.EDIT_UPDATE();
         tool.refresh();
      }

   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("TransformRotate: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;

      //*** Rotate the geometry if we're active.
      if(%plugin.active)
      {
         //*** Change our tool's undo label based on the axis
         switch(%plugin.axis)
         {
            case 0:
               tool.setUndoEntryLabel("Rotate Single Axis: X");

            case 1:
               tool.setUndoEntryLabel("Rotate Single Axis: Y");

            case 2:
               tool.setUndoEntryLabel("Rotate Single Axis: Z");
         }

         //*** Rotate the geometry by building up the rotation offset
         %offset = "0.0 0.0 0.0";
         %offset = setWord(%offset, %plugin.axis, %plugin.angle);
         %ac = pref.getInt("ToolActionCenter");
         if(%ac == 2)
         {
            //*** Rotate about the item's group center
            %edit.rotateItemsGroupRelBuf(%offset, %plugin.center, %plugin.angle);
         
         } else if(%ac == 3)
         {
            //*** Rotate about the item's origin
            %edit.rotateItemsOriginRelBuf(%offset, %plugin.center, %plugin.angle);
         
         } else
         {
            %edit.rotateItemsCenterRelBuf(%offset, %plugin.center, %plugin.endPos, %plugin.angle);
         }

         %edit.updateSelectionBounds();
      }
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("TransformRotate: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: the displayed offset).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_DONOTHING(); // Don't do anything when activated.  We only perform the actual change from EndEditAction()
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            TransformRotate_Reset(%plugin);
            %plugin.update = tool.EDIT_DONOTHING(); // Don't do anything when activated.  We only perform the actual change from EndEditAction()
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            %plugin.update = tool.EDIT_REJECT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();

         //*** The user has changed the current Action Center.
         case tool.EVENT_ACTIONCENTERCHANGE():
            %plugin.active = false; //*** Don't allow the change in action center prematurely modify geometry
            TransformRotate_Reset(%plugin);
            %plugin.update = tool.EDIT_ACCEPT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has modified what items are selected under non-modal selection.
         case tool.EVENT_SELECTIONCHANGE():
            %plugin.active = false; //*** Don't allow the change in selection prematurely modify geometry
            TransformRotate_Reset(%plugin);
            %plugin.update = tool.EDIT_ACCEPT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("TransformRotate: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Transform: Rotate");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Axis"     ,"axis3");
      %form.addField( 2, "Angle"    ,"angle");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("TransformRotate: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %plugin.center;
            return %value;

         //*** Handle the 'Axis' field
         case 1:
            %value = %plugin.axis;
            return %value;

         //*** Handle the 'Angle' field
         case 2:
            %value = %plugin.angle;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("TransformRotate: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      %axischange = false;
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %plugin.center = %value;

         //*** Handle the 'Axis' field
         case 1:
            %plugin.axis = %value;
            %axischange = true;

         //*** Handle the 'Angle' field
         case 2:
            %plugin.angle = %value;
      }
      
      if(%axischange)
      {
         %plugin.active = false; //*** Don't allow the change in axis prematurely modify geometry
         TransformRotate_Reset(%plugin, true);
         %plugin.update = tool.EDIT_ACCEPT();
         %plugin.dirty = tool.DIRTY_APPEARANCE();

      } else
      {
         //*** Indicate that everything needs to be redrawn
         %plugin.active = true;
         %plugin.update = tool.EDIT_UPDATE();
         %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the transform instance to default values
   function TransformRotate_Reset(%plugin, %angleonly)
   {
      //*** Setup default values
      if(!%angleonly)
      {
         %plugin.center = "0 0 0"; // x y z
         %plugin.axis = 2; // z
         TransformRotate_UpdateActionCenter(%plugin);
      }

      %plugin.angle = 0; // degrees
   }
   
   //*** Update the center of rotation based on the action center.
   //*** Returns true if the center was changed.
   function TransformRotate_UpdateActionCenter(%plugin)
   {
      %changed = false;
      %ac = pref.getInt("ToolActionCenter");
      if(%ac == 0 && !pref.getBool("ToolModalSelection"))
      {
         //*** Use the selection center under non-modal selection.
         %plugin.center = select.getSelectionCenter();
         %changed = true;

      } else if(%ac == 1)
      {
         //*** We're using the selection center, so force it
         %plugin.center = select.getSelectionCenter();
         %changed = true;
               
      } else if(%ac == 4)
      {
         //*** We're using the world origin, so force it
         %plugin.center = "0 0 0";
         %changed = true;
               
      } else if(%ac == 5)
      {
         //*** We're using the custom origin, so force it
         %scene = scene.getCurrent();
         %plugin.center = %scene.getDetailLevelActionCenter(%scene.getCurrentDetailLevel());
         %changed = true;
      }
      
      return %changed;
   }
   
};

tool.register("TransformRotate", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS() + tool.RFLAG_ALLOWNONMODALSELECTION(), "Rotate Single Axis" );
