//
// transformpivotpoint.cs
//
// Defines the TransformPivotPoint tool plug-in to move a brush's pivot point.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
//
// The instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// map          - The map the tool is operating on
// numSel       - The number of currently selected brushes
// brushID      - The brush ID the tool is operating on
// pivot        - Defines the "x y z" position of the brush's pivot point
// rotation     - Defines the "x y z angle" of the brush's original pivot point rotation
// original     - Defines the whole pivot point for the brush prior to the tool being used
// changeOffset - Flag to indicate that the tool's offset has changed

// Revision History:
// January 26, 2007		David Wyand		Created script file
//

package TransformPivotPoint
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("TransformPivotPoint: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Only brush selection is supported
      if(select.getSelectionType() !$= "SelectPrimitives")
      {
         tool.activateErrorMsg = "This tool only works in the 'Brushes' selection mode.";
         return tool.FUNC_BADSELECTMODE();
      }

      %scene = scene.getCurrent();
      %map = scene.getCurrentMap();
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();

      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.changeOffset = false;

      %plugin.map = %map;
      TransformPivotPoint_Setup(%plugin);

      //*** Notify the user of the tool's limitations
      TransformPivotPoint_Check(%plugin);
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_RESETBUTTON() + tool.IFLAG_MAKEBUTTON() + tool.IFLAG_DRAWALLSAME();
      %inst.flagsApply = tool.AFLAG_NONE();

      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("TransformPivotPoint: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("TransformPivotPoint: MouseDown(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;
      
      //*** If zero or two or more brushes are selected then indicate that the mouse
      //*** controls the tool.  This prevents any handle interaction.
      if(%plugin.numSel != 1)
      {
         return true;

      } else
      {
         //*** Make use of handles.
         return false;
      }
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("TransformPivotPoint: HandleCount(" @ %inst @ "," @ %event @ ")");

      //*** The single pivot point handle
      return 1;
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("TransformPivotPoint: HandleInit(" @ %inst @ "," @ %event @ ")");
      
      //*** There are always active handles so no need to initialize.  In fact,
      //*** this method should never be called.
      
      return -1;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("TransformPivotPoint: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      if(%hindex == 0)
      {
         %info.pos[0] = getWord(%plugin.pivot, 0);
         %info.pos[1] = getWord(%plugin.pivot, 1);
         %info.pos[2] = getWord(%plugin.pivot, 2);
      
         //*** Return the handle's priority
         return %hindex; // set the handle's priority the same as its index
      }
      
      //*** The given handle index is not one of ours
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("TransformPivotPoint: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;
      
      //*** Move the appropriate handle.
      if(%hindex == 0) // Center handle
      {
         //*** Store the center in case we're constrained
         %center = %plugin.pivot;

         //*** Calculate the position based on our view.  We'd prefer to not
         //*** modify coordinates that the view itself doesn't work with.
         %plugin.pivot = %event.getMousePointOnWorkplaneThroughPoint(%plugin.pivot);
         if(%event.gridBuildAxis >= 0 && %event.gridBuildAxis <= 2)
         {
            //*** Substitute the old value for the axis the view doesn't modify
            %plugin.pivot = setWord(%plugin.pivot, %event.gridBuildAxis, getWord(%center, %event.gridBuildAxis));
         }

         //*** Adjust for constrained movement
         if(%event.FLAG_CONSTRAINED())
         {
            if(%event.FLAG_CONSTRAINEDX())
            {
               %plugin.pivot = getWord(%plugin.pivot, 0) SPC getWord(%center, 1) SPC getWord(%center, 2);
            
            } else if(%event.FLAG_CONSTRAINEDY())
            {
               %plugin.pivot = getWord(%plugin.center, 0) SPC getWord(%plugin.pivot, 1) SPC getWord(%center, 2);
            
            } else if(%event.FLAG_CONSTRAINEDZ())
            {
               %plugin.pivot = getWord(%plugin.center, 0) SPC getWord(%center, 1) SPC getWord(%plugin.pivot, 2);
            }
         }
         
         //*** Record that the center is being changed
         %plugin.changeOffset = true;
      }
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      return %hindex;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("TransformPivotPoint: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty;
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("TransformPivotPoint: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;

      //*** Draw the pivot point handle if we have one selected brush
      if(%plugin.numSel == 1)
      {
         %draw.handle(%draw.HANDLE_PLUS(), "255 255 255", %plugin.pivot);
      }

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("TransformPivotPoint: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      error("TransformPivotPoint: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      if(!%keep)
      {
         //*** Restore the back to their original position
         TransformPivotPoint_Reset(%plugin);
         %plugin.update = tool.EDIT_UPDATE();
         tool.refresh();

      } else
      {
         TransformPivotPoint_Setup(%plugin);
      }

      %plugin.changeOffset = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      error("TransformPivotPoint: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** Set the brush's pivot point
      %plugin.map.setBrushWSPivot(%plugin.brushID, %plugin.pivot SPC %plugin.rotation);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeOffset = false;

      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      error("TransformPivotPoint: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: the displayed offset).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeOffset = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            TransformPivotPoint_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeOffset = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            %plugin.update = tool.EDIT_REJECT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has modified what items are selected under non-modal selection.
         case tool.EVENT_SELECTIONCHANGE():
            %plugin.active = false; //*** Don't allow the change in selection prematurely modify geometry
            TransformPivotPoint_Setup(%plugin);
            TransformPivotPoint_Check(%plugin);
            %plugin.update = tool.EDIT_ACCEPT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("TransformPivotPoint: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Transform: Pivot Point");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Position"   ,"distance3");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("TransformPivotPoint: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Offset' field
         case 0:
            if(%plugin.numSel == 1)
            {
               %value = %plugin.pivot;

            } else
            {
               %value = "";
            }
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("TransformPivotPoint: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Offset' field
         case 0:
            if(%plugin.numSel == 1)
            {
               %plugin.pivot = %value;
            }
      }
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Check if the tool's limitations have been met
   function TransformPivotPoint_Check(%plugin)
   {
      if(%plugin.numSel > 1)
      {
         MessageBoxOK("MOVE BRUSH PIVOT POINT", "This tools only operates with a single selected brush.");
      }
   }
   
   //*** Set up the values from the selected brush
   function TransformPivotPoint_Setup(%plugin)
   {
      %selbrushes = %plugin.map.getSelectedBrushes();
      %plugin.numSel = getWordCount(%selbrushes);
      %plugin.brushID = getWord(%selbrushes, 0);
      %plugin.original = %plugin.map.getBrushWSPivot(%plugin.brushID);
      %plugin.rotation = getWords(%plugin.original, 3);
      %plugin.pivot = getWords(%plugin.original, 0, 2);
   }
   
   //*** Reset the transform instance to default values
   function TransformPivotPoint_Reset(%plugin)
   {
      //*** Reset the brush's pivot point
      %plugin.map.setBrushWSPivot(%plugin.brushID, %plugin.original);
      %plugin.rotation = getWords(%plugin.original, 3);
      %plugin.pivot = getWords(%plugin.original, 0, 2);

      //*** Update the Axis Gizmo
      select.__updateSelectionAxisGizmo(scene.getCurrent(), scene.getCurrentMap());
   }
   
};

tool.register("TransformPivotPoint", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS() + tool.RFLAG_ALLOWNONMODALSELECTION(), "Move Brush Pivot Point" );
