// Script functions to build and work with a standard bounding box for tools.
//
// To use these helper functions with your tool plug-in, some conventions will need
// to be followed:
//
// - Your plug-in's static data class needs to contain two Torque Script arrays to
//   store the bounding box's center and size.  Here's an example of initializing
//   this structure with a center point at the origin and a size of 1 along all axis
//   (and where %static points to your static data class):
//
//      %static.center[0] = 0.0; // x
//      %static.center[1] = 0.0; // y
//      %static.center[2] = 0.0; // z
//      
//      %static.size[0] = 1.0; // x
//      %static.size[1] = 1.0; // y
//      %static.size[2] = 1.0; // z
//
// - These helper functions will add a Torque script array named boundingboxtype[0-2] to your
//   static data class that stores the type of bounding box that will be created.  This
//   determines how the bounding box behaves when the user drags a handle.  The bounding box
//   type is set through the ToolBB::setType() function, and may be changed at any time -- it
//   only affects the response to the dragging of handles and not how the bounding box is drawn.
//   If a ToolBB::setType() is never called, then the bounding box defaults to typeBox().
//
// - These helper functions will add a boolean named 'dynamicboundingbox' to your static data
//   class that will be set to true if the bounding box type has been set to be dynamic.  The
//   type is set through the ToolBB::setType() function.  Also during the initial drag out of
//   the bounding box a boolean named 'latchsphere' may be added to your static data class.
//   It indicates if a modifier was held while the mouse button was pressed to make a dyanmic
//   type bounding box behave like a sphere type.
//
// - These helper functions will add a two-dimensional Torque script array named
//   cornerpos[1-8,0-2] to your plug-in's instance object.  These store the corners
//   of the bounding box used when drawing.
//
// - 8 sizing handles (indexed 0-7) are used by the bounding box.  Index 0 represents
//   the handle in the center to reposition the bounding box.  Indices 1 and 2 are the
//   -x and +x axis handles.  Indices 3 and 4 are the -y and +y axis handles.  Indices
//   5 and 6 are the -z and +z axis handles.  Index 7 is a special handle that is only
//   used when the user is drawing out the bounding box for the first time in
//   HandleInit().  The ToolBB::getHandleCount() and ToolBB::getLastHandleIndex()
//   functions may be used to retrieve information about the handles.  If your plug-in
//   requires additional handles, they may start at index 8 (ToolBB::getLastHandleIndex() + 1).
//
// - These helper functions will add a two-dimensional Torque script array named
//   handlepos[0-7,0-2] to your plug-in's instance object.  These store the positions
//   of the handles used by the bounding box.  You may freely extend this array to store
//   additional handle positions that are used by your plug-in.
//
// - These helper functions will add a Torque script array named startpos[0-2] that stores
//   the initial mouse down position during a HandleInit() call.  This allows the bounding
//   box to be dragged out from opposite corners.
//
// - These helper functions will add a string called 'boundingBoxCustomColor' to your static
//   data class.  It is used to store a custom color for drawing the bounding box in as
//   defined by ToolBB::setCustomColor().
//
// - Your plug-in's HandleCount() function should return ToolBB::getHandleCount() handles
//   once the bounding box has been initially created.  You may safely add additional
//   handles to this count if your plug-in requires them.
//
// - The bounding box helper functions make use of two bool variables in your instance:
//   changeSize and changeCenter.  These are used to indicate that the bounding box's size
//   and center point have been modified.  Your plug-in will generally make use of these
//   variables (and set them itself as appropriate) to determine what parts of the plug-in
//   will need to be rebuilt ie: geometry and tool appearance.  You'll want to initialize
//   these two variables to false within the plug-in's Activate() function.
//

//**********************************************************************************
//*** Bounding Box type return functions used in ToolBB::setType().
//**********************************************************************************

//*** The far edge to the dragged handle is fixed and the bounding box's centre is changed
//*** to the middle of the edges.
function ToolBB::typeBox()
{
   return 0;
}

//*** The far edge to the dragged handle along the X-axis is fixed and the bounding box's
//*** centre is changed to the middle of the two edges.  Dragging handles along the Y- and
//*** Z-axis keeps the centre fixed and the opposite handle is moved to match the dragged
//*** handle.
function ToolBB::typeCylinderX()
{
   return 1;
}

//*** The far edge to the dragged handle along the Y-axis is fixed and the bounding box's
//*** centre is changed to the middle of the two edges.  Dragging handles along the X- and
//*** Z-axis keeps the centre fixed and the opposite handle is moved to match the dragged
//*** handle.
function ToolBB::typeCylinderY()
{
   return 2;
}

//*** The far edge to the dragged handle along the Z-axis is fixed and the bounding box's
//*** centre is changed to the middle of the two edges.  Dragging handles along the X- and
//*** Y-axis keeps the centre fixed and the opposite handle is moved to match the dragged
//*** handle.
function ToolBB::typeCylinderZ()
{
   return 3;
}

//*** The centre is fixed and the handle on the opposite side of the dragged handle is moved
//*** to keep the bounding box symetrical.
function ToolBB::typeSphere()
{
   return 4;
}

//*** Without modifiers it acts like a box.  With ALT down it acts like a sphere.  With SHIFT
//*** down on a corner handle it is a uniform scale (inc. with ALT) in 2D.  During initial
//*** drag out, SHIFT gives uniform scale in 3D. With the center/move handle, SHIFT will
//*** constrain to an axis as with other bounding box types.
function ToolBB::typeDynamic()
{
   return 5;
}

//**********************************************************************************
//*** Bounding Box functions used by your plug-in
//**********************************************************************************

//*** This function sets how the bounding box will behave when the handles are dragged.
function ToolBB::setType(%static, %type, %corners, %allowNegative)
{
   //*** Set all axis to a box-like resize
   %static.boundingboxtype[0] = false;
   %static.boundingboxtype[1] = false;
   %static.boundingboxtype[2] = false;
   
   //*** Other flags and values
   %static.ToolBBCornerHandles = false;     // Corner handles?
   %static.ToolBBAllowNegativeSize = false; // Allow the user to make the bounding box have a negative size?

   if( %corners )
   {
      %static.ToolBBCornerHandles = true;
   }

   if( %allowNegative )
   {
      %static.ToolBBAllowNegativeSize = true;
   }
      
   //*** Now set each axis to have a symetrical resize based on the type
   if(%type == 1 || %type == 4)
   {
      %static.boundingboxtype[1] = true;
      %static.boundingboxtype[2] = true;
   }
   if(%type == 2 || %type == 4)
   {
      %static.boundingboxtype[0] = true;
      %static.boundingboxtype[2] = true;
   }
   if(%type == 3 || %type == 4)
   {
      %static.boundingboxtype[0] = true;
      %static.boundingboxtype[1] = true;
   }

   if(%type == 5)
   {
      %static.dynamicboundingbox = true;

   } else
   {
      %static.dynamicboundingbox = false;
   }
}

//*** This function sets a custom color for the bounding box outline.
function ToolBB::setCustomColor(%static, %color)
{
   %static.boundingBoxCustomColor = %color;
}

//*** This function will rebuild the bounding box after the plug-in has modified
//*** its sizing and center arrays.
function ToolBB::rebuildBoundingBox(%inst, %static)
{
   ToolBB::__BuildHandles(%inst, %static, -1);
}

//*** This function will rebuild the bounding box corners.  Normally this is called
//*** automatically but this information can often be useful to the plug-in so it
//*** is exposed here.  This fills in the %inst.cornerpos[1-8,0-2] Torque script array.
function ToolBB::rebuildBoundingBoxCorners(%inst, %static)
{
   ToolBB::__BuildCorners(%inst, %static);
}

//*** This function returns the number of handles used by the bounding box.
function ToolBB::getHandleCount()
{
   return 16;
}

//*** This function returns the last handle index that is used by the bounding box.
function ToolBB::getLastHandleIndex()
{
   return 15;
}

//*** This function is called within your plug-in's HandleInit(), passing along your
//*** plug-in's instance and static objects, and the event that triggered this function
//*** call.  It returns the index of the handle that the user will control as they
//*** continue to drag the mouse.
function ToolBB::initHandles(%inst, %static, %event)
{
   //*** Set the bounding box's center and size based on where the user clicked.  Also
   //*** store this start position for this first time that the user is dragging out
   //*** the bounding box by the corner.
   for(%i=0; %i<3; %i++)
   {
      %static.center[%i] = getWord(%event.pos, %i);
      %static.size[%i] = 0.0;
      %inst.startpos[%i] = %static.center[%i];
   }
      
   //*** Build the handles
   ToolBB::rebuildBoundingBox(%inst, %static);
      
   //*** Record that the size is being changed
   %inst.changeSize = true;
   %inst.changeCenter = true;

   //*** Is the user enabling the sphere-type mode on a dynamic bounding box for
   //*** the initial drag out?      
   if(%static.dynamicboundingbox == true && PlatformIsALTDown())
   {
      %static.latchsphere = true;

   } else
   {
      %static.latchsphere = false;
   }

   //*** Return the corner sizing handle index as that is what the user will move as the
   //*** mouse moves for this inital drag out.  This handle is not used after the initial draw.
   return 0;
}

//*** This function is called within your plug-in's Handle(), passing along the same
//*** parameters that the Tool Manager sent your plug-in.  This function returns the
//*** priority of the %hindex handle.  If your plug-in makes use of its own handles
//*** in addition to those of the bounding box, only call this function if %hindex is
//*** less than or equal to ToolBB::getLastHandleIndex().
function ToolBB::getHandle(%inst, %static, %event, %hindex, %info)
{
   //*** Handle the ortho viewing cases
   if(%event.view == 0 && (%hindex == 1 || %hindex == 2))
      return -1;
   if(%event.view == 1 && (%hindex == 3 || %hindex == 4))
      return -1;
   if(%event.view == 2 && (%hindex == 5 || %hindex == 6))
      return -1;
      
   //*** Handle the perspective view and corner handles
   //if(%event.view == 3 && %hindex >= 7 && %hindex <= 14)
   //   return -1;
         
   //*** Tap into a Tool Manager back door to determine if the initial sizing
   //*** handle is the one being used.  If not, then disable this handle if it
   //*** is being asked about.
   if(%hindex == 0 && tool.__curHandle != 0)
   {
      return -1;
   }
   
   //*** Work with the index for the initial drag handle (index 0) and the six middle handles (1-6),
   //*** or the center handle (index 15).
   if( (%hindex >= 0 && %hindex <= 6) || %hindex == 15 )
   {
      %info.pos[0] = %inst.handlepos[%hindex,0];
      %info.pos[1] = %inst.handlepos[%hindex,1];
      %info.pos[2] = %inst.handlepos[%hindex,2];

      //*** If the size of the bounding box is zero along any side then
      //*** lower the priority of the center handle to allow a sizing handle
      //*** to gain control.
      if(%hindex == 15 && (%static.size[0] == 0 || %static.size[1] == 0 || %static.size[2] == 0) )
      {
         return 0;
      }
      
      //*** Return the handle's priority
      return %hindex; // set the handle's priority the same as its index
   }

   //*** If corner handles are active then work with those eight handles (7-14).
   if(%static.ToolBBCornerHandles && %hindex >= 7 && %hindex <= 14)
   {
      %info.pos[0] = %inst.cornerpos[(%hindex-6),0];
      %info.pos[1] = %inst.cornerpos[(%hindex-6),1];
      %info.pos[2] = %inst.cornerpos[(%hindex-6),2];
      
      //*** Return the handle's priority
      return %hindex; // set the handle's priority the same as its index
   }
      
   //*** The given handle index is not one of ours
   return -1;
}

//*** This function is called in your plug-in's HandleMoved(), passing along the
//*** event that triggered the function call and the index of the handle that moved.
//*** It returns the index of the handle that the user will continue to move, which
//*** may not be the same as the handle index passed in.
function ToolBB::moveHandle(%inst, %static, %event, %hindex)
{
   //*** When rebuilding the handles, is there an axis we should not recalculate
   //*** because that is the axis the user is moving?
   %ignoreAxis = -1;
      
   //*** Normally we return the same handle index that is currently being manipulated
   //*** by the user.
   %returnHandle = %hindex;
      
   //*** Change the bounding box based on which handle has been moved
   if(%hindex == 0) // Initial bounding box drawing handle
   {
      %pos0 = getWord(%event.pos, 0);
      %pos1 = getWord(%event.pos, 1);
      %pos2 = getWord(%event.pos, 2);

      //*** Check if under a dynamic-type bounding box if the ALT key's state has changed
      if(%static.dynamicboundingbox == true && PlatformIsALTDown())
      {
         if(!%static.latchsphere)
         {
            //*** Switch to a sphere-like sizing
            for(%i=0; %i<3; %i++)
            {
               %static.center[%i] = %inst.startpos[%i];
            }
         }
         %static.latchsphere = true;

      } else
      {
         //*** Switch to box-like sizing
         %static.latchsphere = false;
      }

      //*** Depending on the type, the centre may move or remain fixed.
      if( !%static.latchsphere &&
          (%static.boundingboxtype[0] == false && %static.boundingboxtype[1] == false && %static.boundingboxtype[2] == false) ||
          (%static.boundingboxtype[0] == true && %event.view != 0) &&
          (%static.boundingboxtype[1] == true && %event.view != 1) &&
          (%static.boundingboxtype[2] == true && %event.view != 2) )
      {
         %static.center[0] = (%pos0 - %inst.startpos[0]) / 2.0 + %inst.startpos[0];
         %static.center[1] = (%pos1 - %inst.startpos[1]) / 2.0 + %inst.startpos[1];
         %static.center[2] = (%pos2 - %inst.startpos[2]) / 2.0 + %inst.startpos[2];
      }

      %static.size[0] = 2.0 * mAbs(%pos0 - %static.center[0]);
      %static.size[1] = 2.0 * mAbs(%pos1 - %static.center[1]);
      %static.size[2] = 2.0 * mAbs(%pos2 - %static.center[2]);

      //*** If this initial handle is dragged out in an orthographic, 2D view, then
      //*** we need to size the 3rd axis here.  We'll just make it the same size as the
      //*** smaller of the other two axis.  Some calculations may look cumbersome, but
      //*** it is to keep the number of multiply and divides to a minimum to maintain
      //*** floating point stability.
      if(%event.view == 0 || (%event.view == 3 && %event.gridBuildAxis == 0) )
      {
         if(PlatformIsSHIFTDown())
         {
            if(%static.size[1] > %static.size[2])
            {
               %static.size[0] = %static.size[1];
               %static.size[2] = %static.size[1];
               if(%static.boundingboxtype[0] == false && !%static.latchsphere)
               {
                  %dy = mAbs(%pos1 - %static.center[1]);
                  %static.center[0] = %inst.startpos[0] + %dy * %event.gridBuildDir;
                  if((%pos2 - %static.center[2]) >= 0)
                  {
                     %static.center[2] = %inst.startpos[2] + %dy;

                  } else
                  {
                     %static.center[2] = %inst.startpos[2] - %dy;
                  }
               }

            } else
            {
               %static.size[0] = %static.size[2];
               %static.size[1] = %static.size[2];
               if(%static.boundingboxtype[0] == false && !%static.latchsphere)
               {
                  %dz = mAbs(%pos2 - %static.center[2]);
                  %static.center[0] = %inst.startpos[0] + %dz * %event.gridBuildDir;
                  if((%pos1 - %static.center[1]) >= 0)
                  {
                     %static.center[1] = %inst.startpos[1] + %dz;

                  } else
                  {
                     %static.center[1] = %inst.startpos[1] - %dz;
                  }
               }
            }

         } else
         {
            %static.size[0] = %static.size[1] < %static.size[2] ? %static.size[1] : %static.size[2];
            if(%static.boundingboxtype[0] == false && !%static.latchsphere)
            {
               if(%static.size[1] < %static.size[2])
               {
                  %static.center[0] = %inst.startpos[0] + mAbs(%pos1 - %static.center[1]) * %event.gridBuildDir;
               
               } else
               {
                  %static.center[0] = %inst.startpos[0] + mAbs(%pos2 - %static.center[2]) * %event.gridBuildDir;
               }
            }
         }
         
      } else if(%event.view == 1 || (%event.view == 3 && %event.gridBuildAxis == 1) )
      {
         if(PlatformIsSHIFTDown())
         {
            if(%static.size[0] > %static.size[2])
            {
               %static.size[1] = %static.size[0];
               %static.size[2] = %static.size[0];
               if(%static.boundingboxtype[1] == false && !%static.latchsphere)
               {
                  %dx = mAbs(%pos0 - %static.center[0]);
                  %static.center[1] = %inst.startpos[1] + %dx * %event.gridBuildDir;
                  if((%pos2 - %static.center[2]) >= 0)
                  {
                     %static.center[2] = %inst.startpos[2] + %dx;

                  } else
                  {
                     %static.center[2] = %inst.startpos[2] - %dx;
                  }
               }

            } else
            {
               %static.size[0] = %static.size[2];
               %static.size[1] = %static.size[2];
               if(%static.boundingboxtype[1] == false && !%static.latchsphere)
               {
                  %dz = mAbs(%pos2 - %static.center[2]);
                  %static.center[1] = %inst.startpos[1] + %dz * %event.gridBuildDir;
                  if((%pos0 - %static.center[0]) >= 0)
                  {
                     %static.center[0] = %inst.startpos[0] + %dz;

                  } else
                  {
                     %static.center[0] = %inst.startpos[0] - %dz;
                  }
               }
            }

         } else
         {
            %static.size[1] = %static.size[0] < %static.size[2] ? %static.size[0] : %static.size[2];
            if(%static.boundingboxtype[1] == false && !%static.latchsphere)
            {
               if(%static.size[0] < %static.size[2])
               {
                  %static.center[1] = %inst.startpos[1] + mAbs(getWord(%event.pos, 0) - %static.center[0]) * %event.gridBuildDir;

               } else
               {
                  %static.center[1] = %inst.startpos[1] + mAbs(getWord(%event.pos, 2) - %static.center[2]) * %event.gridBuildDir;
               }
            }
         }
         
      } else if(%event.view == 2 || (%event.view == 3 && %event.gridBuildAxis == 2) )
      {
         if(PlatformIsSHIFTDown())
         {
            if(%static.size[0] > %static.size[1])
            {
               %static.size[1] = %static.size[0];
               %static.size[2] = %static.size[0];
               if(%static.boundingboxtype[2] == false && !%static.latchsphere)
               {
                  %dx = mAbs(%pos0 - %static.center[0]);
                  %static.center[2] = %inst.startpos[2] + %dx * %event.gridBuildDir;
                  if((%pos1 - %static.center[1]) >= 0)
                  {
                     %static.center[1] = %inst.startpos[1] + %dx;

                  } else
                  {
                     %static.center[1] = %inst.startpos[1] - %dx;
                  }
               }

            } else
            {
               %static.size[0] = %static.size[1];
               %static.size[2] = %static.size[1];
               if(%static.boundingboxtype[2] == false && !%static.latchsphere)
               {
                  %dy = mAbs(%pos1 - %static.center[1]);
                  %static.center[2] = %inst.startpos[2] + %dz * %event.gridBuildDir;
                  if((%pos0 - %static.center[0]) >= 0)
                  {
                     %static.center[0] = %inst.startpos[0] + %dy;

                  } else
                  {
                     %static.center[0] = %inst.startpos[0] - %dy;
                  }
               }
            }

         } else
         {
            %static.size[2] = %static.size[0] < %static.size[1] ? %static.size[0] : %static.size[1];
            if(%static.boundingboxtype[2] == false && !%static.latchsphere)
            {
               if(%static.size[0] < %static.size[1])
               {
                  %static.center[2] = %inst.startpos[2] + mAbs(getWord(%event.pos, 0) - %static.center[0]) * %event.gridBuildDir;
               
               } else
               {
                  %static.center[2] = %inst.startpos[2] + mAbs(getWord(%event.pos, 1) - %static.center[1]) * %event.gridBuildDir;
               }
            }
         }
      }
      
      //*** Record that the size and center are being changed
      %inst.changeSize = true;
      %inst.changeCenter = true;
      
   } else if(%hindex == 15) // Center handle
   {
      //*** Store the center in case we're constrained
      %center[0] = %static.center[0];
      %center[1] = %static.center[1];
      %center[2] = %static.center[2];
      
      //*** If the move is being performed in the perspective view, we need
      //*** to calculate the movement based on a plane parallel to the grid
      //*** but going through our current centre.
      if(%event.view == 3)
      {
         %point = %event.getMousePointOnWorkplaneThroughPoint(%static.center[0] SPC %static.center[1] SPC %static.center[2]);
         %static.center[0] = getWord(%point, 0);
         %static.center[1] = getWord(%point, 1);
         %static.center[2] = getWord(%point, 2);
         
      } else
      {
         %static.center[0] = getWord(%event.pos, 0);
         %static.center[1] = getWord(%event.pos, 1);
         %static.center[2] = getWord(%event.pos, 2);
      }

      //*** Adjust for constrained movement
      if(%event.FLAG_CONSTRAINED())
      {
         if(%event.FLAG_CONSTRAINEDX())
         {
            %static.center[1] = %center[1];
            %static.center[2] = %center[2];
            
         } else if(%event.FLAG_CONSTRAINEDY())
         {
            %static.center[0] = %center[0];
            %static.center[2] = %center[2];
            
         } else if(%event.FLAG_CONSTRAINEDZ())
         {
            %static.center[0] = %center[0];
            %static.center[1] = %center[1];
         }
      }

      //*** Record that the center is being changed
      %inst.changeCenter = true;
         
   } else if(%hindex >= 1 && %hindex < 15)
   {
      %forcesphere = %static.dynamicboundingbox == true && PlatformIsALTDown();
      %center = %static.center[0] SPC %static.center[1] SPC %static.center[2];
      switch(%hindex)
      {
         case 1:
            // -X-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %left = setWord(%center, 0, (getWord(%center,0) - 1.0) );
               %point = %event.getMousePointAlongLocalX(%center, %left);
               %pos = getWord(%point, 0);
            } else
            {
               %pos = getWord(%event.pos, 0);
            }
            if(%static.boundingboxtype[0] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %inst.handlepos[2,0])
                  {
                     %inst.handlepos[1,0] = %pos;
                  
                  } else
                  {
                     //*** The -X handle is being moved above the +X handle, so switch control
                     //*** over to the +X handle.
                     %inst.handlepos[1,0] = %inst.handlepos[2,0];
                     %inst.handlepos[2,0] = %pos;
                     %returnHandle = 2;
                  }
               } else
               {
                  %inst.handlepos[1,0] = %pos;
               }
               %static.size[0] = %inst.handlepos[2,0] - %inst.handlepos[1,0];
               %static.center[0] = (%inst.handlepos[2,0] - %inst.handlepos[1,0]) / 2.0 + %inst.handlepos[1,0];
               %ignoreAxis = 0;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %static.center[0])
                  {
                     %inst.handlepos[1,0] = %pos;
                  
                  } else
                  {
                     //*** The -X handle is being moved past the centre, so switch control
                     //*** over to the +X handle.
                     %inst.handlepos[2,0] = %pos;
                     %returnHandle = 2;
                  }
               } else
               {
                  %inst.handlepos[1,0] = %pos;
               }
               %static.size[0] = (%static.center[0] - %pos) * 2.0;
               %ignoreAxis = -1;
            }
            
         case 2:
            // +X-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %right = setWord(%center, 0, (getWord(%center,0) + 1.0) );
               %point = %event.getMousePointAlongLocalX(%center, %right);
               %pos = getWord(%point, 0);
            } else
            {
               %pos = getWord(%event.pos, 0);
            }
            if(%static.boundingboxtype[0] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %inst.handlepos[1,0])
                  {
                     %inst.handlepos[2,0] = %pos;
                  
                  } else
                  {
                     //*** The +X handle is being moved past the -X handle, so switch control
                     //*** over to the -X handle.
                     %inst.handlepos[2,0] = %inst.handlepos[1,0];
                     %inst.handlepos[1,0] = %pos;
                     %returnHandle = 1;
                  }
               } else
               {
                  %inst.handlepos[2,0] = %pos;
               }
               %static.size[0] = %inst.handlepos[2,0] - %inst.handlepos[1,0];
               %static.center[0] = (%inst.handlepos[2,0] - %inst.handlepos[1,0]) / 2.0 + %inst.handlepos[1,0];
               %ignoreAxis = 0;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %static.center[0])
                  {
                     %inst.handlepos[2,0] = %pos;
                  
                  } else
                  {
                     //*** The +X handle is being moved past the centre, so switch control
                     //*** over to the -X handle.
                     %inst.handlepos[1,0] = %pos;
                     %returnHandle = 1;
                  }
               } else
               {
                  %inst.handlepos[2,0] = %pos;
               }
               %static.size[0] = (%pos - %static.center[0]) * 2.0;
               %ignoreAxis = -1;
            }

         case 3:
            // -Y-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %back = setWord(%center, 1, (getWord(%center,1) - 1.0) );
               %point = %event.getMousePointAlongLocalY(%center, %back);
               %pos = getWord(%point, 1);
            } else
            {
               %pos = getWord(%event.pos, 1);
            }
            if(%static.boundingboxtype[1] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %inst.handlepos[4,1])
                  {
                     %inst.handlepos[3,1] = %pos;
                  
                  } else
                  {
                     //*** The -Y handle is being moved past the +Y handle, so switch control
                     //*** over to the +Y handle.
                     %inst.handlepos[3,1] = %inst.handlepos[4,1];
                     %inst.handlepos[4,1] = %pos;
                     %returnHandle = 4;
                  }
               } else
               {
                  %inst.handlepos[3,1] = %pos;
               }
               %static.size[1] = %inst.handlepos[4,1] - %inst.handlepos[3,1];
               %static.center[1] = (%inst.handlepos[4,1] - %inst.handlepos[3,1]) / 2.0 + %inst.handlepos[3,1];
               %ignoreAxis = 1;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %static.center[1])
                  {
                     %inst.handlepos[3,0] = %pos;
                  
                  } else
                  {
                     //*** The -Y handle is being moved past the centre, so switch control
                     //*** over to the +Y handle.
                     %inst.handlepos[4,0] = %pos;
                     %returnHandle = 4;
                  }
               } else
               {
                  %inst.handlepos[3,0] = %pos;
               }
               %static.size[1] = (%static.center[1] - %pos) * 2.0;
               %ignoreAxis = -1;
            }
            
         case 4:
            // +Y-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %front = setWord(%center, 1, (getWord(%center,1) + 1.0) );
               %point = %event.getMousePointAlongLocalY(%center, %front);
               %pos = getWord(%point, 1);
            } else
            {
               %pos = getWord(%event.pos, 1);
            }
            if(%static.boundingboxtype[1] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %inst.handlepos[3,1])
                  {
                     %inst.handlepos[4,1] = %pos;
                  
                  } else
                  {
                     //*** The +Y handle is being moved past the -Y handle, so switch control
                     //*** over to the -Y handle.
                     %inst.handlepos[4,1] = %inst.handlepos[3,1];
                     %inst.handlepos[3,1] = %pos;
                     %returnHandle = 3;
                  }
               } else
               {
                  %inst.handlepos[4,1] = %pos;
               }
               %static.size[1] = %inst.handlepos[4,1] - %inst.handlepos[3,1];
               %static.center[1] = (%inst.handlepos[4,1] - %inst.handlepos[3,1]) / 2.0 + %inst.handlepos[3,1];
               %ignoreAxis = 1;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %static.center[1])
                  {
                     %inst.handlepos[4,0] = %pos;
                  
                  } else
                  {
                     //*** The +Y handle is being moved past the centre, so switch control
                     //*** over to the -Y handle.
                     %inst.handlepos[3,0] = %pos;
                     %returnHandle = 3;
                  }
               } else
               {
                  %inst.handlepos[4,0] = %pos;
               }
               %static.size[1] = (%pos - %static.center[1]) * 2.0;
               %ignoreAxis = -1;
            }

         case 5:
            // -Z-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %down = setWord(%center, 2, (getWord(%center,2) - 1.0) );
               %point = %event.getMousePointAlongLocalZ(%center, %down);
               %pos = getWord(%point, 2);
            } else
            {
               %pos = getWord(%event.pos, 2);
            }
            if(%static.boundingboxtype[2] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %inst.handlepos[6,2])
                  {
                     %inst.handlepos[5,2] = %pos;
                  
                  } else
                  {
                     //*** The -Z handle is being moved above the +Z handle, so switch control
                     //*** over to the +Z handle.
                     %inst.handlepos[5,2] = %inst.handlepos[6,2];
                     %inst.handlepos[6,2] = %pos;
                     %returnHandle = 6;
                  }
               } else
               {
                  %inst.handlepos[5,2] = %pos;
               }
               %static.size[2] = %inst.handlepos[6,2] - %inst.handlepos[5,2];
               %static.center[2] = (%inst.handlepos[6,2] - %inst.handlepos[5,2]) / 2.0 + %inst.handlepos[5,2];
               %ignoreAxis = 2;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos <= %static.center[2])
                  {
                     %inst.handlepos[5,0] = %pos;
                  
                  } else
                  {
                     //*** The -Z handle is being moved past the centre, so switch control
                     //*** over to the +Z handle.
                     %inst.handlepos[6,0] = %pos;
                     %returnHandle = 6;
                  }
               } else
               {
                  %inst.handlepos[5,0] = %pos;
               }
               %static.size[2] = (%static.center[2] - %pos) * 2.0;
               %ignoreAxis = -1;
            }
            
         case 6:
            // +Z-axis
            //*** If the handle is being modified in a perspective view, then
            //*** use the helper %event function.  Otherwise use the values that
            //*** were given by %event.
            if(%event.view == 3)
            {
               %up = setWord(%center, 2, (getWord(%center,2) + 1.0) );
               %point = %event.getMousePointAlongLocalZ(%center, %up);
               %pos = getWord(%point, 2);
            } else
            {
               %pos = getWord(%event.pos, 2);
            }
            if(%static.boundingboxtype[2] == false && !%forcesphere)
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %inst.handlepos[5,2])
                  {
                     %inst.handlepos[6,2] = %pos;
                  
                  } else
                  {
                     //*** The +Z handle is being moved below the -Z handle, so switch control
                     //*** over to the -Z handle.
                     %inst.handlepos[6,2] = %inst.handlepos[5,2];
                     %inst.handlepos[5,2] = %pos;
                     %returnHandle = 5;
                  }
               } else
               {
                  %inst.handlepos[6,2] = %pos;
               }
               %static.size[2] = %inst.handlepos[6,2] - %inst.handlepos[5,2];
               %static.center[2] = (%inst.handlepos[6,2] - %inst.handlepos[5,2]) / 2.0 + %inst.handlepos[5,2];
               %ignoreAxis = 2;
            } else
            {
               if(!%static.ToolBBAllowNegativeSize)
               {
                  if(%pos >= %static.center[2])
                  {
                     %inst.handlepos[6,0] = %pos;
                  
                  } else
                  {
                     //*** The +Z handle is being moved past the centre, so switch control
                     //*** over to the -Z handle.
                     %inst.handlepos[5,0] = %pos;
                     %returnHandle = 5;
                  }
               } else
               {
                  %inst.handlepos[6,0] = %pos;
               }
               %static.size[2] = (%pos - %static.center[2]) * 2.0;
               %ignoreAxis = -1;
            }
            
         case 7:
            //*** -x -y -z corner                                                   c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 1, 2, 4, 5,  3,  6,  8,   7, -1, -1, -1) + 6;
            
         case 8:
            //*** x -y -z corner                                                    c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 2, 1, 3, 6,  4,  5,  7,   8,  1, -1, -1) + 6;
            
         case 9:
            //*** x y -z corner                                                     c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 3, 4, 2, 7,  1,  8,  6,   5,  1,  1, -1) + 6;
            
         case 10:
            //*** -x y -z corner                                                    c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 4, 3, 1, 8,  2,  7,  5,   6, -1,  1, -1) + 6;
            
         case 11:
            //*** -x -y z corner                                                    c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 5, 6, 8, 1,  7,  2,  4,   3, -1, -1,  1) + 6;
            
         case 12:
            //*** x -y z corner                                                     c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 6, 5, 7, 2,  8,  1,  3,   4,  1, -1,  1) + 6;
            
         case 13:
            //*** x y z corner                                                      c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 7, 8, 6, 3,  5,  4,  2,   1,  1,  1,  1) + 6;
            
         case 14:
            //*** -x y z corner                                                     c ox oy oz oxy oxz oyz oxyz  bx  by  bz
            %returnHandle = ToolBB::__CalculateCornerHandle(%inst, %static, %event, 8, 7, 5, 4,  6,  3,  1,   2, -1,  1,  1) + 6;
      }
      
      //*** Record that the size is being changed
      %inst.changeSize = true;
      %inst.changeCenter = true;
   }
      
   //*** Move the handles into position for the new redraw
   ToolBB::__BuildHandles(%inst, %static, %ignoreAxis);
   
   //*** Return the handle that is to be controlled by the user
   return %returnHandle;
}

//*** This function is called in your plug-in's Draw(), passing along the
//*** ToolDrawing class.  It draws the bounding box and all appropriate handles.
function ToolBB::draw(%inst, %static, %draw)
{
   //*** Draw the centre handle's standard image
   %draw.handle(%draw.HANDLE_CUBE(), "0 255 255", %inst.handlepos[15,0], %inst.handlepos[15,1], %inst.handlepos[15,2]);

   %view = %draw.getView();
      
   //*** Draw the x-axis handles' standard image
   if(%view != 4 && %view != 5)
   {
      %draw.handle(%draw.HANDLE_CUBE(), "255 0 0", %inst.handlepos[1,0], %inst.handlepos[1,1], %inst.handlepos[1,2]);
      %draw.handle(%draw.HANDLE_CUBE(), "255 0 0", %inst.handlepos[2,0], %inst.handlepos[2,1], %inst.handlepos[2,2]);
   }
      
   //*** Draw the y-axis handles' standard image
   if(%view != 2 && %view != 3)
   {
      %draw.handle(%draw.HANDLE_CUBE(), "0 255 0", %inst.handlepos[3,0], %inst.handlepos[3,1], %inst.handlepos[3,2]);
      %draw.handle(%draw.HANDLE_CUBE(), "0 255 0", %inst.handlepos[4,0], %inst.handlepos[4,1], %inst.handlepos[4,2]);
   }
      
   //*** Draw the z-axis handles' standard image
   if(%view != 0 && %view != 1)
   {
      %draw.handle(%draw.HANDLE_CUBE(), "0 0 255", %inst.handlepos[5,0], %inst.handlepos[5,1], %inst.handlepos[5,2]);
      %draw.handle(%draw.HANDLE_CUBE(), "0 0 255", %inst.handlepos[6,0], %inst.handlepos[6,1], %inst.handlepos[6,2]);
   }
      
   //*** Build the corners of the tool's bounding box
   ToolBB::__BuildCorners(%inst, %static);
   
   //*** Draw the corner handles
   for(%i=1; %i<=8; %i++)
   {
      %draw.handle(%draw.HANDLE_CUBE(), "255 255 255", %inst.cornerpos[%i,0], %inst.cornerpos[%i,1], %inst.cornerpos[%i,2]);
   }
      
   if(getWordCount(%static.boundingBoxCustomColor) > 0)
   {
      %draw.changeColor(%static.boundingBoxCustomColor);

   } else
   {
      %draw.changeColor(pref.getColorI("CreateGeometryLineColor"));
   }

   //*** Draw the lines around the bottom
   %draw.moveTo(tool.DRAW_ABSOLUTE(), %inst.cornerpos[1,0], %inst.cornerpos[1,1], %inst.cornerpos[1,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[2,0], %inst.cornerpos[2,1], %inst.cornerpos[2,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[3,0], %inst.cornerpos[3,1], %inst.cornerpos[3,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[4,0], %inst.cornerpos[4,1], %inst.cornerpos[4,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[1,0], %inst.cornerpos[1,1], %inst.cornerpos[1,2]);

   //*** Draw the lines around the top
   %draw.moveTo(tool.DRAW_ABSOLUTE(), %inst.cornerpos[5,0], %inst.cornerpos[5,1], %inst.cornerpos[5,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[6,0], %inst.cornerpos[6,1], %inst.cornerpos[6,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[7,0], %inst.cornerpos[7,1], %inst.cornerpos[7,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[8,0], %inst.cornerpos[8,1], %inst.cornerpos[8,2]);
   %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[5,0], %inst.cornerpos[5,1], %inst.cornerpos[5,2]);

   //*** Draw the lines from top to bottom corners
   for(%i=1; %i<=4; %i++)
   {
      %draw.moveTo(tool.DRAW_ABSOLUTE(), %inst.cornerpos[%i,0], %inst.cornerpos[%i,1], %inst.cornerpos[%i,2]);
      %draw.lineTo((tool.DRAW_SOLID() + tool.DRAW_ABSOLUTE()), %inst.cornerpos[%i+4,0], %inst.cornerpos[%i+4,1], %inst.cornerpos[%i+4,2]);
   }
}

//**********************************************************************************
//*** Bounding Box functions used internally
//**********************************************************************************

//*** Build the handles
function ToolBB::__BuildHandles(%inst, %static, %ignoreAxis)
{
   for(%i=0; %i<3; %i++)
   {
      //*** The centre handle
      %inst.handlepos[15,%i] = %static.center[%i];
      
      //*** Helper
      %halfsize[%i] = %static.size[%i] * 0.5;
   }
   
   //*** The middle face handles
   if(%ignoreAxis != 0)
   {
      //*** -x handle
      %inst.handlepos[1,0] = %static.center[0] - %halfsize[0];
      %inst.handlepos[1,1] = %static.center[1];
      %inst.handlepos[1,2] = %static.center[2];
      
      //*** x handle
      %inst.handlepos[2,0] = %static.center[0] + %halfsize[0];
      %inst.handlepos[2,1] = %static.center[1];
      %inst.handlepos[2,2] = %static.center[2];
   }
      
   if(%ignoreAxis != 1)
   {
      //*** -y handle
      %inst.handlepos[3,0] = %static.center[0];
      %inst.handlepos[3,1] = %static.center[1] - %halfsize[1];
      %inst.handlepos[3,2] = %static.center[2];
      
      //*** y handle
      %inst.handlepos[4,0] = %static.center[0];
      %inst.handlepos[4,1] = %static.center[1] + %halfsize[1];
      %inst.handlepos[4,2] = %static.center[2];
   }
      
   if(%ignoreAxis != 2)
   {
      //*** -z handle
      %inst.handlepos[5,0] = %static.center[0];
      %inst.handlepos[5,1] = %static.center[1];
      %inst.handlepos[5,2] = %static.center[2] - %halfsize[2];
      
      //*** z handle
      %inst.handlepos[6,0] = %static.center[0];
      %inst.handlepos[6,1] = %static.center[1];
      %inst.handlepos[6,2] = %static.center[2] + %halfsize[2];
   }
      
   //*** Initial sizing handle
   %inst.handlepos[0,0] = %static.center[0] - %halfsize[0];
   %inst.handlepos[0,1] = %static.center[1] + %halfsize[1];
   %inst.handlepos[0,2] = %static.center[2] + %halfsize[2];
}

//*** Build the corners of the bounding box used for drawing
function ToolBB::__BuildCorners(%inst, %static)
{
   for(%i=0; %i<3; %i++)
   {
      //*** Helper
      %halfsize[%i] = %static.size[%i] * 0.5;
   }
   
   //*** -x -y -z corner
   %inst.cornerpos[1,0] = %static.center[0] - %halfsize[0];
   %inst.cornerpos[1,1] = %static.center[1] - %halfsize[1];
   %inst.cornerpos[1,2] = %static.center[2] - %halfsize[2];
      
   //*** x -y -z corner
   %inst.cornerpos[2,0] = %static.center[0] + %halfsize[0];
   %inst.cornerpos[2,1] = %static.center[1] - %halfsize[1];
   %inst.cornerpos[2,2] = %static.center[2] - %halfsize[2];
      
   //*** x y -z corner
   %inst.cornerpos[3,0] = %static.center[0] + %halfsize[0];
   %inst.cornerpos[3,1] = %static.center[1] + %halfsize[1];
   %inst.cornerpos[3,2] = %static.center[2] - %halfsize[2];
      
   //*** -x y -z corner
   %inst.cornerpos[4,0] = %static.center[0] - %halfsize[0];
   %inst.cornerpos[4,1] = %static.center[1] + %halfsize[1];
   %inst.cornerpos[4,2] = %static.center[2] - %halfsize[2];
      
   //*** -x -y z corner
   %inst.cornerpos[5,0] = %static.center[0] - %halfsize[0];
   %inst.cornerpos[5,1] = %static.center[1] - %halfsize[1];
   %inst.cornerpos[5,2] = %static.center[2] + %halfsize[2];
      
   //*** x -y z corner
   %inst.cornerpos[6,0] = %static.center[0] + %halfsize[0];
   %inst.cornerpos[6,1] = %static.center[1] - %halfsize[1];
   %inst.cornerpos[6,2] = %static.center[2] + %halfsize[2];
      
   //*** x y z corner
   %inst.cornerpos[7,0] = %static.center[0] + %halfsize[0];
   %inst.cornerpos[7,1] = %static.center[1] + %halfsize[1];
   %inst.cornerpos[7,2] = %static.center[2] + %halfsize[2];
      
   //*** -x y z corner
   %inst.cornerpos[8,0] = %static.center[0] - %halfsize[0];
   %inst.cornerpos[8,1] = %static.center[1] + %halfsize[1];
   %inst.cornerpos[8,2] = %static.center[2] + %halfsize[2];
}

//*** This will perform the necessary calculations for a corner handle.
//*** %ox, %oy and %oz are the opposite corners from the given one.
//*** %bx, %by and %bz are a bias for each axis to indicate if the handle
//*** is normally negative or positive from the center.  Returns
//*** the corner number that the user should continue to manipulate which is
//*** usually the same as the one passed to this function.
function ToolBB::__CalculateCornerHandle(%inst, %static, %event, %corner, %ox, %oy, %oz, %oxy, %oxz, %oyz, %oxyz, %bx, %by, %bz)
{
   %forcesphere = %static.dynamicboundingbox == true && PlatformIsALTDown();

   //*** Store the current position in case we're constrained
   %orig[0] = %inst.cornerpos[%corner, 0];
   %orig[1] = %inst.cornerpos[%corner, 1];
   %orig[2] = %inst.cornerpos[%corner, 2];
      
   //*** If the move is being performed in the perspective view, we need
   //*** to calculate the movement based on a plane parallel to the grid
   //*** but going through our current corner.
   if(%event.view == 3)
   {
      %point = %event.getMousePointOnWorkplaneThroughPoint(%orig[0] SPC %orig[1] SPC %orig[2]);
      %inst.cornerpos[%corner, 0] = getWord(%point, 0);
      %inst.cornerpos[%corner, 1] = getWord(%point, 1);
      %inst.cornerpos[%corner, 2] = getWord(%point, 2);
         
   } else
   {
      %inst.cornerpos[%corner, 0] = getWord(%event.pos, 0);
      %inst.cornerpos[%corner, 1] = getWord(%event.pos, 1);
      %inst.cornerpos[%corner, 2] = getWord(%event.pos, 2);
   }

   //*** Adjust for constrained movement
   //if(%event.FLAG_CONSTRAINED())
   //{
   //   if(%event.FLAG_CONSTRAINEDX())
   //   {
   //      %inst.cornerpos[%corner, 1] = %orig[1];
   //      %inst.cornerpos[%corner, 2] = %orig[2];
            
   //   } else if(%event.FLAG_CONSTRAINEDY())
   //   {
   //      %inst.cornerpos[%corner, 0] = %orig[0];
   //      %inst.cornerpos[%corner, 2] = %orig[2];
            
   //   } else if(%event.FLAG_CONSTRAINEDZ())
   //   {
   //      %inst.cornerpos[%corner, 0] = %orig[0];
   //      %inst.cornerpos[%corner, 1] = %orig[1];
   //   }
   //}

   //*** Adjust for uniform scale
   if(PlatformIsSHIFTDown() && (%event.view == 0 || (%event.view == 3 && %event.gridBuildAxis == 0)) )
   {
      if(%forcesphere)
      {
         %dy = %inst.cornerpos[%corner, 1] - %static.center[1];
         %dz = %inst.cornerpos[%corner, 2] - %static.center[2];
         %ady = mAbs(%dy);
         %adz = mAbs(%dz);
         if(%ady > %adz)
         {
            if(%dz >= 0)
            {
               %inst.cornerpos[%corner, 2] = %static.center[2] + %ady;

            } else
            {
               %inst.cornerpos[%corner, 2] = %static.center[2] - %ady;
            }
         } else
         {
            if(%dy >= 0)
            {
               %inst.cornerpos[%corner, 1] = %static.center[1] + %adz;

            } else
            {
               %inst.cornerpos[%corner, 1] = %static.center[1] - %adz;
            }
         }

      } else
      {
         %dy = %inst.cornerpos[%corner, 1] - %inst.cornerpos[%oyz, 1];
         %dz = %inst.cornerpos[%corner, 2] - %inst.cornerpos[%oyz, 2];
         %ady = mAbs(%dy);
         %adz = mAbs(%dz);
         if(%ady > %adz)
         {
            if(%dz >= 0)
            {
               %inst.cornerpos[%corner, 2] = %inst.cornerpos[%oyz, 2] + %ady;

            } else
            {
               %inst.cornerpos[%corner, 2] = %inst.cornerpos[%oyz, 2] - %ady;
            }
         } else
         {
            if(%dy >= 0)
            {
               %inst.cornerpos[%corner, 1] = %inst.cornerpos[%oyz, 1] + %adz;

            } else
            {
               %inst.cornerpos[%corner, 1] = %inst.cornerpos[%oyz, 1] - %adz;
            }
         }
      }

   } else if(PlatformIsSHIFTDown() && (%event.view == 1 || (%event.view == 3 && %event.gridBuildAxis == 1)) )
   {
      if(%forcesphere)
      {
         %dx = %inst.cornerpos[%corner, 0] - %static.center[0];
         %dz = %inst.cornerpos[%corner, 2] - %static.center[2];
         %adx = mAbs(%dx);
         %adz = mAbs(%dz);
         if(%adx > %adz)
         {
            if(%dz >= 0)
            {
               %inst.cornerpos[%corner, 2] = %static.center[2] + %adx;

            } else
            {
               %inst.cornerpos[%corner, 2] = %static.center[2] - %adx;
            }
         } else
         {
            if(%dx >= 0)
            {
               %inst.cornerpos[%corner, 0] = %static.center[0] + %adz;

            } else
            {
               %inst.cornerpos[%corner, 0] = %static.center[0] - %adz;
            }
         }

      } else
      {
         %dx = %inst.cornerpos[%corner, 0] - %inst.cornerpos[%oxz, 0];
         %dz = %inst.cornerpos[%corner, 2] - %inst.cornerpos[%oxz, 2];
         %adx = mAbs(%dx);
         %adz = mAbs(%dz);
         if(%adx > %adz)
         {
            if(%dz >= 0)
            {
               %inst.cornerpos[%corner, 2] = %inst.cornerpos[%oxz, 2] + %adx;

            } else
            {
               %inst.cornerpos[%corner, 2] = %inst.cornerpos[%oxz, 2] - %adx;
            }
         } else
         {
            if(%dx >= 0)
            {
               %inst.cornerpos[%corner, 0] = %inst.cornerpos[%oxz, 0] + %adz;

            } else
            {
               %inst.cornerpos[%corner, 0] = %inst.cornerpos[%oxz, 0] - %adz;
            }
         }
      }

   } else if(PlatformIsSHIFTDown() && (%event.view == 2 || (%event.view == 3 && %event.gridBuildAxis == 2)) )
   {
      if(%forcesphere)
      {
         %dx = %inst.cornerpos[%corner, 0] - %static.center[0];
         %dy = %inst.cornerpos[%corner, 1] - %static.center[1];
         %adx = mAbs(%dx);
         %ady = mAbs(%dy);
         if(%adx > %ady)
         {
            if(%dy >= 0)
            {
               %inst.cornerpos[%corner, 1] = %static.center[1] + %adx;

            } else
            {
               %inst.cornerpos[%corner, 1] = %static.center[1] - %adx;
            }
         } else
         {
            if(%dx >= 0)
            {
               %inst.cornerpos[%corner, 0] = %static.center[0] + %ady;

            } else
            {
               %inst.cornerpos[%corner, 0] = %static.center[0] - %ady;
            }
         }

      } else
      {
         %dx = %inst.cornerpos[%corner, 0] - %inst.cornerpos[%oxy, 0];
         %dy = %inst.cornerpos[%corner, 1] - %inst.cornerpos[%oxy, 1];
         %adx = mAbs(%dx);
         %ady = mAbs(%dy);
         if(%adx > %ady)
         {
            if(%dy >= 0)
            {
               %inst.cornerpos[%corner, 1] = %inst.cornerpos[%oxy, 1] + %adx;

            } else
            {
               %inst.cornerpos[%corner, 1] = %inst.cornerpos[%oxy, 1] - %adx;
            }
         } else
         {
            if(%dx >= 0)
            {
               %inst.cornerpos[%corner, 0] = %inst.cornerpos[%oxy, 0] + %ady;

            } else
            {
               %inst.cornerpos[%corner, 0] = %inst.cornerpos[%oxy, 0] - %ady;
            }
         }
      }
   }
   
   //*** Flag to indicate which corner the user should continue to manipulate
   %cornerflag = 0;
   %returncorner = %corner;
   
   //*** X-axis
   if(%static.boundingboxtype[0] || %forcesphere)
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[0] - %static.center[0];
      %newcalc = %inst.cornerpos[%corner, 0] - %static.center[0];
      
      //*** Resize about the center
      %static.size[0] = (%inst.cornerpos[%corner, 0] - %static.center[0]) * 2.0;
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[0] = mAbs(%static.size[0]);
      }
      
   } else
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[0] - %inst.cornerpos[%ox, 0];
      %newcalc = %inst.cornerpos[%corner, 0] - %inst.cornerpos[%ox, 0];

      //*** Resize with the opposite corner
      %static.size[0] = (%inst.cornerpos[%corner, 0] - %inst.cornerpos[%ox, 0]) * %bx;
      %absSize = mAbs(%static.size[0]);
      if(%inst.cornerpos[%corner, 0] < %inst.cornerpos[%ox, 0])
      {
         %static.center[0] = %inst.cornerpos[%corner, 0] + %absSize / 2.0;

      } else
      {
         %static.center[0] = %inst.cornerpos[%ox, 0] + %absSize / 2.0;
      }
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[0] = %absSize;
      }
   }
   //*** Checked if we've just gone past the center and need to switch corner handles
   if(!%static.ToolBBAllowNegativeSize && (%origcalc > 0 && %newcalc <= 0 || %origcalc < 0 && %newcalc >= 0))
   {
      %cornerflag = %cornerflag | 1;
   }

   //*** Y-axis
   if(%static.boundingboxtype[1] || %forcesphere)
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[1] - %static.center[1];
      %newcalc = %inst.cornerpos[%corner, 1] - %static.center[1];
      
      //*** Resize about the center
      %static.size[1] = (%inst.cornerpos[%corner, 1] - %static.center[1]) * 2.0;
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[1] = mAbs(%static.size[1]);
      }
      
   } else
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[1] - %inst.cornerpos[%oy, 1];
      %newcalc = %inst.cornerpos[%corner, 1] - %inst.cornerpos[%oy, 1];

      //*** Resize with the opposite corner
      %static.size[1] = (%inst.cornerpos[%corner, 1] - %inst.cornerpos[%oy, 1]) * %by;
      %absSize = mAbs(%static.size[1]);
      if(%inst.cornerpos[%corner, 1] < %inst.cornerpos[%oy, 1])
      {
         %static.center[1] = %inst.cornerpos[%corner, 1] + %absSize / 2.0;
      
      } else
      {
         %static.center[1] = %inst.cornerpos[%oy, 1] + %absSize / 2.0;
      }
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[1] = %absSize;
      }
   }
   //*** Checked if we've just gone past the center and need to switch corner handles
   if(!%static.ToolBBAllowNegativeSize && (%origcalc > 0 && %newcalc <= 0 || %origcalc < 0 && %newcalc >= 0))
   {
      %cornerflag = %cornerflag | 2;
   }

   //*** Z-axis
   if(%static.boundingboxtype[2] || %forcesphere)
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[2] - %static.center[2];
      %newcalc = %inst.cornerpos[%corner, 2] - %static.center[2];
      
      //*** Resize about the center
      %static.size[2] = (%inst.cornerpos[%corner, 2] - %static.center[2]) * 2.0;
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[2] = mAbs(%static.size[2]);
      }
      
   } else
   {
      //*** Calculate distances for a later check if we need to switch handles
      %origcalc = %orig[2] - %inst.cornerpos[%oz, 2];
      %newcalc = %inst.cornerpos[%corner, 2] - %inst.cornerpos[%oz, 2];

      //*** Resize with the opposite corner
      %static.size[2] = (%inst.cornerpos[%corner, 2] - %inst.cornerpos[%oz, 2]) * %bz;
      %absSize = mAbs(%static.size[2]);
      if(%inst.cornerpos[%corner, 2] < %inst.cornerpos[%oz, 2])
      {
         %static.center[2] = %inst.cornerpos[%corner, 2] + %absSize / 2.0;
   
      } else
      {
         %static.center[2] = %inst.cornerpos[%oz, 2] + %absSize / 2.0;
      }
      if(!%static.ToolBBAllowNegativeSize)
      {
         %static.size[2] = %absSize;
      }
   }
   //*** Checked if we've just gone past the center and need to switch corner handles
   if(!%static.ToolBBAllowNegativeSize && (%origcalc > 0 && %newcalc <= 0 || %origcalc < 0 && %newcalc >= 0))
   {
      %cornerflag = %cornerflag | 4;
   }
   
   //*** Do we need to change to a different corner handle?
   switch(%cornerflag)
   {
      case 1:
         //*** Switch X-axis handle
         %returncorner = %ox;

      case 2:
         //*** Switch Y-axis handle
         %returncorner = %oy;

      case 3:
         //*** Switch X & Y-axis handle
         %returncorner = %oxy;

      case 4:
         //*** Switch Z-axis handle
         %returncorner = %oz;

      case 5:
         //*** Switch X & Z-axis handle
         %returncorner = %oxz;

      case 6:
         //*** Switch Y & Z-axis handle
         %returncorner = %oyz;

      case 7:
         //*** Switch X & Y & Z-axis handle
         %returncorner = %oxyz;
   }
   
   return %returncorner;
}
