//*** Base methods that all tools may override.

   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      tool.activateErrorMsg = "Tool is missing required Activate() method.";
      return tool.FUNC_BADGENERAL();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      %plugin = %inst.instance;
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      return true;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may interact with.
      return 1;
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse button has just been
   // depressed.  %event will contain the particulars of where the mouse button was pressed
   // to allow for the tool to set itself up for the first time.  The value returned is the
   // (zero-based) index of the handle that is now active and will be dragged by the user
   // until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      return 0;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine which handle
   // has been selected.  In both cases, %info is a ScriptObject that contains the
   // .pos[3] fields that are to be filled in with the requested handle's position.
   // This function returns the priority of the handle, the higher the number the higher
   // the priority.  This is used to determine which handle should be selected when two
   // or more handles overlap on the screen.  If -1 is returned, then the handle is
   // considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //*** Fill in the handle's position and return its priority
      if(%hindex == 0)
      {
         %info.pos[0] = 0;
         %info.pos[1] = 0;
         %info.pos[2] = 0;
      
         //*** Return the handle's priority
         return %hindex; // set the handle's priority the same as its index
      }
      
      //*** The given handle index is not one of ours
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      return %hindex;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      return tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      return tool.EDIT_DONOTHING();
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  %data is optional
   // and holds any event data, although most events don't produce any.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent, %data)
   {
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      return tool.FUNC_OK();
   }
