//
// TextureJustifyFit.cs
//
// Defines the TextureJustifyFit tool plug-in for fitting and aligning textures to a face.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// offset   - Defines the "x y z" offset that the geometry was last moved by - the relative mouse delta
//
// The instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeOffset - Flag to indicate that the tool's offset has changed

// Revision History:
// July 14, 2006		Matt Fairfax		Created script file
//

package TextureJustifyFit
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("TextureJustifyFit: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         TextureJustifyFit_InitStatic(%static);
      }
      
      //*** Get the values from the scene if the selection criteria is met
      TextureJustifyFit_GetValues(%static);
      
      //*** Restore the current selection type when this tool is deactivated
      %static.selectRestore = select.getSelectionType();
      
      if (select.getSelectionType() !$= "SelectFaces")
         select.setSelectionType("SelectFaces");
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;

      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = false;
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeOffset = false;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_NOTOOLCURSOR(); // Set up the tool with the standard transform tool GUI
      %inst.flagsApply = tool.AFLAG_STANDARDTRANSFORM();     // Set up the tool with the standard transform tool apply() operation

      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("TextureJustifyFit: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
      
      select.setSelectionType(%static.selectRestore);
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("TextureJustifyFit: MouseDown(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;

      //*** We're automatically activated when the mouse goes down
      %plugin.active = true;
      
      //*** We want to process the mouse drags directly, so return true.
      return true;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("TextureJustifyFit: MouseDrag(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;

      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("TextureJustifyFit: MouseUp(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;
      
      %scene = scene.getCurrent();
      %map = scene.getCurrentMap();
      
      %faces = scene.getCurrentMap().raySelectFaces(%event.pos, %event.endPos);
      
      if (getWordCount(%faces) > 0)
      {
         scene.getCurrentMap().unselectAllFaces();
         scene.getCurrentMap().selectFaces(getWords(%faces, 0, 1));
         
         //*** Refresh our tool properties
         TextureJustifyFit_GetValues(%plugin.static);
         %plugin.dirty = tool.DIRTY_APPEARANCE();
         select.__updateSelectionAxisGizmo(%scene, %map);
      }
      
      //*** Notify that we have performed are action and are now done
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_ACCEPT();
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("TextureJustifyFit: HandleCount(" @ %inst @ "," @ %event @ ")");

      //*** We don't have any handles
      return 0;
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("TextureJustifyFit: HandleInit(" @ %inst @ "," @ %event @ ")");

      //*** We don't have any handles
      return -1;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("TextureJustifyFit: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      //*** We don't have any handles
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("TextureJustifyFit: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      //*** We don't have any handles
      return -1;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("TextureJustifyFit: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("TextureJustifyFit: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** We dont' have anything to draw, so indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("TextureJustifyFit: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("TextureJustifyFit: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      if(%keep)
      {
         %plugin.update = tool.EDIT_UPDATE();
         tool.refresh();
         
      } else
      {
         %plugin.update = tool.EDIT_DONOTHING();
      }
      %plugin.active = false;
      %plugin.changeOffset = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("TextureJustifyFit: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("TextureJustifyFit: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: the displayed offset).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_DONOTHING(); // Don't do anything when activated.  We only perform the actual change from EndEditAction()
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeOffset = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            TextureJustifyFit_Reset(%plugin);
            %plugin.update = tool.EDIT_DONOTHING(); // Don't do anything when activated.  We only perform the actual change from EndEditAction()
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeOffset = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("TextureJustifyFit: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Texture: Justify and Fit");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField(-1, "Justify:",    "divider");
      %form.addField( 0, "Left",        "button");
      %form.addField( 1, "Right",       "button");
      %form.addField( 2, "Top",         "button");
      %form.addField( 3, "Bottom",      "button");
      %form.addField(-1, "Fit to:",     "divider");
      %form.addField( 4, "Width",       "button");
      %form.addField( 5, "Height",      "button");
      %form.addField( 6, "Face",        "button");
      %form.addField(-1, "Flip:",       "divider");
      %form.addField( 7, "Vertical",    "button");
      %form.addField( 8, "Horizontal",  "button");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("TextureJustifyFit: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("TextureJustifyFit: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");
      
      %scene = scene.getCurrent();
      %map = scene.getCurrentMap();

      %plugin = %inst.instance;
      
      %brush = getWord(%plugin.static.faces, 0);
      %face = getWord(%plugin.static.faces, 1);
      
      switch(%id)
      {
         //*** Handle the 'Justify Left' button
         case 0:
            scene.getCurrentMap().justifyLeft(%brush, %face);
         //*** Handle the 'Justify Right' button
         case 1:
            scene.getCurrentMap().justifyRight(%brush, %face);
         //*** Handle the 'Justify Top' button
         case 2:
            scene.getCurrentMap().justifyTop(%brush, %face);
         //*** Handle the 'Justify Bottom' button
         case 3:
            scene.getCurrentMap().justifyBottom(%brush, %face);
         //*** Handle the 'Fit to Width' button
         case 4:
            scene.getCurrentMap().fitToWidth(%brush, %face);
         //*** Handle the 'Fit to Height' button
         case 5:
            scene.getCurrentMap().fitToHeight(%brush, %face);
         //*** Handle the 'Fit to Face' button
         case 6:
            scene.getCurrentMap().fitToFace(%brush, %face);
         //*** Handle the 'Flip Vertical' button
         case 7:
            %plugin.static.vaxis = VectorScale(%plugin.static.vaxis, -1);
            
            %shiftU = getWord(%plugin.static.shift, 0);
            %shiftV = getWord(%plugin.static.shift, 1);
            
            %scaleU = getWord(%plugin.static.scale, 0);
            %scaleV = getWord(%plugin.static.scale, 1);
            
            %texgen = %plugin.static.uaxis SPC %shiftU SPC %plugin.static.vaxis SPC %shiftV SPC %plugin.static.rotation SPC %scaleU SPC %scaleV;
         
            for (%i = 0; %i < getWordCount(%plugin.static.faces); %i+=2)
            {
               %brush = getWord(%plugin.static.faces, %i);
               %face = getWord(%plugin.static.faces, %i + 1);

               %map.setFaceTexGens(%brush, %face, %texgen);
            }
         //*** Handle the 'Flip Horizontal' button
         case 8:
            %plugin.static.uaxis = VectorScale(%plugin.static.uaxis, -1);
            
            %shiftU = getWord(%plugin.static.shift, 0);
            %shiftV = getWord(%plugin.static.shift, 1);
            
            %scaleU = getWord(%plugin.static.scale, 0);
            %scaleV = getWord(%plugin.static.scale, 1);
            
            %texgen = %plugin.static.uaxis SPC %shiftU SPC %plugin.static.vaxis SPC %shiftV SPC %plugin.static.rotation SPC %scaleU SPC %scaleV;
         
            for (%i = 0; %i < getWordCount(%plugin.static.faces); %i+=2)
            {
               %brush = getWord(%plugin.static.faces, %i);
               %face = getWord(%plugin.static.faces, %i + 1);

               %map.setFaceTexGens(%brush, %face, %texgen);
            }
      }
      
      //*** Refresh our tool properties
      TextureJustifyFit_GetValues(%plugin.static);
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      select.__updateSelectionAxisGizmo(%scene, %map);
      
      //*** Indicate that everything needs to be redrawn
      //%plugin.update = tool.EDIT_UPDATE();
      //%plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function TextureJustifyFit_DefaultValues(%static)
   {
      %static.faces = "";
   }
   
   //*** Get the values from the scene if we meet the proper selection criteria
   function TextureJustifyFit_GetValues(%static)
   {
      //error("TextureJustifyFit_GetValues called");
   
      %map = scene.getCurrentMap();
      
      if(%map)
      {
         if(%map.getNumSelectedFaces() > 0)
         {
            %static.faces = %map.getSelectedFaces();
            
            %brush = getWord(%static.faces, 0);
            %face = getWord(%static.faces, 1);
            
            %texgens = %map.getFaceTexgens(%brush, %face);
            
            // x y z shift x y z shift rotation scalex scaley
            %static.uaxis = getWords(%texgens, 0, 2);
            %static.vaxis = getWords(%texgens, 4, 6);
            %static.shift = getWord(%texgens, 3) SPC getWord(%texgens, 7);
            %static.rotation = getWord(%texgens, 8);
            %static.scale = getWord(%texgens, 9) SPC getWord(%texgens, 10);
         }
      }
   }

   //*** Init the static object
   function TextureJustifyFit_InitStatic(%static)
   {
      //*** Setup default values
      TextureJustifyFit_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the transform instance to default values
   function TextureJustifyFit_Reset(%inst)
   {
      //*** Setup default values
      TextureJustifyFit_GetValues(%inst.static);
   }
   
};

tool.register("TextureJustifyFit", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS(), "Justify and Fit Textures" );
