//
// resetpivotpoint.cs
//
// Resets the pivot points of all selected brushes.
//
// Revision History:
// March 24, 2007			David Wyand		Created script file
//

package ResetPivotPoint
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("ResetPivotPoint: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Only brush selection is supported
      if(select.getSelectionType() !$= "SelectPrimitives")
      {
         tool.activateErrorMsg = "This tool only works in the 'Brushes' selection mode.";
         return tool.FUNC_BADSELECTMODE();
      }
      
      %scene = scene.getCurrent();
      %map = scene.getCurrentMap();

      //*** Make sure that at least one brush is selected...
      if(%map.getNumSelectedBrushes() == 0)
      {
         tool.activateErrorMsg = "Please select one or more brushes to reset.";
         return tool.FUNC_BADGENERAL();
      }
      
      //*** This tool doesn't need to make use of the instance
      %inst.instance = 0;
      %inst.flagsInterface = tool.IFLAG_NONE(); // No special interface flags as there will be no interface
      %inst.flagsApply = tool.AFLAG_NONE();     // No special actions when tool is applied
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("ResetPivotPoint: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // Execute()
   //
   // This function is called to perform the actual work of the tool, and does not
   // return a value.
   function Plugin::Execute(%this, %inst)
   {
      //error("ResetPivotPoint: Execute(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      //*** Perform the pivot point reset
      %map = scene.getCurrentMap();
      %brushes = %map.getSelectedBrushes();
      %count = getWordCount(%brushes);
      for(%i=0; %i<%count; %i++)
      {
         %map.setBrushPivot(getWord(%brushes, %i), "0 0 0 1 0 0 0");
      }

      MessageBoxOK("RESET PIVOT POINTS", %count SPC "brushes have been reset");
   }
};

tool.register("ResetPivotPoint", tool.typeGeneric(), tool.RFLAG_NONE(), "Reset Selected Pivot Points" );
