//
// quantize.cs
//
// Defines the Quantize tool plug-in that snaps the current selection to the grid.
//
// Revision History:
// August 17, 2006			David Wyand		Created script file
//

package Quantize
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Quantize: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Only brush and vertex selection is supported
      if(select.getSelectionType() !$= "SelectPrimitives" && select.getSelectionType() !$= "SelectVertices")
      {
         tool.activateErrorMsg = "This tool only works in the 'Brushes' or 'Vertices' selection modes.";
         return tool.FUNC_BADSELECTMODE();
      }
      
      %scene = scene.getCurrent();
      %map = scene.getCurrentMap();

      //*** Make sure that at least one brush is selected...
      if(select.getSelectionType() $= "SelectPrimitives")
      {
         if(%map.getNumSelectedBrushes() == 0 && %scene.getNumSelectedShapes() == 0)
         {
            tool.activateErrorMsg = "Please select one or more brushes or shapes for this tool to operate on.";
            return tool.FUNC_BADGENERAL();
         }
         
         tool.setUndoEntryLabel("Quantize: Brushes");
      }

      //*** ...or make sure that at least one vertex is selected
      if(select.getSelectionType() $= "SelectVertices")
      {
         if(%map.getNumSelectedVertices() == 0 && %scene.getNumSelectedShapes() == 0)
         {
            tool.activateErrorMsg = "Please select one or more vertices or shapes for this tool to operate on.";
            return tool.FUNC_BADGENERAL();
         }
         
         tool.setUndoEntryLabel("Quantize: Vertices");
      }
      
      //*** This tool doesn't need to make use of the instance
      %inst.instance = 0;
      %inst.flagsInterface = tool.IFLAG_NONE(); // No special interface flags as there will be no interface
      %inst.flagsApply = tool.AFLAG_NONE();     // No special actions when tool is applied
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Quantize: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // Execute()
   //
   // This function is called to perform the actual work of the tool, and does not
   // return a value.
   function Plugin::Execute(%this, %inst)
   {
      //error("Quantize: Execute(" @ %inst @ ")");

      %plugin = %inst.instance;

      %ac = pref.getInt("ToolActionCenter");
      if(%ac == 2)
      {
         %scene = scene.getCurrent();
         %map = scene.getCurrentMap();
         
         //*** Group
         if(select.getSelectionType() $= "SelectPrimitives")
         {
            %sel = %map.getSelectedBrushes();
            %count = getWordCount(%sel);
            for(%i=0; %i<%count; %i++)
            {
               %brushID = getWord(%sel, %i);
               %groupID = %map.getBrushGroup(%brushID);
               if(%groupID != -1)
               {
                  %selcenter = %scene.getGroupCenter(%groupID);
                  
               } else
               {
                  %selcenter = %map.getBrushPosition(%brushID);
               }
               
               %snapped = workplane.snapToGridForced(%selcenter);
               %move = VectorSub(%snapped, %selcenter);
               %map.moveBrush(%brushID, %move);
            }
            
         } else if(select.getSelectionType() $= "SelectVertices")
         {
            //*** Vertices don't have groups so treat as Origin
            Quantize_SnapVerticesAboutOrigin(%map);
         }
         
         //*** Shapes
         %sel = %scene.getSelectedShapes();
         %count = getWordCount(%sel);
         for(%i=0; %i<%count; %i++)
         {
            %shapeID = getWord(%sel, %i);
            %groupID = %scene.getShapeGroup(%shapeID);
            if(%groupID != -1)
            {
               %selcenter = %scene.getGroupCenter(%groupID);
                  
            } else
            {
               %selcenter = %scene.getShapePosition(%shapeID);
            }
               
            %snapped = workplane.snapToGridForced(%selcenter);
            %move = VectorSub(%snapped, %selcenter);
            %scene.moveShape(%shapeID, %move);
         }
         
      } else if(%ac == 3)
      {
         %scene = scene.getCurrent();
         %map = scene.getCurrentMap();
         
         //*** Origin
         if(select.getSelectionType() $= "SelectPrimitives")
         {
            %sel = %map.getSelectedBrushes();
            %count = getWordCount(%sel);
            for(%i=0; %i<%count; %i++)
            {
               %brushID = getWord(%sel, %i);
               %selcenter = %map.getBrushPosition(%brushID);
               %snapped = workplane.snapToGridForced(%selcenter);
               %move = VectorSub(%snapped, %selcenter);
               %map.moveBrush(%brushID, %move);
            }
         
         } else if(select.getSelectionType() $= "SelectVertices")
         {
            Quantize_SnapVerticesAboutOrigin(%map);
         }
         
         //*** Shapes
         %sel = %scene.getSelectedShapes();
         %count = getWordCount(%sel);
         for(%i=0; %i<%count; %i++)
         {
            %shapeID = getWord(%sel, %i);
            %selcenter = %scene.getShapePosition(%shapeID);
            %snapped = workplane.snapToGridForced(%selcenter);
            %move = VectorSub(%snapped, %selcenter);
            %scene.moveShape(%shapeID, %move);
         }
         
      } else
      {
         //*** Selection
         %selcenter = select.getSelectionCenter();
         %snapped = workplane.snapToGridForced(%selcenter);
         %move = VectorSub(%snapped, %selcenter);
         select.moveItemsRel(%move);
      }
   }
   
   function Quantize_SnapVerticesAboutOrigin(%map)
   {
      %sel = %map.getSelectedVertices(); // brushID SPC vertexID
      %count = getWordCount(%sel);
      for(%i=0; %i<%count; %i=%i+2)
      {
         %brushID = getWord(%sel, %i);
         %vertID = getWord(%sel, %i+1);

         %selcenter = %map.getVertexPosition(%brushID, %vertID);
               
         %snapped = workplane.snapToGridForced(%selcenter);
         %move = VectorSub(%snapped, %selcenter);
         %map.moveVertex(%brushID, %vertID, %move);
      }
   }
};

tool.register("Quantize", tool.typeGeneric(), tool.RFLAG_STORETRANSFORMS(), "Quantize" );
