//
// primitiveaddtube.cs
//
// Defines the PrimitiveAddTube tool plug-in to create a tube.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the primitive as a 3 element array
// center[] - Defines the xyz center point of the primitive as a 3 element array
// sides    - Defines the number of sides the base of the tube will have, min=3
// axis     - Defines the axis along which the tube will be built 0-2 = xyz
// xaxis    - Constant to do some tricks based on axis
// yaxis    - Constant to do some tricks based on axis
// zaxis    - Constant to do some tricks based on axis
// uscale	- Texture scale along the 'u' axis
// vscale	- Texture scale along the 'v' axis
// uoffset	- Texture offset along the 'u' axis
// voffset	- Texture offset along the 'v' axis
// texturestyle - Integer defining the method used to wrap the texture around the tube. 0=repeat, 1=map per face
//
// The tube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Revision History:
// May 30, 2005			David Wyand		Created script file
//

package PrimitiveAddTube
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("PrimitiveAddTube: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         PrimitiveAddTube_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %tube = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %tube.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::rebuildBoundingBox(%tube, %static);

      //*** Setup some additional attributes for the instance
      %tube.dirty = tool.DIRTY_NONE();
      %tube.active = false;
      %tube.update = tool.EDIT_DONOTHING();
      %tube.changeCenter = false;
      %tube.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %tube;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("PrimitiveAddTube: Done(" @ %inst @ "," @ %static @ ")");

      %tube = %inst.instance;
      
      if(%tube)
      {         
         //*** Delete our instance
         %tube.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("PrimitiveAddTube: MouseDown(" @ %inst @ "," @ %event @ ")");

      return false;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("PrimitiveAddTube: MouseMove(" @ %inst @ "," @ %event @ ")");
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("PrimitiveAddTube: MouseUp(" @ %inst @ "," @ %event @ ")");
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("PrimitiveAddTube: HandleCount(" @ %inst @ "," @ %event @ ")");

      %tube = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** toolPrimitiveHandleInit().  Otherwise, return the number of handles
      //*** the user may interact with.
      return %tube.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("PrimitiveAddTube: HandleInit(" @ %inst @ "," @ %event @ ")");

      %tube = %inst.instance;

      //*** Make the tool active
      if(!%tube.active)
      {
         %tube.active = true;
      }
      
      //*** Change the axis type based on the view the user has clicked in
      if(%event.view == 0)
      {
         %tube.static.axis = 0;
         //ToolBB::setType(%tube.static, ToolBB::typeCylinderX(), true);
         
      } else if(%event.view == 1)
      {
         %tube.static.axis = 1;
         //ToolBB::setType(%tube.static, ToolBB::typeCylinderY(), true);
         
      } else if(%event.view == 2)
      {
         %tube.static.axis = 2;
         //ToolBB::setType(%tube.static, ToolBB::typeCylinderZ(), true);
      }
      ToolBB::setType(%tube.static, ToolBB::typeDynamic(), true);
      
      return ToolBB::initHandles(%tube, %tube.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("PrimitiveAddTube: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %tube = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%tube, %tube.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddTube: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %tube = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%tube, %tube.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %tube.dirty = tool.DIRTY_APPEARANCE();
      %tube.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("PrimitiveAddTube: Dirty(" @ %inst @ ")");

      %tube = %inst.instance;
      
      return %tube.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("PrimitiveAddTube: Draw(" @ %inst @ "," @ %draw @ ")");

      %tube = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%tube.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%tube, %tube.static, %draw);

      //*** Indicate that we've drawn the tool
      %tube.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("PrimitiveAddTube: CheckEditAction(" @ %inst @ ")");

      %tube = %inst.instance;
      
      return %tube.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("PrimitiveAddTube: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %tube = %inst.instance;
      
      %tube.update = tool.EDIT_DONOTHING();
      %tube.active = false;
      %tube.changeSize = false;
      %tube.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("PrimitiveAddTube: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %tube = %inst.instance;
      
      // Work on the actual geometry.
      PrimitiveAddTube_MakeTubeGeometry(%tube, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %tube.update = tool.EDIT_DONOTHING();
      %tube.changeSize = false;
      %tube.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("PrimitiveAddTube: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %tube = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %tube.update = tool.EDIT_UPDATE();
            %tube.active = true;
            %tube.dirty = tool.DIRTY_APPEARANCE();
            %tube.changeSize = true;
            %tube.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            PrimitiveAddTube_ResetTube(%tube);
            %tube.update = tool.EDIT_UPDATE();
            %tube.active = true;
            %tube.dirty = tool.DIRTY_APPEARANCE();
            %tube.changeSize = true;
            %tube.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%tube.active)
            {
               %tube.update = tool.EDIT_REJECT();
            }
            %tube.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%tube.active)
            {
               %tube.changeSize = true;
               %tube.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("PrimitiveAddTube: Interface(" @ %inst @ "," @ %form @ ")");

      %tube = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Primitive: Tube");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      %form.addField( -1, "Geometry","divider");
      %form.addField( 2, "Axis"     ,"axis3");
      %form.addField( 3, "Sides"    ,"numericinteger");
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 8, ""         ,"popup"); // Texture style popup.  No label indicates the popup should fill the Tool Properties Form
      %form.addField( 4, "U Scale"  ,"numeric");
      %form.addField( 5, "V Scale"  ,"numeric");
      %form.addField( 6, "U Offset" ,"numeric");
      %form.addField( 7, "V Offset" ,"numeric");
      
      //*** Add items to the 'Style' popup
      %form.addFieldListItem(8, "Repeat wraparound");
      %form.addFieldListItem(8, "Per face mapping");
      
      //*** Set limits on some controls
      %form.setFieldMinLimit(3, 3);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("PrimitiveAddTube: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %tube = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %tube.static.center[0] SPC %tube.static.center[1] SPC %tube.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %tube.static.size[0] SPC %tube.static.size[1] SPC %tube.static.size[2];
            return %value;
      
         //*** Handle the 'Axis' field
         case 2:
            %value = %tube.static.axis;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %tube.static.sides;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 4:
            %value = %tube.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 5:
            %value = %tube.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 6:
            %value = %tube.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 7:
            %value = %tube.static.voffset;
            return %value;
      
         //*** Handle the 'Style' field
         case 8:
            %value = %tube.static.texturestyle;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("PrimitiveAddTube: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %tube = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %tube.static.center[%i] = getWord(%value, %i);
            }
            %tube.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %tube.static.size[%i] = getWord(%value, %i);
            }
            %tube.changeSize = true;
      
         //*** Handle the 'Axis' field
         case 2:
            %tube.static.axis = %value < 0 ? 0 :(%value > 2 ? 2 : %value);
            //if(%tube.static.axis == 0)
            //{
            //   ToolBB::setType(%tube.static, ToolBB::typeCylinderX(), true);
               
            //} else if(%tube.static.axis == 1)
            //{
            //   ToolBB::setType(%tube.static, ToolBB::typeCylinderY(), true);
               
            //} else
            //{
            //   ToolBB::setType(%tube.static, ToolBB::typeCylinderZ(), true);
            //}
            %tube.changeSize = true;
      
         //*** Handle the 'Sides' field
         case 3:
            %tube.static.sides = %value < 3 ? 3 :%value;
            %tube.changeSize = true;
      
         //*** Handle the 'U Scale' field
         case 4:
            %tube.static.uscale = %value;
            %tube.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 5:
            %tube.static.vscale = %value;
            %tube.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 6:
            %tube.static.uoffset = %value;
            %tube.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 7:
            %tube.static.voffset = %value;
            %tube.changeSize = true;
      
         //*** Handle the 'Style' field
         case 8:
            %tube.static.texturestyle = %value;
            %tube.changeSize = true;
      }

      //*** If we're not yet active, make it as if a EVENT_ACTIVATE has been received
      if(!%tube.active)
      {
         %tube.active = true;
         %tube.changeSize = true;
         %tube.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %tube.update = tool.EDIT_UPDATE();
      %tube.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%tube, %tube.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function PrimitiveAddTube_DefaultValues(%static)
   {
      %static.center[0] = 0.0; // x
      %static.center[1] = 0.0; // y
      %static.center[2] = 0.5; // z
      
      %static.size[0] = 1.0; // x
      %static.size[1] = 1.0; // y
      %static.size[2] = 1.0; // z
      
      %static.sides = 6;
      
      %static.axis = 2; // z-axis
      ToolBB::setType(%static, ToolBB::typeDynamic(), true);
      
      %static.uscale = 1.0;
      %static.vscale = 1.0;
      %static.uoffset = 0.0;
      %static.voffset = 0.0;
      
      %static.texturestyle = 0;
      
   }

   //*** Init the static object
   function PrimitiveAddTube_InitStatic(%static)
   {
      //*** Setup default values
      PrimitiveAddTube_DefaultValues(%static);
      
      //*** Setup our constants
      %static.xaxis = "1 2 0"; //"1 2 0";
      %static.yaxis = "2 0 1"; //"2 0 1";
      %static.zaxis = "0 1 2"; //"0 1 2";
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the tube instance to default values
   function PrimitiveAddTube_ResetTube(%inst)
   {
      //*** Setup default values
      PrimitiveAddTube_DefaultValues(%inst.static);
      ToolBB::rebuildBoundingBox(%inst, %inst.static);
   }
   
   //*** Build/modify the actual geometry
   function PrimitiveAddTube_MakeTubeGeometry(%inst, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%inst.active)
         return;
         
      //*** Check if there has already been a brush created.  If not, then we'll
      //*** build one here.
      %count = %edit.getEditBrushCount();
      if(%count == 0)
      {
         %brush = new MapBrush();
         
      } else
      {
         %brush = %edit.getEditBrush(0);
      }
   
      //*** Work on the sizing of the brush, if appropriate
      if(%inst.changeSize == true)
      {
         //*** Clear all planes on the brush to rebuild them.
         %brush.clearAllPlanes();

         //*** Use the sides entered by the user
         %sides = %inst.static.sides;
         
         //*** Build some constants
         %z = %inst.static.axis;
         %x = getWord(%inst.static.xaxis, %inst.static.axis);
         %y = getWord(%inst.static.yaxis, %inst.static.axis);
         %dt = 2.0 * 3.14159265358979323846 / %sides;
         %du = 1.0/(%sides); //*** Will wrap the texture around the tube once.
         
         for(%i=0; %i<3; %i++)
         {
            %halfsize[%i] = %inst.static.size[%i] * 0.5;
         }
         
         //*** Build the points for the bottom and top
         %pnts = new array();
         %theta = 0.0;
         for(%p=0; %p<%sides; %p++)
         {
            //*** Build the bottom point
//            %pt[%z] = %inst.static.center[%z] - %halfsize[%z];
//            %pt[%x] = %inst.static.center[%x] + %halfsize[%x] * mCos(%theta);
//            %pt[%y] = %inst.static.center[%y] + %halfsize[%y] * mSin(%theta);
            %pt[%z] = 0.0 - %halfsize[%z];
            %pt[%x] = %halfsize[%x] * mCos(%theta);
            %pt[%y] = %halfsize[%y] * mSin(%theta);
            %pnts.add(%p, %pt[0] SPC %pt[1] SPC %pt[2]);
            
            //*** Build the top point
//            %pt[%z] = %inst.static.center[%z] + %halfsize[%z];
            %pt[%z] = 0.0 + %halfsize[%z];
            %pnts.add(%p, %pt[0] SPC %pt[1] SPC %pt[2]);
            
            %theta += %dt;
         }

         //*** Calculate the length of each side
         %sidenum = 0;
         %sidelen[%sidenum] = 0.0;
         for(%i=0; %i<(%sides-2)*2+1; %i+=2)
         {
            %sidenum++;
            if(%inst.static.texturestyle == 1)
            {
               //*** Perform per face texture mapping.
               %sidelen[%sidenum] = 0.0;
               
            } else
            {
               //*** Calculate the cumulative side length
               %sidelen[%sidenum] = %sidelen[%sidenum-1] + VectorDist(%pnts.getValue(%i+2), %pnts.getValue(%i)) * %inst.static.uscale;
            }
         }
         %sidenum++;
         if(%inst.static.texturestyle == 1)
         {
            //*** Perform per face texture mapping.
            %sidelen[%sidenum] = 0.0;
               
         } else
         {
            %sidelen[%sidenum] = %sidelen[%sidenum-1] + VectorDist(%pnts.getValue(0), %pnts.getValue((%sides-1)*2)) * %inst.static.uscale;
         }
         
         //*** Build the tube's planes
         %center = %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2];
         %brush.setOrigin(%center);
         %brush.setTextureOffset(%inst.static.uoffset, %inst.static.voffset);
         %brush.setTextureScale(%inst.static.uscale, %inst.static.vscale);
         %sidenum = 0;
         for(%i=0; %i<(%sides-2)*2+1; %i+=2)
         {
//            %brush.addPlaneByPoints(%pnts.getValue(%i),%pnts.getValue(%i+1),%pnts.getValue(%i+3), %sidelen[%sidenum], 1.0, 0.0, 1.0);
//            %brush.addPlaneByPoints(%pnts.getValue(%i+2),%pnts.getValue(%i),%pnts.getValue(%i+1), %center, %sidelen[%sidenum], 1.0, 0.0, 1.0);
            %brush.setTexturePlanesByPoints2(%pnts.getValue(%i+2),%pnts.getValue(%i),%pnts.getValue(%i+1), %sidelen[%sidenum], 0.0);
            %brush.addPlaneByPoints(%pnts.getValue(%i+2),%pnts.getValue(%i),%pnts.getValue(%i+1));
            %sidenum++;
         }
//         %brush.addPlaneByPoints(%pnts.getValue((%sides-2)*2),%pnts.getValue((%sides-1)*2-1),%pnts.getValue(%sides*2-1));
//         %brush.addPlaneByPoints(%pnts.getValue((%sides-1)*2),%pnts.getValue(%sides*2-1),%pnts.getValue(1));
//         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue((%sides-1)*2),%pnts.getValue((%sides-1)*2+1), %center, %sidelen[%sidenum], 1.0, 0.0, 1.0);
         %brush.setTexturePlanesByPoints2(%pnts.getValue(0),%pnts.getValue((%sides-1)*2),%pnts.getValue((%sides-1)*2+1), %sidelen[%sidenum], 0.0);
         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue((%sides-1)*2),%pnts.getValue((%sides-1)*2+1));
         
         //*** Bottom cap
//         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue(2),%pnts.getValue((%sides-1)*2), %center, %sidelen[%sidenum], 1.0, 0.0, 1.0);
         %pt1[%z] = 0.0 - %halfsize[%z];
         %pt1[%x] = 0.0 - %halfsize[%x];
         %pt1[%y] = 0.0;
         %pt2[%z] = 0.0 - %halfsize[%z];
         %pt2[%x] = 0.0;
         %pt2[%y] = 0.0;
         %pt3[%z] = 0.0 - %halfsize[%z];
         %pt3[%x] = 0.0;
         %pt3[%y] = %halfsize[%y];
         %brush.setTexturePlanesByPointsCentered(%pt1[0] SPC %pt1[1] SPC %pt1[2], %pt2[0] SPC %pt2[1] SPC %pt2[2], %pt3[0] SPC %pt3[1] SPC %pt3[2], 0.5, 0.5);
         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue(2),%pnts.getValue((%sides-1)*2));
         
         //*** Top cap
//         %brush.addPlaneByPoints(%pnts.getValue((%sides*2)-1),%pnts.getValue(3),%pnts.getValue(1), %center, %sidelen[%sidenum], 1.0, 0.0, 1.0);
         %pt1[%z] = %halfsize[%z];
         %pt1[%x] = %halfsize[%x];
         %pt1[%y] = 0.0;
         %pt2[%z] = %halfsize[%z];
         %pt2[%x] = 0.0;
         %pt2[%y] = 0.0;
         %pt3[%z] = %halfsize[%z];
         %pt3[%x] = 0.0;
         %pt3[%y] = %halfsize[%y];
         %brush.setTexturePlanesByPointsCentered(%pt1[0] SPC %pt1[1] SPC %pt1[2], %pt2[0] SPC %pt2[1] SPC %pt2[2], %pt3[0] SPC %pt3[1] SPC %pt3[2], 0.5, 0.5);
         %brush.addPlaneByPoints(%pnts.getValue((%sides*2)-1),%pnts.getValue(3),%pnts.getValue(1));
         
         //*** Delete the points array
         %pnts.empty();
         %pnts.delete();
      }
         
      //*** Work on the position of the brush, if appropriate
      if(%inst.changeCenter == true)
      {
         %id = %brush.getBrushID();
         //*** Check if his brush has not yet been added to the map.  In this case, the building
         //*** of the planes using the change in size above will put it in the correct
         //*** location.
         if(%id != -1)
         {
            %edit.setBrushPosition(%id, %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2]);
         }
      }
         
      //*** Notify the edit operation of the new brush or change in an existing brush.
      %edit.updateBrush(%brush);
   }
   
};

tool.register("PrimitiveAddTube", tool.typeInteractive(), tool.RFLAG_SUPPORTONECLICKCREATE(), "Build Tube" );
