//
// primitiveaddsphere.cs
//
// Defines the PrimitiveAddSphere tool plug-in to create a sphere.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the primitive as a 3 element array
// center[] - Defines the xyz center point of the primitive as a 3 element array
// sides    - Defines the number of sides around the axis the sphere will have, min=3
// segments - Defines the number of segments along the axis, min=2
// axis     - Defines the axis along which the sphere will be built 0-2 = xyz
// xaxis    - Constant to do some tricks based on axis
// yaxis    - Constant to do some tricks based on axis
// zaxis    - Constant to do some tricks based on axis
// uscale	- Texture scale along the 'u' axis
// vscale	- Texture scale along the 'v' axis
// uoffset	- Texture offset along the 'u' axis
// voffset	- Texture offset along the 'v' axis
// texturestyle - Integer defining the method used to wrap the texture around the sphere. 0=repeat, 1=map per face, 2=size to fit
//
// The sphere instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
// draghandle   - Bool indicating if the user is dragging a size handle.  Use to draw the wireframe sphere.
//
// Revision History:
// May 27, 2005			David Wyand		Created script file
//

package PrimitiveAddSphere
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("PrimitiveAddSphere: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         PrimitiveAddSphere_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %sphere = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %sphere.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::rebuildBoundingBox(%sphere, %static);

      //*** Setup some additional attributes for the instance
      %sphere.dirty = tool.DIRTY_NONE();
      %sphere.active = false;
      %sphere.update = tool.EDIT_DONOTHING();
      %sphere.changeCenter = false;
      %sphere.changeSize = false;
      %sphere.draghandle = false;
      
      //*** Pass along the instance
      %inst.instance = %sphere;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("PrimitiveAddSphere: Done(" @ %inst @ "," @ %static @ ")");

      %sphere = %inst.instance;
      
      if(%sphere)
      {         
         //*** Delete our instance
         %sphere.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("PrimitiveAddSphere: MouseDown(" @ %inst @ "," @ %event @ ")");

      return false;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("PrimitiveAddSphere: MouseMove(" @ %inst @ "," @ %event @ ")");
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("PrimitiveAddSphere: MouseUp(" @ %inst @ "," @ %event @ ")");
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("PrimitiveAddSphere: HandleCount(" @ %inst @ "," @ %event @ ")");

      %sphere = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** toolPrimitiveHandleInit().  Otherwise, return the number of handles
      //*** the user may interact with.
      return %sphere.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("PrimitiveAddSphere: HandleInit(" @ %inst @ "," @ %event @ ")");

      %sphere = %inst.instance;

      //*** Make the tool active
      if(!%sphere.active)
      {
         %sphere.active = true;
      }
      
      ToolBB::setType(%sphere.static, ToolBB::typeDynamic(), true);
      
      return ToolBB::initHandles(%sphere, %sphere.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("PrimitiveAddSphere: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %sphere = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%sphere, %sphere.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddSphere: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %sphere = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%sphere, %sphere.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin
      %sphere.dirty = tool.DIRTY_APPEARANCE();

      //*** Only update the geometry if the center handle is used
      if(%hindex == 15)
      {
         %sphere.update = tool.EDIT_UPDATE();

      } else
      {
         %sphere.update = tool.EDIT_DONOTHING();
         %sphere.draghandle = true;
      }
      
      return %returnHandle;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddSphere: HandleDone(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %sphere = %inst.instance;
      
      //*** Perform an update with the initial handle or a sizing handle
      if(%hindex == 0 || %hindex >= 1 && %hindex < 15)
      {
         %sphere.draghandle = false;
         %sphere.update = tool.EDIT_UPDATE();
         %sphere.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("PrimitiveAddSphere: Dirty(" @ %inst @ ")");

      %sphere = %inst.instance;
      
      return %sphere.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("PrimitiveAddSphere: Draw(" @ %inst @ "," @ %draw @ ")");

      %sphere = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%sphere.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%sphere, %sphere.static, %draw);

      if(%sphere.draghandle)
         PrimitiveAddSphere_DrawSphereGeometry(%sphere, %draw);

      //*** Indicate that we've drawn the tool
      %sphere.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("PrimitiveAddSphere: CheckEditAction(" @ %inst @ ")");

      %sphere = %inst.instance;
      
      return %sphere.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("PrimitiveAddSphere: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %sphere = %inst.instance;
      
      %sphere.update = tool.EDIT_DONOTHING();
      %sphere.active = false;
      %sphere.changeSize = false;
      %sphere.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("PrimitiveAddSphere: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %sphere = %inst.instance;
      
      // Work on the actual geometry.
      PrimitiveAddSphere_MakeSphereGeometry(%sphere, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %sphere.update = tool.EDIT_DONOTHING();
      %sphere.changeSize = false;
      %sphere.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("PrimitiveAddSphere: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %sphere = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %sphere.update = tool.EDIT_UPDATE();
            %sphere.active = true;
            %sphere.dirty = tool.DIRTY_APPEARANCE();
            %sphere.changeSize = true;
            %sphere.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            PrimitiveAddSphere_ResetSphere(%sphere);
            %sphere.update = tool.EDIT_UPDATE();
            %sphere.active = true;
            %sphere.dirty = tool.DIRTY_APPEARANCE();
            %sphere.changeSize = true;
            %sphere.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%sphere.active)
            {
               %sphere.update = tool.EDIT_REJECT();
            }
            %sphere.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%sphere.active)
            {
               %sphere.changeSize = true;
               %sphere.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("PrimitiveAddSphere: Interface(" @ %inst @ "," @ %form @ ")");

      %sphere = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Primitive: Sphere");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      %form.addField( -1, "Geometry","divider");
      %form.addField( 2, "Axis"     ,"axis3");
      %form.addField( 3, "Sides"    ,"numericinteger");
      %form.addField( 4, "Segments" ,"numericinteger");
      %form.addField( -1, "Texturing" ,"divider");
//      %form.addField( 5, ""         ,"popup"); // Texture style popup.  No label indicates the popup should fill the Tool Properties Form
      %form.addField( 6, "U Scale"  ,"numeric");
      %form.addField( 7, "V Scale"  ,"numeric");
      %form.addField( 8, "U Offset" ,"numeric");
      %form.addField( 9, "V Offset" ,"numeric");
      
      //*** Add items to the 'Style' popup
//      %form.addFieldListItem(5, "Repeat wraparound");
//      %form.addFieldListItem(5, "Per face mapping");
//      %form.addFieldListItem(5, "Size to fit");
      
      //*** Set limits on some controls
      %form.setFieldMinLimit(3, 3);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("PrimitiveAddSphere: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %sphere = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %sphere.static.center[0] SPC %sphere.static.center[1] SPC %sphere.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %sphere.static.size[0] SPC %sphere.static.size[1] SPC %sphere.static.size[2];
            return %value;
      
         //*** Handle the 'Axis' field
         case 2:
            %value = %sphere.static.axis;
            return %value;
      
         //*** Handle the 'Sides' field
         case 3:
            %value = %sphere.static.sides;
            return %value;
      
         //*** Handle the 'Segments' field
         case 4:
            %value = %sphere.static.segments;
            return %value;
      
         //*** Handle the 'Style' field
         case 5:
            %value = %sphere.static.texturestyle;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 6:
            %value = %sphere.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 7:
            %value = %sphere.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 8:
            %value = %sphere.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 9:
            %value = %sphere.static.voffset;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("PrimitiveAddSphere: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %sphere = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %sphere.static.center[%i] = getWord(%value, %i);
            }
            %sphere.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %sphere.static.size[%i] = getWord(%value, %i);
            }
            %sphere.changeSize = true;
      
         //*** Handle the 'Axis' field
         case 2:
            %sphere.static.axis = %value < 0 ? 0 :(%value > 2 ? 2 : %value);
            %sphere.changeSize = true;
      
         //*** Handle the 'Sides' field
         case 3:
            %sphere.static.sides = %value < 3 ? 3 :%value;
            %sphere.changeSize = true;
      
         //*** Handle the 'Segments' field
         case 4:
            %sphere.static.segments = %value < 2 ? 2 :%value;
            %sphere.changeSize = true;
      
         //*** Handle the 'Style' field
         case 5:
            %sphere.static.texturestyle = %value;
            %sphere.changeSize = true;
      
         //*** Handle the 'U Scale' field
         case 6:
            %sphere.static.uscale = %value;
            %sphere.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 7:
            %sphere.static.vscale = %value;
            %sphere.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 8:
            %sphere.static.uoffset = %value;
            %sphere.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 9:
            %sphere.static.voffset = %value;
            %sphere.changeSize = true;
      }

      //*** If we're not yet active, make it as if a EVENT_ACTIVATE has been received
      if(!%sphere.active)
      {
         %sphere.active = true;
         %sphere.changeSize = true;
         %sphere.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %sphere.update = tool.EDIT_UPDATE();
      %sphere.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%sphere, %sphere.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function PrimitiveAddSphere_DefaultValues(%static)
   {
      %static.center[0] = 0.0; // x
      %static.center[1] = 0.0; // y
      %static.center[2] = 0.0; // z
      
      %static.size[0] = 1.0; // x
      %static.size[1] = 1.0; // y
      %static.size[2] = 1.0; // z
      
      %static.sides = 12;
      %static.segments = 6;
      
      %static.axis = 2; // z-axis
      ToolBB::setType(%static, ToolBB::typeDynamic(), true);
      
      %static.uscale = 1.0;
      %static.vscale = 1.0;
      %static.uoffset = 0.0;
      %static.voffset = 0.0;
      
      %static.texturestyle = 1;
      
   }

   //*** Init the static object
   function PrimitiveAddSphere_InitStatic(%static)
   {
      //*** Setup default values
      PrimitiveAddSphere_DefaultValues(%static);
      
      //*** Setup our constants
      %static.xaxis = "1 2 0"; //"2 1 0";
      %static.yaxis = "2 0 1"; //"0 2 1";
      %static.zaxis = "0 1 2"; //"1 0 2";
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the sphere instance to default values
   function PrimitiveAddSphere_ResetSphere(%inst)
   {
      //*** Setup default values
      PrimitiveAddSphere_DefaultValues(%inst.static);
      ToolBB::rebuildBoundingBox(%inst, %inst.static);
   }
      
   //*** Draw a wireframe sphere
   function PrimitiveAddSphere_DrawSphereGeometry(%inst, %draw)
   {
      if(!%inst.active)
         return;

      %draw.moveTo(%draw.COORD_ABSOLUTE(), %inst.static.center[0], %inst.static.center[1], %inst.static.center[2]);

      %x = getWord(%inst.static.xaxis, %inst.static.axis);
      %y = getWord(%inst.static.yaxis, %inst.static.axis);
      %z = getWord(%inst.static.zaxis, %inst.static.axis);

      %size = %inst.static.size[0] SPC %inst.static.size[1] SPC %inst.static.size[2];
      %draw.buildSphere(%draw.POLYGON_OUTLINE()+%draw.COORD_RELATIVE(), %size, %inst.static.segments, %inst.static.sides, %x, %y, %z);
   }

   //*** Build/modify the actual geometry
   function PrimitiveAddSphere_MakeSphereGeometry(%inst, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%inst.active)
         return;
         
      //*** Check if there has already been a brush created.  If not, then we'll
      //*** build one here.
      %count = %edit.getEditBrushCount();
      if(%count == 0)
      {
         %brush = new MapBrush();
         
      } else
      {
         %brush = %edit.getEditBrush(0);
      }
   
      //*** Work on the sizing of the brush, if appropriate
      if(%inst.changeSize == true)
      {
         //*** Clear all planes on the brush to rebuild them.
         %brush.clearAllPlanes();
         
         //*** Build some constants
         %x = getWord(%inst.static.xaxis, %inst.static.axis);
         %y = getWord(%inst.static.yaxis, %inst.static.axis);
         %z = getWord(%inst.static.zaxis, %inst.static.axis);
         %pi = 3.14159265358979323846;
         %dt = %pi / %inst.static.segments;
         %ds = 2.0 * %pi / %inst.static.sides;
         
         for(%i=0; %i<3; %i++)
         {
            %halfsize[%i] = %inst.static.size[%i] * 0.5;
         }

         //*** Set up the brush
         %center = %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2];
         %brush.setOrigin(%center);
         %brush.setTextureOffset(%inst.static.uoffset, %inst.static.voffset);
         %brush.setTextureScale(%inst.static.uscale, %inst.static.vscale);

         for(%i = 0; %i<%inst.static.segments; %i++)
         {
            %angleb = %i * %dt - %pi/2;
            %anglet = (%i+1) * %dt - %pi/2;

            for(%j=0; %j<%inst.static.sides; %j++)
            {
               %angle1 = %j * %ds;
               %angle2 = (%j+1) * %ds;

               %p1[%x] = %halfsize[%x] * mCos(%anglet) * mCos(%angle2);
               %p1[%y] = %halfsize[%y] * mCos(%anglet) * mSin(%angle2);
               %p1[%z] = %halfsize[%z] * mSin(%anglet);

               %p2[%x] = %halfsize[%x] * mCos(%angleb) * mCos(%angle2);
               %p2[%y] = %halfsize[%y] * mCos(%angleb) * mSin(%angle2);
               %p2[%z] = %halfsize[%z] * mSin(%angleb);

               if(%i >= %inst.static.segments/2)
               {
                  %p3[%x] = %halfsize[%x] * mCos(%angleb) * mCos(%angle1);
                  %p3[%y] = %halfsize[%y] * mCos(%angleb) * mSin(%angle1);
                  %p3[%z] = %halfsize[%z] * mSin(%angleb);

                  //*** Texturing
                  switch(%inst.static.texturestyle)
                  {
                     case 0:
                        //*** Repeat mapping

                     case 1:
                        //*** Per face mapping (like Quark)
                        %pnt[%x] = %p3[%x];
                        %pnt[%y] = %p3[%y];
                        %pnt[%z] = %p3[%z] + %halfsize[%z];
                        %tp = %brush.projectPointOnPlane(%p1[0] SPC %p1[1] SPC %p1[2], %p2[0] SPC %p2[1] SPC %p2[2], %p3[0] SPC %p3[1] SPC %p3[2], %pnt[0] SPC %pnt[1] SPC %pnt[2]);
                        %brush.setTexturePlanesByPoints2(%p2[0] SPC %p2[1] SPC %p2[2], %p3[0] SPC %p3[1] SPC %p3[2], %tp, 0.0, 0.0);

                     case 2:
                        //*** Size to fit
                  }

                  %brush.addPlaneByPoints(%p1[0] SPC %p1[1] SPC %p1[2], %p2[0] SPC %p2[1] SPC %p2[2], %p3[0] SPC %p3[1] SPC %p3[2]);

               } else
               {
                  %p3[%x] = %halfsize[%x] * mCos(%anglet) * mCos(%angle1);
                  %p3[%y] = %halfsize[%y] * mCos(%anglet) * mSin(%angle1);
                  %p3[%z] = %halfsize[%z] * mSin(%anglet);

                  //*** Texturing
                  switch(%inst.static.texturestyle)
                  {
                     case 0:
                        //*** Repeat mapping

                     case 1:
                        //*** Per face mapping (like Quark)
                        %p = NearestPointOnLine(%p2[0] SPC %p2[1] SPC %p2[2], %p3[0] SPC %p3[1] SPC %p3[2], %p1[0] SPC %p1[1] SPC %p1[2]);
                        %vec = VectorSub(%p2[0] SPC %p2[1] SPC %p2[2], %p);
                        %tp1 = VectorAdd(%p1[0] SPC %p1[1] SPC %p1[2], %vec);
                        %tp2 = VectorAdd(%p3[0] SPC %p3[1] SPC %p3[2], %vec);
                        %tp3 = VectorAdd(%tp2, %vec);
                        %brush.setTexturePlanesByPoints2(%tp1, %tp2, %p3[0] SPC %p3[1] SPC %p3[2], 0.0, 0.0);

                     case 2:
                        //*** Size to fit
                  }

                  %brush.addPlaneByPoints(%p3[0] SPC %p3[1] SPC %p3[2], %p1[0] SPC %p1[1] SPC %p1[2], %p2[0] SPC %p2[1] SPC %p2[2]);
               }
            }
         }

         //*** Update the brush
         %edit.updateBrush(%brush);

         //*** Make sure the center is also updated
         %inst.changeCenter = true;
      }
         
      //*** Work on the position of the brush, if appropriate
      if(%inst.changeCenter == true)
      {
         %id = %brush.getBrushID();
         //*** Check if his brush has not yet been added to the map.  In this case, the building
         //*** of the planes using the change in size above will put it in the correct
         //*** location.
         if(%id != -1)
         {
            %edit.setBrushPosition(%id, %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2]);
            %edit.updateBrush(%brush);
         }
      }
   }
   
};

tool.register("PrimitiveAddSphere", tool.typeInteractive(), tool.RFLAG_SUPPORTONECLICKCREATE(), "Build Sphere" );
