//
// primitiveaddramp.cs
//
// Defines the PrimitiveAddRamp tool plug-in to create a ramp.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the primitive as a 3 element array
// center[] - Defines the xyz center point of the primitive as a 3 element array
// axis     - Defines the axis along which the ramp will be built 0-2 = xyz
// xaxis    - Constant to do some tricks based on axis
// yaxis    - Constant to do some tricks based on axis
// zaxis    - Constant to do some tricks based on axis
// uscale	- Texture scale along the 'u' axis
// vscale	- Texture scale along the 'v' axis
// uoffset	- Texture offset along the 'u' axis
// voffset	- Texture offset along the 'v' axis
// textureorigin - The texture's origin. 0=corner 1=centered
//
// The ramp instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xzy center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
//
// Revision History:
// May 27, 2005			David Wyand		Created script file
//

package PrimitiveAddRamp
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("PrimitiveAddRamp: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         PrimitiveAddRamp_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %ramp = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %ramp.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeDynamic(), true);
      ToolBB::rebuildBoundingBox(%ramp, %static);

      //*** Setup some additional attributes for the instance
      %ramp.dirty = tool.DIRTY_NONE();
      %ramp.active = false;
      %ramp.update = tool.EDIT_DONOTHING();
      %ramp.changeCenter = false;
      %ramp.changeSize = false;
      
      //*** Pass along the instance
      %inst.instance = %ramp;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("PrimitiveAddRamp: Done(" @ %inst @ "," @ %static @ ")");

      %ramp = %inst.instance;
      
      if(%ramp)
      {         
         //*** Delete our instance
         %ramp.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("PrimitiveAddRamp: MouseDown(" @ %inst @ "," @ %event @ ")");

      return false;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("PrimitiveAddRamp: MouseMove(" @ %inst @ "," @ %event @ ")");
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't
   // return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("PrimitiveAddRamp: MouseUp(" @ %inst @ "," @ %event @ ")");
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("PrimitiveAddRamp: HandleCount(" @ %inst @ "," @ %event @ ")");

      %ramp = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** toolPrimitiveHandleInit().  Otherwise, return the number of handles
      //*** the user may interact with.
      return %ramp.active ? ToolBB::getHandleCount() : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("PrimitiveAddRamp: HandleInit(" @ %inst @ "," @ %event @ ")");

      %ramp = %inst.instance;

      //*** Make the tool active
      if(!%ramp.active)
      {
         %ramp.active = true;
      }
      
      //*** Change the axis type based on the view the user has clicked in
      if(%event.view == 0)
      {
         %ramp.static.axis = 0;
         
      } else if(%event.view == 1)
      {
         %ramp.static.axis = 1;
         
      } else if(%event.view == 2)
      {
         %ramp.static.axis = 2;
      }
      
      return ToolBB::initHandles(%ramp, %ramp.static, %event);
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("PrimitiveAddRamp: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %ramp = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      return ToolBB::getHandle(%ramp, %ramp.static, %event, %hindex, %info);
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddRamp: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %ramp = %inst.instance;
      
      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%ramp, %ramp.static, %event, %hindex);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %ramp.dirty = tool.DIRTY_APPEARANCE();
      %ramp.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("PrimitiveAddRamp: Dirty(" @ %inst @ ")");

      %ramp = %inst.instance;
      
      return %ramp.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("PrimitiveAddRamp: Draw(" @ %inst @ "," @ %draw @ ")");

      %ramp = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%ramp.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%ramp, %ramp.static, %draw);

      //*** Indicate that we've drawn the tool
      %ramp.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("PrimitiveAddRamp: CheckEditAction(" @ %inst @ ")");

      %ramp = %inst.instance;
      
      return %ramp.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("PrimitiveAddRamp: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %ramp = %inst.instance;
      
      %ramp.update = tool.EDIT_DONOTHING();
      %ramp.active = false;
      %ramp.changeSize = false;
      %ramp.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("PrimitiveAddRamp: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %ramp = %inst.instance;
      
      // Work on the actual geometry.
      PrimitiveAddRamp_MakeRampGeometry(%ramp, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %ramp.update = tool.EDIT_DONOTHING();
      %ramp.changeSize = false;
      %ramp.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("PrimitiveAddRamp: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %ramp = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %ramp.update = tool.EDIT_UPDATE();
            %ramp.active = true;
            %ramp.dirty = tool.DIRTY_APPEARANCE();
            %ramp.changeSize = true;
            %ramp.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            PrimitiveAddRamp_ResetRamp(%ramp);
            %ramp.update = tool.EDIT_UPDATE();
            %ramp.active = true;
            %ramp.dirty = tool.DIRTY_APPEARANCE();
            %ramp.changeSize = true;
            %ramp.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%ramp.active)
            {
               %ramp.update = tool.EDIT_REJECT();
            }
            %ramp.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%ramp.active)
            {
               %ramp.changeSize = true;
               %ramp.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("PrimitiveAddRamp: Interface(" @ %inst @ "," @ %form @ ")");

      %ramp = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Primitive: Ramp");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      %form.addField( -1, "Geometry","divider");
      %form.addField( 2, "Axis"     ,"axis3");
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 3, "Origin"  ,"popup");
      %form.addField( 4, "U Scale"  ,"numeric");
      %form.addField( 5, "V Scale"  ,"numeric");
      %form.addField( 6, "U Offset" ,"numeric");
      %form.addField( 7, "V Offset" ,"numeric");
      
      //*** TEMP FOR DEBUGGING
      %form.addField( 100, "Debug" ,"checkbox");
      
      //*** Add items to the 'Origin' popup
      %form.addFieldListItem(3, "Corner");
      %form.addFieldListItem(3, "Center");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("PrimitiveAddRamp: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %ramp = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %ramp.static.center[0] SPC %ramp.static.center[1] SPC %ramp.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %ramp.static.size[0] SPC %ramp.static.size[1] SPC %ramp.static.size[2];
            return %value;
      
         //*** Handle the 'Axis' field
         case 2:
            %value = %ramp.static.axis;
            return %value;
      
         //*** Handle the 'Origin' field
         case 3:
            %value = %ramp.static.textureorigin;
            return %value;
      
         //*** Handle the 'U Scale' field
         case 4:
            %value = %ramp.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 5:
            %value = %ramp.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 6:
            %value = %ramp.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 7:
            %value = %ramp.static.voffset;
            return %value;
      
         //*** TEMP FOR DEBUGGING
         case 100:
            %value = %ramp.static.debugrender;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("PrimitiveAddRamp: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %ramp = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %ramp.static.center[%i] = getWord(%value, %i);
            }
            %ramp.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %ramp.static.size[%i] = getWord(%value, %i);
            }
            %ramp.changeSize = true;
      
         //*** Handle the 'Axis' field
         case 2:
            %ramp.static.axis = %value < 0 ? 0 :(%value > 2 ? 2 : %value);
            %ramp.changeSize = true;
      
         //*** Handle the 'Origin' field
         case 3:
            %ramp.static.textureorigin = %value;
            %ramp.changeSize = true;
      
         //*** Handle the 'U Scale' field
         case 4:
            %ramp.static.uscale = %value;
            %ramp.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 5:
            %ramp.static.vscale = %value;
            %ramp.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 6:
            %ramp.static.uoffset = %value;
            %ramp.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 7:
            %ramp.static.voffset = %value;
            %ramp.changeSize = true;
      
         //*** TEMP FOR DEBUGGING
         case 100:
            %ramp.static.debugrender = %value;
            %ramp.changeSize = true; // Piggy-back the resize flag.  It's kind of a resize! :o)
      }

      //*** If we're not yet active, make it as if a EVENT_ACTIVATE has been received
      if(!%ramp.active)
      {
         %ramp.active = true;
         %ramp.changeSize = true;
         %ramp.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %ramp.update = tool.EDIT_UPDATE();
      %ramp.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%ramp, %ramp.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function PrimitiveAddRamp_DefaultValues(%static)
   {
      %static.center[0] = 0.0; // x
      %static.center[1] = 0.0; // y
      %static.center[2] = 0.5; // z
      
      %static.size[0] = 1.0; // x
      %static.size[1] = 1.0; // y
      %static.size[2] = 1.0; // z
      
      %static.axis = 2; // z-axis
      
   }

   //*** Init the static object
   function PrimitiveAddRamp_InitStatic(%static)
   {
      //*** Setup default values
      PrimitiveAddRamp_DefaultValues(%static);
      
      //*** Setup our constants
      %static.xaxis = "1 2 0";
      %static.yaxis = "2 0 1";
      %static.zaxis = "0 1 2";
      
      //*** Signal we're all setup.
      %static.init = true;
      
      %static.uscale = 1.0;
      %static.vscale = 1.0;
      %static.uoffset = 0.0;
      %static.voffset = 0.0;
      
      %static.textureorigin = 0;
      
      //*** CREATE THE DEBUG BRUSH BUILDER
      %static.debugrender = 0;
   }
   
   //*** Reset the ramp instance to default values
   function PrimitiveAddRamp_ResetRamp(%inst)
   {
      //*** Setup default values
      PrimitiveAddRamp_DefaultValues(%inst.static);
      ToolBB::rebuildBoundingBox(%inst, %inst.static);
   }
   
   //*** Build/modify the actual geometry
   function PrimitiveAddRamp_MakeRampGeometry(%inst, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%inst.active)
         return;
         
      //*** Check if there has already been a brush created.  If not, then we'll
      //*** build one here.
      %count = %edit.getEditBrushCount();
      if(%count == 0)
      {
         %brush = new MapBrush();
         
      } else
      {
         %brush = %edit.getEditBrush(0);
      }
   
      //*** Work on the sizing of the brush, if appropriate
      if(%inst.changeSize == true)
      {
         //*** Clear all planes on the brush to rebuild them.
         %brush.clearAllPlanes();
         
         //*** Build some constants
         %z = %inst.static.axis;
         %x = getWord(%inst.static.xaxis, %inst.static.axis);
         %y = getWord(%inst.static.yaxis, %inst.static.axis);
         %dt = 2.0 * 3.14159265358979323846 / %sides;
         
         for(%i=0; %i<3; %i++)
         {
            %halfsize[%i] = %inst.static.size[%i] * 0.5;
         }
         
         //*** Setup geometry and texture states
         %center = %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2];
         %brush.setOrigin(%center);
         %brush.setTextureOffset(%inst.static.uoffset, %inst.static.voffset);
         %brush.setTextureScale(%inst.static.uscale, %inst.static.vscale);
         
         //*** Build the points for the base.  There's only eight, so build them manually.
         %pnts = new array();
         
         %pt[%x] = 0.0 - %halfsize[%x];
         %pt[%y] = 0.0 - %halfsize[%y];
         %pt[%z] = 0.0 - %halfsize[%z];
         %pnts.add(0, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x -y -z 0
         
         %pt[%x] = %halfsize[%x];
         %pnts.add(1, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x -y -z 1

         %pt[%y] = %halfsize[%y];
         %pnts.add(2, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x +y -z 2

         %pt[%x] = 0.0 - %halfsize[%x];
         %pnts.add(3, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x +y -z 3

         %pt[%y] = 0.0 - %halfsize[%y];
         %pt[%z] = %halfsize[%z];
         %pnts.add(4, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x -y +z 4

         %pt[%x] = %halfsize[%x];
         %pnts.add(5, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x -y +z 5

         %pt[%y] = %halfsize[%y];
         %pnts.add(6, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x +y +z 6

         %pt[%x] = 0.0 - %halfsize[%x];
         %pnts.add(7, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x +y +z 7

         //*** Setup some texture coordinates.
         %tx = %halfsize[0];
         %ty = %halfsize[1];
         %tz = %halfsize[2];
         
         //*** Build a points array for the centered texture
         if(%inst.static.textureorigin == 1)
         {
            %cpnts = new array();
            
            %pt[%x] = 0.0 - %halfsize[%x];
            %pt[%y] = 0.0 - %halfsize[%y];
            %pt[%z] = 0.0;
            %cpnts.add(0, %pt[0] SPC %pt[1] SPC %pt[2]);  // +y plane 0
            %pt[%x] = 0.0;
            %cpnts.add(1, %pt[0] SPC %pt[1] SPC %pt[2]);  // +y plane 1
            %pt[%z] = %halfsize[%z];
            %cpnts.add(2, %pt[0] SPC %pt[1] SPC %pt[2]);  // +y plane 2

            %pt[%x] = %halfsize[%x];
            %pt[%y] = %halfsize[%y];
            %pt[%z] = 0.0;
            %cpnts.add(3, %pt[0] SPC %pt[1] SPC %pt[2]);  // -y plane 3
            %pt[%x] = 0.0;
            %cpnts.add(4, %pt[0] SPC %pt[1] SPC %pt[2]);  // -y plane 4
            %pt[%z] = %halfsize[%z];
            %cpnts.add(5, %pt[0] SPC %pt[1] SPC %pt[2]);  // -y plane 5

            %pt[%x] = 0.0 - %halfsize[%x];
            %pt[%y] = %halfsize[%y];
            %pt[%z] = 0.0;
            %cpnts.add(6, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x plane 6
            %pt[%y] = 0.0;
            %cpnts.add(7, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x plane 7
            %pt[%z] = %halfsize[%z];
            %cpnts.add(8, %pt[0] SPC %pt[1] SPC %pt[2]);  // +x plane 8

            %pt[%x] = %halfsize[%x];
            %pt[%y] = 0.0 - %halfsize[%y];
            %pt[%z] = 0.0;
            %cpnts.add(9, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x plane 9
            %pt[%y] = 0.0;
            %cpnts.add(10, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x plane 10
            %pt[%z] = %halfsize[%z];
            %cpnts.add(11, %pt[0] SPC %pt[1] SPC %pt[2]);  // -x plane 11

            %pt[%x] = 0.0 - %halfsize[%x];
            %pt[%y] = 0.0;
            %pt[%z] = 0.0 - %halfsize[%z];
            %cpnts.add(12, %pt[0] SPC %pt[1] SPC %pt[2]);  // +z plane 12
            %pt[%x] = 0.0;
            %cpnts.add(13, %pt[0] SPC %pt[1] SPC %pt[2]);  // +z plane 13
            %pt[%y] = %halfsize[%y];
            %cpnts.add(14, %pt[0] SPC %pt[1] SPC %pt[2]);  // +z plane 14
            
            %pt[%x] = %halfsize[%x];
            %pt[%y] = 0.0;
            %pt[%z] = %halfsize[%z];
            %cpnts.add(15, %pt[0] SPC %pt[1] SPC %pt[2]);  // -z plane 15
            %pt[%x] = 0.0;
            %cpnts.add(16, %pt[0] SPC %pt[1] SPC %pt[2]);  // -z plane 16
            %pt[%y] = 0.0 - %halfsize[%y];
            %cpnts.add(17, %pt[0] SPC %pt[1] SPC %pt[2]);  // -z plane 17
            
            %pt[%x] = 0.0;
            %pt[%y] = %halfsize[%y];
            %pt[%z] = 0.0;
            %cpnts.add(18, %pt[0] SPC %pt[1] SPC %pt[2]);  // ramp 18
            %pt[%y] = 0.0;
            %cpnts.add(19, %pt[0] SPC %pt[1] SPC %pt[2]);  // ramp 19
            %pt[%x] = 0.0 - %halfsize[%x];
            %pt[%z] = %halfsize[%z];
            %cpnts.add(20, %pt[0] SPC %pt[1] SPC %pt[2]);  // ramp 20
         }

         
         //*** Build the ramp's planes.

         // Bottom
         if(%inst.static.textureorigin == 0)
         {
//            %brush.setTexturePlanesByPoints(-(%tx) SPC -(%ty) SPC -(%tz), %tx SPC -(%ty) SPC -(%tz), %tx SPC %ty SPC -(%tz), 0.0, 0.0);
            %brush.setTexturePlanesByPoints2(%pnts.getValue(0),%pnts.getValue(1),%pnts.getValue(2), 0.0, 0.0);
               
         } else
         {
            %brush.setTexturePlanesByPointsCentered(%cpnts.getValue(12),%cpnts.getValue(13),%cpnts.getValue(14), 0.5, 0.5);
         }
         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue(1),%pnts.getValue(2));

         // Front
         if(%inst.static.textureorigin == 0)
         {
            %brush.setTexturePlanesByPoints2(%pnts.getValue(1),%pnts.getValue(0),%pnts.getValue(4), 0.0, 0.0);

         } else
         {
            %brush.setTexturePlanesByPointsCentered(%cpnts.getValue(3),%cpnts.getValue(4),%cpnts.getValue(5), 0.5, 0.5);
         }
         %brush.addPlaneByPoints(%pnts.getValue(1),%pnts.getValue(0),%pnts.getValue(4));
         
         // Back
         if(%inst.static.textureorigin == 0)
         {
            %brush.setTexturePlanesByPoints2(%pnts.getValue(3),%pnts.getValue(2),%pnts.getValue(6), 0.0, 0.0);

         } else
         {
            %brush.setTexturePlanesByPointsCentered(%cpnts.getValue(0),%cpnts.getValue(1),%cpnts.getValue(2), 0.5, 0.5);
         }
         %brush.addPlaneByPoints(%pnts.getValue(3),%pnts.getValue(2),%pnts.getValue(6));
         
         // Left
         if(%inst.static.textureorigin == 0)
         {
            %brush.setTexturePlanesByPoints2(%pnts.getValue(0),%pnts.getValue(3),%pnts.getValue(7), 0.0, 0.0);

         } else
         {
            %brush.setTexturePlanesByPointsCentered(%cpnts.getValue(9),%cpnts.getValue(10),%cpnts.getValue(11), 0.5, 0.5);
         }
         %brush.addPlaneByPoints(%pnts.getValue(0),%pnts.getValue(3),%pnts.getValue(7));
         
         // Ramp
         if(%inst.static.textureorigin == 0)
         {
            %brush.setTexturePlanesByPoints2(%pnts.getValue(2),%pnts.getValue(1),%pnts.getValue(4), 0.0, 0.0);

         } else
         {
            %brush.setTexturePlanesByPointsCentered(%cpnts.getValue(18),%cpnts.getValue(19),%cpnts.getValue(20), 0.5, 0.5);
         }
//*** As a test, we'll draw as a cube rather than a ramp with it using the same texture coords.
      if(%inst.static.debugrender)
      {
         %brush.addPlaneByPoints(%pnts.getValue(2),%pnts.getValue(1),%pnts.getValue(5));
         %brush.addPlaneByPoints(%pnts.getValue(6),%pnts.getValue(5),%pnts.getValue(4));

      } else
      {
         %brush.addPlaneByPoints(%pnts.getValue(2),%pnts.getValue(1),%pnts.getValue(4));
      }
               
         //*** Delete the points array
         %pnts.empty();
         %pnts.delete();

         //*** Delete the texture points array
         if(%inst.static.textureorigin == 1)
         {
            %cpnts.empty();
            %cpnts.delete();
         }

      }
         
      //*** Work on the position of the brush, if appropriate
      if(%inst.changeCenter == true)
      {
         %id = %brush.getBrushID();
         //*** Check if his brush has not yet been added to the map.  In this case, the building
         //*** of the planes using the change in size above will put it in the correct
         //*** location.
         if(%id != -1)
         {
            %edit.setBrushPosition(%id, %inst.static.center[0] SPC %inst.static.center[1] SPC %inst.static.center[2]);
         }
      }
         
      //*** Notify the edit operation of the new brush or change in an existing brush.
      %edit.updateBrush(%brush);
   }
   
};

tool.register("PrimitiveAddRamp", tool.typeInteractive(), tool.RFLAG_SUPPORTONECLICKCREATE(), "Build Ramp" );
