//
// PrimitiveAddArch.cs
//
// Defines the PrimitiveAddArch tool plug-in to create a Bezier curve based arch.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// size[]   - Defines the xyz size of the cube as a 3 element array
// center[] - Defines the xyz center point of the cube as a 3 element array
// bezier[4,3]     - Defines the xyz points of the Bezier curve handles as a 4 element array
// height[]        - Defines the xyz point of the leg height handle
// startwidth      - Defines the width at the first control point -- from left bounding edge to control point
// endwidth        - Defines the width at the last control point -- from top bounding edge to control point
// heightctrl      - Defines the z height of the 'legs'
// bezierctrl[2,2] - Defines the xy relative to the start and end points of the two Bezier curve control points as a 2 element array
// changeoverslope - The slope of the Bezier curve when horizontal segments rather than vertical ones should be used
// widthtolerance  - The minimum width at the endpoints before changing to a wedge
// segments - The number of segments (brushes) to build the curve
// cubic    - Flag to indicate that a cubic Bezier curve should be used instead of a quadratic one
// axis     - Defines the axis along which the object will be built 0-2 = xyz
// xaxis    - Constant to do some tricks based on axis
// yaxis    - Constant to do some tricks based on axis
// zaxis    - Constant to do some tricks based on axis
// uscale	- Texture scale along the 'u' axis
// vscale	- Texture scale along the 'v' axis
// uoffset	- Texture offset along the 'u' axis
// voffset	- Texture offset along the 'v' axis
//
// The cube instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos[9] - Defines the xyz center point of the nine user controlable handles, each as a 3 element array. 0=center handle, 1-8=sizing corner handles
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeSize   - Flag to indicate that the tool's bounding box size has changed
// initialdrag  - Flag to indicate that the user is dragging out the bounding box for the first time.  Once the user lets go of the initial drag handle this will be false.
//
// Revision History:
// April 18, 2007		David Wyand		Created script file
//
// References:
// Bzier curve From Wikipedia, the free encyclopedia http://en.wikipedia.org/wiki/B%C3%A9zier_curve
// Drawing a circle with Bzier Curves by G. Adam Stanislav http://www.whizkidtech.redprince.net/bezier/circle/
//

package PrimitiveAddArch
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("PrimitiveAddArch: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         PrimitiveAddArch_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the cube.  This is to share some properties
      //*** such as the cube's centre and size
      %plugin.static = %static;

      //*** Setup the standard bounding box based on the default values
      ToolBB::setType(%static, ToolBB::typeDynamic(), true);
      ToolBB::rebuildBoundingBox(%plugin, %static);

      //*** Setup some additional attributes for the cube instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = false;
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeCenter = false;
      %plugin.changeSize = false;
      %plugin.initialdrag = false;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLAXISSAME(); // Set up the tool with the standard geometry creation GUI
      %inst.flagsApply = tool.AFLAG_STANDARDGEOMETRY();     // Set up the tool with the standard geometry creation post apply selections
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("PrimitiveAddArch: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("PrimitiveAddArch: MouseDown(" @ %inst @ "," @ %event @ ")");

      //*** We only use handles so return 'false'
      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("PrimitiveAddArch: HandleCount(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may
      //*** interact with.  We're using the bounding box helper exclusively here
      //*** so allow it to return the number of handles.
      return %plugin.active ? (ToolBB::getHandleCount() + 5) : 0;      
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("PrimitiveAddArch: HandleInit(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** Make the tool active
      if(!%plugin.active)
      {
         %plugin.active = true;
      }
      
      %plugin.initialdrag = true;

      //*** Allow the bounding box helper to set up the handles
      %handle = ToolBB::initHandles(%plugin, %plugin.static, %event);

      //*** Build the Bezier curve handles
      PrimitiveAddArch_BuildInitialBezierHandles(%plugin);

      return %handle;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine
   // which handle has been selected.  In both cases, %info is a ScriptObject that
   // contains the .pos[3] fields that are to be filled in with the requested handle's
   // position.  This function returns the priority of the handle, the higher the number
   // the higher the priority.  This is used to determine which handle should be selected
   // when two or more handles overlap on the screen.  If -1 is returned, then the handle
   // is considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("PrimitiveAddArch: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      %priority = ToolBB::getHandle(%plugin, %plugin.static, %event, %hindex, %info);

      //*** Work with the Bezier curve handles
      %lastBBHandle = ToolBB::getLastHandleIndex();
      if(%hindex > %lastBBHandle && %hindex <= (%lastBBHandle+4))
      {
         %index = %hindex - %lastBBHandle - 1;

         //*** Don't allow the second control handle if we're not in cubic mode
         if(%index == 2 && !%plugin.static.cubic)
            return -1;

         %info.pos[0] = %plugin.static.bezier[%index,0];
         %info.pos[1] = %plugin.static.bezier[%index,1];
         %info.pos[2] = %plugin.static.bezier[%index,2];
         %priority = %hindex;
      }

      //*** Work with height handle
      if(%hindex == (%lastBBHandle+5))
      {
         %info.pos[0] = %plugin.static.height[0];
         %info.pos[1] = %plugin.static.height[1];
         %info.pos[2] = %plugin.static.height[2];
         %priority = %hindex;
      }

      return %priority;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddArch: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;

      %lastBBHandle = ToolBB::getLastHandleIndex();
      %center[0] = %plugin.static.center[0];
      %center[1] = %plugin.static.center[1];
      %center[2] = %plugin.static.center[2];

      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
      %sizez = %plugin.static.size[%z];

      //*** Move the appropriate bounding box handle.
      %returnHandle = ToolBB::moveHandle(%plugin, %plugin.static, %event, %hindex);

      if(%hindex == 0)
      {
         //*** Adjust the Bezier handles based on the initial drag out
         PrimitiveAddArch_BuildInitialBezierHandles(%plugin);

      } else if(%hindex == 15)
      {
         //*** Center handle was moved so update the height and Bezier handles
         PrimitiveAddArch_HandleCenterHandleChange(%plugin, %center[0], %center[1], %center[2]);

      } else if(%hindex >= 1 && %hindex < 15)
      {
         //*** A sizing handle has been moved so update the height and Bezier handles
         PrimitiveAddArch_RebuildHeightHandle(%plugin, (%plugin.static.size[%z] - %sizez), true);
         PrimitiveAddArch_BezierBoundsCheck(%plugin, 0);
         PrimitiveAddArch_BezierBoundsCheck(%plugin, 3);
         PrimitiveAddArch_BezierBoundsCheck(%plugin, 1);
         PrimitiveAddArch_BezierBoundsCheck(%plugin, 2);

      } else if(%hindex > %lastBBHandle && %hindex <= (%lastBBHandle+4))
      {
         //*** Move the Bezier handles.
         %bezierindex = %hindex - %lastBBHandle - 1;

         //*** Store the position in case we're constrained
         %pos[0] = %plugin.static.bezier[%bezierindex,0];
         %pos[1] = %plugin.static.bezier[%bezierindex,1];
         %pos[2] = %plugin.static.bezier[%bezierindex,2];

         //*** Calculate the position based on our view.  We'd prefer to not
         //*** modify coordinates that the view itself doesn't work with.
         %point = %event.getMousePointOnWorkplaneThroughPoint(%plugin.static.bezier[%bezierindex,0] SPC %plugin.static.bezier[%bezierindex,1] SPC %plugin.static.bezier[%bezierindex,2]);
         %plugin.static.bezier[%bezierindex,0] = getWord(%point, 0);
         %plugin.static.bezier[%bezierindex,1] = getWord(%point, 1);
         %plugin.static.bezier[%bezierindex,2] = getWord(%point, 2);
         if(%event.gridBuildAxis >= 0 && %event.gridBuildAxis <= 2)
         {
            //*** Substitute the old value for the axis the view doesn't modify
            %plugin.static.bezier[%bezierindex,gridBuildAxis] = %pos[%event.gridBuildAxis];
         }

         //*** Adjust for constrained movement
         if(%event.FLAG_CONSTRAINED())
         {
            if(%event.FLAG_CONSTRAINEDX())
            {
               %plugin.static.bezier[%bezierindex,1] = %pos[1];
               %plugin.static.bezier[%bezierindex,2] = %pos[2];
            
            } else if(%event.FLAG_CONSTRAINEDY())
            {
               %plugin.static.bezier[%bezierindex,0] = %pos[0];
               %plugin.static.bezier[%bezierindex,2] = %pos[2];
            
            } else if(%event.FLAG_CONSTRAINEDZ())
            {
               %plugin.static.bezier[%bezierindex,0] = %pos[0];
               %plugin.static.bezier[%bezierindex,1] = %pos[1];
            }
         }

         //*** Make sure the handles don't go out of bounds
         %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
         %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
         %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
         if(%bezierindex == 0)
         {
            //*** Start endpoint handle: Can only move along 'x'
            %plugin.static.bezier[0,%y] = %pos[%y];
            %plugin.static.bezier[0,%z] = %pos[%z];

            PrimitiveAddArch_BezierBoundsCheck(%plugin, %bezierindex);

         } else if(%bezierindex == 1)
         {
            //*** Can only move along 'x' and 'z'
            %plugin.static.bezier[1,%y] = %pos[%y];

            PrimitiveAddArch_BezierBoundsCheck(%plugin, %bezierindex);

         } else if(%bezierindex == 2)
         {
            //*** Can only move along 'x' and 'z'
            %plugin.static.bezier[1,%y] = %pos[%y];

            PrimitiveAddArch_BezierBoundsCheck(%plugin, %bezierindex);

         } else if(%bezierindex == 3)
         {
            //*** End endpoint handle: Can only move along 'z'
            %plugin.static.bezier[3,%x] = %pos[%x];
            %plugin.static.bezier[3,%y] = %pos[%y];

            PrimitiveAddArch_BezierBoundsCheck(%plugin, %bezierindex);
         }

         //*** Rebuild the GUI fields
         PrimitiveAddArch_RebuildHandleGUIFields(%plugin);

         %returnHandle = %hindex;

      } else if(%hindex == (%lastBBHandle+5))
      {
         //*** Move the height handle.

         //*** Store the position in case we're constrained
         %pos[0] = %plugin.static.height[0];
         %pos[1] = %plugin.static.height[1];
         %pos[2] = %plugin.static.height[2];

         //*** Calculate the position based on our view.  We'd prefer to not
         //*** modify coordinates that the view itself doesn't work with.
         %point = %event.getMousePointOnWorkplaneThroughPoint(%plugin.static.height[0] SPC %plugin.static.height[1] SPC %plugin.static.height[2]);
         %plugin.static.height[0] = getWord(%point, 0);
         %plugin.static.height[1] = getWord(%point, 1);
         %plugin.static.height[2] = getWord(%point, 2);
         if(%event.gridBuildAxis >= 0 && %event.gridBuildAxis <= 2)
         {
            //*** Substitute the old value for the axis the view doesn't modify
            %plugin.static.height[gridBuildAxis] = %pos[%event.gridBuildAxis];
         }

         //*** Make sure the handle doesn't go out of bounds
         %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
         %y = getWord(%plugin.static.yaxis, %plugin.static.axis);

         //*** Can only move along 'z'
         %plugin.static.height[%x] = %pos[%x];
         %plugin.static.height[%y] = %pos[%y];

         PrimitiveAddArch_HeightHandleBoundsCheck(%plugin);

         //*** Rebuild the GUI fields
         PrimitiveAddArch_RebuildHeightHandleGUIFields(%plugin);
         PrimitiveAddArch_RebuildHandleGUIFields(%plugin);

         %returnHandle = %hindex;
      }
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      //%plugin.update = tool.EDIT_UPDATE();
      
      return %returnHandle;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
      //error("PrimitiveAddArch: HandleDone(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;

      //*** Indicate that the user is done with the initial drag handle
      if(%plugin.initialdrag && %hindex == 0)
      {
         %plugin.initialdrag = false;
      }

      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("PrimitiveAddArch: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("PrimitiveAddArch: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%plugin.active)
         return;

      //*** Draw the standard bounding box
      ToolBB::draw(%plugin, %plugin.static, %draw);

      //*** Draw the Bezier curve handles
      for(%i=0; %i<4; %i++)
      {
         if(%i != 2 || %plugin.static.cubic == 1)
         {
            %draw.handle(%draw.HANDLE_CUBE(), "0 0 0", %plugin.static.bezier[%i,0], %plugin.static.bezier[%i,1], %plugin.static.bezier[%i,2]);
         }
      }

      //*** And the lines between the Bezier handles
      %draw.changeColor(0,0,0);
      if(!%plugin.static.cubic)
      {
         %draw.moveTo(%draw.COORD_ABSOLUTE(), %plugin.static.bezier[0,0], %plugin.static.bezier[0,1], %plugin.static.bezier[0,2]);
         %draw.lineTo(%draw.COORD_ABSOLUTE()+%draw.LINE_DASHED(), %plugin.static.bezier[1,0], %plugin.static.bezier[1,1], %plugin.static.bezier[1,2]);
         %draw.lineTo(%draw.COORD_ABSOLUTE()+%draw.LINE_DASHED(), %plugin.static.bezier[3,0], %plugin.static.bezier[3,1], %plugin.static.bezier[3,2]);

      } else
      {
         %draw.moveTo(%draw.COORD_ABSOLUTE(), %plugin.static.bezier[0,0], %plugin.static.bezier[0,1], %plugin.static.bezier[0,2]);
         %draw.lineTo(%draw.COORD_ABSOLUTE()+%draw.LINE_DASHED(), %plugin.static.bezier[1,0], %plugin.static.bezier[1,1], %plugin.static.bezier[1,2]);

         %draw.moveTo(%draw.COORD_ABSOLUTE(), %plugin.static.bezier[3,0], %plugin.static.bezier[3,1], %plugin.static.bezier[3,2]);
         %draw.lineTo(%draw.COORD_ABSOLUTE()+%draw.LINE_DASHED(), %plugin.static.bezier[2,0], %plugin.static.bezier[2,1], %plugin.static.bezier[2,2]);
      }

      //*** And the Bezier curve itself
      %draw.changeColor(0,0,0);
      %draw.moveTo(%draw.COORD_ABSOLUTE(), %plugin.static.bezier[0,0], %plugin.static.bezier[0,1], %plugin.static.bezier[0,2]);
      %step = 1.0 / %plugin.static.segments;
      for(%i=1; %i<=%plugin.static.segments; %i++)
      {
         %t = %i * %step;

         %draw.lineTo(%draw.COORD_ABSOLUTE(), PrimitiveAddArch_PointOnBezier(%plugin, %t));
      }

      //*** Draw the leg height lines and handle
      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
      %pnt[%x] = %plugin.static.center[%x] - %plugin.static.size[%x] * 0.5;
      %pnt[%y] = %plugin.static.center[%y];
      %pnt[%z] = %plugin.static.height[%z];
      %draw.changeColor(255,128,0);
      %draw.moveTo(%draw.COORD_ABSOLUTE(), %pnt[0], %pnt[1], %pnt[2]);
      %pnt[%x] = %plugin.static.center[%x] + %plugin.static.size[%x] * 0.5;
      %draw.lineTo(%draw.COORD_ABSOLUTE(), %pnt[0], %pnt[1], %pnt[2]);

      %pnt[%x] = %plugin.static.center[%x];
      %pnt[%y] = %plugin.static.center[%y] - %plugin.static.size[%y] * 0.5;
      %draw.moveTo(%draw.COORD_ABSOLUTE(), %pnt[0], %pnt[1], %pnt[2]);
      %pnt[%y] = %plugin.static.center[%y] + %plugin.static.size[%y] * 0.5;
      %draw.lineTo(%draw.COORD_ABSOLUTE(), %pnt[0], %pnt[1], %pnt[2]);

      %type = %draw.HANDLE_ACTION_X();
      if(%plugin.static.axis == 1)
      {
         %type = %draw.HANDLE_ACTION_Y();
         
      } else if(%plugin.static.axis == 2)
      {
         %type = %draw.HANDLE_ACTION_Z();
      }
      %draw.handle(%type, "255 255 255", %plugin.static.height[0], %plugin.static.height[1], %plugin.static.height[2]);

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("PrimitiveAddArch: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("PrimitiveAddArch: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.active = false;
      %plugin.changeSize = false;
      %plugin.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("PrimitiveAddArch: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      // Work on the actual geometry.
      PrimitiveAddArch_MakeGeometry(%plugin, %edit);
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeSize = false;
      %plugin.changeCenter = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("PrimitiveAddArch: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new cube.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeSize = true;
            %plugin.changeCenter = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            PrimitiveAddArch_DefaultValues(%plugin.static);
            ToolBB::rebuildBoundingBox(%plugin, %plugin.static);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeSize = true;
            %plugin.changeCenter = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         case tool.EVENT_TEXTURECHANGE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_UPDATE();
            }
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("PrimitiveAddArch: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Build Bezier Arch");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Center"   ,"distance3");
      %form.addField( 1, "Size"     ,"distance3");
      %form.addField( 6, "Axis"     ,"axis3");
      %form.addField( 16, "Leg Height"   ,"distance");
      %form.addField( -1, "Curve" ,"divider");
      %form.addField( 7, "No. Segments"  ,"numericinteger");
      %form.addField( 15, " "  ,"actionpopup");
      %form.addField( 8, "Use Cubic" ,"checkbox");
      %form.addField( 9, "Start Width"   ,"distance");
      %form.addField( 10, "Ctrl 1"   ,"distance2");
      %form.addField( 11, "Ctrl 2"   ,"distance2");
      %form.addField( 12, "End Width"   ,"distance");
      %form.addField( -1, "Texturing" ,"divider");
      %form.addField( 2, "U Scale"  ,"numeric");
      %form.addField( 3, "V Scale"  ,"numeric");
      %form.addField( 4, "U Offset" ,"numeric");
      %form.addField( 5, "V Offset" ,"numeric");
      %form.addField( -1, "Advanced" ,"divider");
      %form.addField( 13, "Change Slope"   ,"numeric");
      %form.addField( 14, "Width Tol"   ,"distance");
      
      //*** Set limits on some controls
      %form.setFieldMinLimit(7, 2);
      %form.setFieldMinLimit(13, 0);
      %form.setFieldMinLimit(14, 0);
      %form.setFieldMinLimit(16, 0);
      
      //*** Add items to the 'Presets' popup
      %form.addFieldListItem(15, "Presets");
      %form.addFieldListItem(15, "Linear");
      %form.addFieldListItem(15, "Circle");
      %form.addFieldListItem(15, "Circle Cutout");
      %form.addFieldListItem(15, "Ease In (Cubic)");
      %form.addFieldListItem(15, "Ease Out (Cubic)");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("PrimitiveAddArch: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            %value = %plugin.static.center[0] SPC %plugin.static.center[1] SPC %plugin.static.center[2];
            return %value;
      
         //*** Handle the 'Size' field
         case 1:
            %value = %plugin.static.size[0] SPC %plugin.static.size[1] SPC %plugin.static.size[2];
            return %value;
      
         //*** Handle the 'U Scale' field
         case 2:
            %value = %plugin.static.uscale;
            return %value;
      
         //*** Handle the 'V Scale' field
         case 3:
            %value = %plugin.static.vscale;
            return %value;
      
         //*** Handle the 'U Offset' field
         case 4:
            %value = %plugin.static.uoffset;
            return %value;
      
         //*** Handle the 'V Offset' field
         case 5:
            %value = %plugin.static.voffset;
            return %value;
      
         //*** Handle the 'Axis' field
         case 6:
            %value = %plugin.static.axis;
            return %value;
      
         //*** Handle the 'No. Segments' field
         case 7:
            %value = %plugin.static.segments;
            return %value;
      
         //*** Handle the 'Four Ctrls' field
         case 8:
            %value = %plugin.static.cubic;
            return %value;
      
         //*** Handle the 'Start Width' field
         case 9:
            %value = %plugin.static.startwidth;
            return %value;
      
         //*** Handle the 'P1' field
         case 10:
            %value = %plugin.static.bezierctrl[0,0] SPC %plugin.static.bezierctrl[0,1];
            return %value;
      
         //*** Handle the 'P2' field
         case 11:
            if(%plugin.static.cubic)
            {
               %value = %plugin.static.bezierctrl[1,0] SPC %plugin.static.bezierctrl[1,1];

            } else
            {
               %value = "";
            }
            return %value;
      
         //*** Handle the 'End Width' field
         case 12:
            %value = %plugin.static.endwidth;
            return %value;
      
         //*** Handle the 'Change Slope' field
         case 13:
            %value = %plugin.static.changeoverslope;
            return %value;
      
         //*** Handle the 'Width Tol' field
         case 14:
            %value = %plugin.static.widthtolerance;
            return %value;
      
         //*** Handle the 'Leg Height' field
         case 16:
            %value = %plugin.static.heightctrl;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("PrimitiveAddArch: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Center' field
         case 0:
            for(%i=0; %i<3; %i++)
            {
               %prevcenter[%i] = %plugin.static.center[%i];
               %plugin.static.center[%i] = getWord(%value, %i);
            }
            PrimitiveAddArch_HandleCenterHandleChange(%plugin, %prevcenter[0], %prevcenter[1], %prevcenter[2]);
            %plugin.changeCenter = true;
      
         //*** Handle the 'Size' field
         case 1:
            for(%i=0; %i<3; %i++)
            {
               %plugin.static.size[%i] = getWord(%value, %i);
            }
            PrimitiveAddArch_RebuildHeightHandle(%plugin);
            PrimitiveAddArch_BezierBoundsCheckAll(%plugin);
            %plugin.changeSize = true;
      
         //*** Handle the 'U Scale' field
         case 2:
            %plugin.static.uscale = %value;
            %plugin.changeSize = true;
      
         //*** Handle the 'V Scale' field
         case 3:
            %plugin.static.vscale = %value;
            %plugin.changeSize = true;
      
         //*** Handle the 'U Offset' field
         case 4:
            %plugin.static.uoffset = %value;
            %plugin.changeSize = true;
      
         //*** Handle the 'V Offset' field
         case 5:
            %plugin.static.voffset = %value;
            %plugin.changeSize = true;
      
         //*** Handle the 'Axis' field
         case 6:
            %plugin.static.axis = %value;
            PrimitiveAddArch_RebuildRelativeHeightHandle(%plugin);
            PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
            PrimitiveAddArch_BezierBoundsCheckAll(%plugin);
            %plugin.changeSize = true;
      
         //*** Handle the 'No. Segments' field
         case 7:
            %plugin.static.segments = %value;
            %plugin.changeSize = true;
      
         //*** Handle the 'Four Ctrls' field
         case 8:
            %plugin.static.cubic = %value;
            PrimitiveAddArch_RebuildHandleGUIFields(%plugin);
            %plugin.changeSize = true;
      
         //*** Handle the 'Start Width' field
         case 9:
            %plugin.static.startwidth = %value;

            PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 0);
            PrimitiveAddArch_BezierBoundsCheck(%plugin, 0);

            %plugin.changeSize = true;
      
         //*** Handle the 'P1' field
         case 10:
            for(%i=0; %i<2; %i++)
            {
               %plugin.static.bezierctrl[0,%i] = getWord(%value, %i);
            }

            PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 1);
            PrimitiveAddArch_BezierBoundsCheck(%plugin, 1);

            %plugin.changeSize = true;
      
         //*** Handle the 'P2' field
         case 11:
            if(%plugin.static.cubic)
            {
               for(%i=0; %i<2; %i++)
               {
                  %plugin.static.bezierctrl[1,%i] = getWord(%value, %i);
               }

            } else
            {
               for(%i=0; %i<2; %i++)
               {
                  %plugin.static.bezierctrl[1,%i] = %plugin.static.bezierctrl[0,%i];
               }
            }

            PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 2);
            PrimitiveAddArch_BezierBoundsCheck(%plugin, 2);

            %plugin.changeSize = true;
      
         //*** Handle the 'End Width' field
         case 12:
            %plugin.static.endwidth = %value;

            PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 3);
            PrimitiveAddArch_BezierBoundsCheck(%plugin, 3);

            %plugin.changeSize = true;
      
         //*** Handle the 'Change Slope' field
         case 13:
            %plugin.static.changeoverslope = %value;

            %plugin.changeSize = true;
      
         //*** Handle the 'Width Tol' field
         case 14:
            %plugin.static.widthtolerance = %value;

            %plugin.changeSize = true;
      
         //*** Handle the 'Presets' field
         case 15:
            switch(%value)
            {
               //*** NOTE: Index 0 is the unchanging text of the action popup

               case 1:
                  //*** Linear
                  for(%i=0; %i<3; %i++)
                  {
                     %line = %plugin.static.bezier[3,%i] - %plugin.static.bezier[0,%i];
                     %line = %line * 0.5;
                     %plugin.static.bezier[1,%i] = %plugin.static.bezier[0,%i] + %line;
                     %plugin.static.bezier[2,%i] = %plugin.static.bezier[0,%i] + %line;
                  }

               case 2:
                  //*** Circle
                  if(%plugin.static.endwidth < %plugin.static.startwidth)
                  {
                     %plugin.static.startwidth = %plugin.static.endwidth;

                  } else
                  {
                     %plugin.static.endwidth = %plugin.static.startwidth;
                  }

                  %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
                  %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
                  %plugin.static.heightctrl = %plugin.static.size[%z] - %plugin.static.endwidth - (%plugin.static.size[%x] * 0.5 - %plugin.static.startwidth);

                  %radius = %plugin.static.size[%x] * 0.5 - %plugin.static.startwidth;
                  if(%plugin.static.cubic)
                  {
                     %len = 0.5522847498 * %radius;
                     %plugin.static.bezierctrl[0,0] = %len;
                     %plugin.static.bezierctrl[0,1] = 0;
                     %plugin.static.bezierctrl[1,0] = 0;
                     %plugin.static.bezierctrl[1,1] = %len * -1.0;

                  } else
                  {
                     %empirical = %radius - (%radius / 0.5 * 0.04);
                     %plugin.static.bezierctrl[0,0] = %empirical;
                     %plugin.static.bezierctrl[0,1] = %empirical * -1.0;
                     %plugin.static.bezierctrl[1,0] = %empirical;
                     %plugin.static.bezierctrl[1,1] = %empirical * -1.0;
                  }

                  PrimitiveAddArch_RebuildRelativeHeightHandle(%plugin);
                  PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
                  PrimitiveAddArch_BezierBoundsCheckAll(%plugin);

               case 3:
                  //*** Circle Cutout
                  if(%plugin.static.endwidth < %plugin.static.startwidth)
                  {
                     %plugin.static.startwidth = %plugin.static.endwidth;

                  } else
                  {
                     %plugin.static.endwidth = %plugin.static.startwidth;
                  }

                  %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
                  %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
                  %plugin.static.heightctrl = %plugin.static.size[%z] - %plugin.static.endwidth - (%plugin.static.size[%x] * 0.5 - %plugin.static.startwidth);

                  %radius = %plugin.static.size[%x] * 0.5 - %plugin.static.startwidth;
                  if(%plugin.static.cubic)
                  {
                     %len = 0.5522847498 * %radius;
                     %plugin.static.bezierctrl[0,0] = 0;
                     %plugin.static.bezierctrl[0,1] = %len;
                     %plugin.static.bezierctrl[1,0] = %len * -1.0;
                     %plugin.static.bezierctrl[1,1] = 0;

                  } else
                  {
                     %empirical = %radius / 0.5 * 0.04;
                     %plugin.static.bezierctrl[0,0] = %empirical;
                     %plugin.static.bezierctrl[0,1] = %empirical * -1.0;
                     %plugin.static.bezierctrl[1,0] = %empirical;
                     %plugin.static.bezierctrl[1,1] = %empirical * -1.0;
                  }

                  PrimitiveAddArch_RebuildRelativeHeightHandle(%plugin);
                  PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
                  PrimitiveAddArch_BezierBoundsCheckAll(%plugin);

               case 4:
                  //*** Ease In (Cubic)
                  if(%plugin.static.cubic)
                  {
                     %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
                     %xlen = %plugin.static.size[%x] - %plugin.static.startwidth;

                     %plugin.static.bezierctrl[0,0] = %xlen;
                     %plugin.static.bezierctrl[0,1] = 0;
                     %plugin.static.bezierctrl[1,0] = %xlen * -1.0;
                     %plugin.static.bezierctrl[1,1] = 0;

                     PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
                     PrimitiveAddArch_BezierBoundsCheckAll(%plugin);
                  }

               case 5:
                  //*** Ease Out (Cubic)
                  if(%plugin.static.cubic)
                  {
                     %z = getWord(%plugin.static.zaxis, %plugin.static.axis);
                     %zlen = %plugin.static.size[%z] - %plugin.static.endwidth;

                     %plugin.static.bezierctrl[0,0] = 0;
                     %plugin.static.bezierctrl[0,1] = %zlen;
                     %plugin.static.bezierctrl[1,0] = 0;
                     %plugin.static.bezierctrl[1,1] = %zlen * -1.0;

                     PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
                     PrimitiveAddArch_BezierBoundsCheckAll(%plugin);
                  }
            }
            %plugin.changeSize = true;
      
         //*** Handle the 'Leg Height' field
         case 16:
            %plugin.static.heightctrl = %value;

            PrimitiveAddArch_RebuildRelativeHeightHandle(%plugin);
            PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin);
            PrimitiveAddArch_BezierBoundsCheckAll(%plugin);

            %plugin.changeSize = true;
      }

      PrimitiveAddArch_RebuildHeightHandleGUIFields(%plugin);
      PrimitiveAddArch_RebuildHandleGUIFields(%plugin);

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%plugin.active)
      {
         %plugin.active = true;
         %plugin.changeSize = true;
         %plugin.changeCenter = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      ToolBB::rebuildBoundingBox(%plugin, %plugin.static);
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function PrimitiveAddArch_DefaultValues(%obj)
   {
      %obj.center[0] = 0.0; // x
      %obj.center[1] = 0.0; // y
      %obj.center[2] = 0.0; // z
      
      %obj.size[0] = 1.0; // x
      %obj.size[1] = 1.0; // y
      %obj.size[2] = 1.0; // z
      
      %obj.startwidth = 0.25;
      %obj.endwidth = 0.25;

      %obj.bezierctrl[0,0] = 0.0;
      %obj.bezierctrl[0,1] = 0.0;

      %obj.bezierctrl[1,0] = 0.0;
      %obj.bezierctrl[1,1] = 0.0;

      %obj.bezier[0,0] = -0.25;
      %obj.bezier[0,1] = 0.0;
      %obj.bezier[0,2] = -0.25;
      
      %obj.bezier[1,0] = -0.25;
      %obj.bezier[1,1] = 0.0;
      %obj.bezier[1,2] = 0.25;
      
      %obj.bezier[2,0] = -0.25;
      %obj.bezier[2,1] = 0.0;
      %obj.bezier[2,2] = 0.25;
      
      %obj.bezier[3,0] = 0.0;
      %obj.bezier[3,1] = 0.0;
      %obj.bezier[3,2] = 0.25;

      %obj.heightctrl = 0.25;
      %obj.height[0] = 0.0;
      %obj.height[1] = 0.0;
      %obj.height[2] = -0.25;

      %obj.uscale = 1.0;
      %obj.vscale = 1.0;
      %obj.uoffset = 0.0;
      %obj.voffset = 0.0;
      
      %obj.segments = 4;
      %obj.axis = 2; // z-axis
      %obj.cubic = 0; // Use only 3 control points for Bezier curve
      %obj.changeoverslope = 4;
      %obj.widthtolerance = 0.006;
   }

   //*** Init the static object
   function PrimitiveAddArch_InitStatic(%static)
   {
      //*** Setup default values
      PrimitiveAddArch_DefaultValues(%static);
      
      //*** Setup our constants
      %static.xaxis = "1 2 0";
      %static.yaxis = "2 0 1";
      %static.zaxis = "0 1 2";
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the cube instance to default values
   function PrimitiveAddArch_ResetObject(%plugin)
   {
      //*** Retrieve a pointer to the static data
      %static = %plugin.static;
      
      //*** Setup default values
      PrimitiveAddArch_DefaultValues(%static);
      ToolBB::rebuildBoundingBox(%plugin, %plugin.static);
   }

   //*** Rebuild the height handle based on a change in size
   function PrimitiveAddArch_RebuildHeightHandle(%plugin, %dsizez, %dobezier)
   {
      %plugin.static.heightctrl += %dsizez;
      if(%plugin.static.heightctrl < 0)
         %plugin.static.heightctrl = 0;

      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      %plugin.static.height[%x] = %plugin.static.center[%x];
      %plugin.static.height[%y] = %plugin.static.center[%y];
      %plugin.static.height[%z] = %plugin.static.center[%z] - %plugin.static.size[%z]*0.5 + %plugin.static.heightctrl;

      if(%dobezier)
      {
         //*** Move the Bezier handles too
         %plugin.static.bezier[0, %z] = %plugin.static.center[%z] - %plugin.static.size[%z]*0.5 + %plugin.static.heightctrl;
         %plugin.static.bezier[3, %z] = %plugin.static.center[%z] + %plugin.static.size[%z]*0.5 - %plugin.static.endwidth;
         if(%plugin.static.cubic)
         {
            %plugin.static.bezier[1, %z] = %plugin.static.bezier[0, %z] + %plugin.static.bezierctrl[0, 1];
            %plugin.static.bezier[2, %z] = %plugin.static.bezier[3, %z] + %plugin.static.bezierctrl[1, 1];

         } else
         {
            %plugin.static.bezier[1, %z] = %plugin.static.bezier[3, %z] + %plugin.static.bezierctrl[0, 1];
            %plugin.static.bezier[2, %z] = %plugin.static.bezier[1, %z];
         }
      }
   }

   //*** Rebuild the height handle based the values from the GUI
   function PrimitiveAddArch_RebuildRelativeHeightHandle(%plugin)
   {
      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      %halfsizez = %plugin.static.size[%z]*0.5;

      %plugin.static.height[%x] = %plugin.static.center[%x];
      %plugin.static.height[%y] = %plugin.static.center[%y];

      if(%plugin.static.heightctrl > %plugin.static.size[%z])
         %plugin.static.heightctrl = %plugin.static.size[%z];

      %plugin.static.height[%z] = %plugin.static.center[%z] - %halfsizez + %plugin.static.heightctrl;
   }

   //*** Checks the bounds on the height handle
   function PrimitiveAddArch_HeightHandleBoundsCheck(%plugin)
   {
      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      %halfSizeZ = %plugin.static.size[%z] * 0.5;

      %plugin.static.height[%x] = %plugin.static.center[%x];
      %plugin.static.height[%y] = %plugin.static.center[%y];

      if(%plugin.static.height[%z] > %plugin.static.center[%z] + %halfSizeZ)
         %plugin.static.height[%z] = %plugin.static.center[%z] + %halfSizeZ;

      if(%plugin.static.height[%z] < %plugin.static.center[%z] - %halfSizeZ)
         %plugin.static.height[%z] = %plugin.static.center[%z] - %halfSizeZ;

      //*** Handle dependants
      PrimitiveAddArch_BezierBoundsCheckAll(%plugin);
   }

   //*** Rebuilds the values displayed to the user based on the
   //*** height handle being moved
   function PrimitiveAddArch_RebuildHeightHandleGUIFields(%plugin)
   {
      %static = %plugin.static;

      %x = getWord(%static.xaxis, %static.axis);
      %y = getWord(%static.yaxis, %static.axis);
      %z = getWord(%static.zaxis, %static.axis);

      %static.heightctrl = mAbs(%static.center[%z] - %static.size[%z]*0.5 - %static.height[%z]);
   }

   //*** Set up the Bezier handles for the initial drag out
   function PrimitiveAddArch_BuildInitialBezierHandles(%plugin)
   {
      %static = %plugin.static;

      %x = getWord(%static.xaxis, %static.axis);
      %y = getWord(%static.yaxis, %static.axis);
      %z = getWord(%static.zaxis, %static.axis);

      %quarterSize[%x] = %static.size[%x]*0.25;
      %quarterSize[%z] = %static.size[%z]*0.25;

      %static.startwidth = %quarterSize[%x];
      %static.endwidth = %quarterSize[%z];

      %halfSize[%x] = %static.size[%x]*0.5;
      %halfSize[%z] = %static.size[%z]*0.5;

      %static.height[%x] = %static.center[%x];
      %static.height[%y] = %static.center[%y];
      %static.height[%z] = %static.center[%z] - %quarterSize[%z];
      %static.heightctrl = %quarterSize[%z];

      %static.bezier[0,%x] = %static.center[%x] - %quarterSize[%x];
      %static.bezier[0,%y] = %static.center[%y];
      %static.bezier[0,%z] = %static.height[%z];

      if(!%static.cubic)
      {
         %static.bezier[1,%x] = %static.center[%x] - %quarterSize[%x];
         %static.bezier[1,%y] = %static.center[%y];
         %static.bezier[1,%z] = %static.center[%z] + %quarterSize[%z];

         %static.bezier[2,%x] = %static.center[%x] - %quarterSize[%x];
         %static.bezier[2,%y] = %static.center[%y];
         %static.bezier[2,%z] = %static.center[%z] + %quarterSize[%z];

         %static.bezierctrl[0,0] = 0.0;
         %static.bezierctrl[0,1] = 0.0;

         %static.bezierctrl[1,0] = 0.0;
         %static.bezierctrl[1,1] = 0.0;

      } else
      {
         %static.bezier[1,%x] = %static.center[%x] - %quarterSize[%x];
         %static.bezier[1,%y] = %static.center[%y];
         %static.bezier[1,%z] = %static.center[%z] - %quarterSize[%z];

         %static.bezier[2,%x] = %static.center[%x] + %quarterSize[%x];
         %static.bezier[2,%y] = %static.center[%y];
         %static.bezier[2,%z] = %static.center[%z] + %quarterSize[%z];

         %static.bezierctrl[0,0] = 0.0;
         %static.bezierctrl[0,1] = %static.bezier[1,%z] - %static.bezier[0,%z];

         %static.bezierctrl[1,0] = %static.bezier[2,%x] - %static.bezier[3,%x];
         %static.bezierctrl[1,1] = 0.0;
      }

      %static.bezier[3,%x] = %static.center[%x];
      %static.bezier[3,%y] = %static.center[%y];
      %static.bezier[3,%z] = %static.center[%z] + %quarterSize[%z];
   }
   
   function PrimitiveAddArch_RebuildAllRelativeBezierHandles(%plugin)
   {
      PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 0);
      PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 3);
      PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 1);
      PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, 2);
   }

   //*** Rebuild the given Bezier handle based the values from the GUI
   function PrimitiveAddArch_RebuildRelativeBezierHandle(%plugin, %bezierindex)
   {
      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      if(%bezierindex == 0)
      {
         %plugin.static.bezier[0,%x] = %plugin.static.center[%x] - %plugin.static.size[%x]*0.5 + %plugin.static.startwidth;
         %plugin.static.bezier[0,%z] = %plugin.static.center[%z] - %plugin.static.size[%z]*0.5 + %plugin.static.heightctrl;

      } else if(%bezierindex == 1)
      {
         if(!%plugin.static.cubic)
         {
            %plugin.static.bezier[1,%x] = %plugin.static.bezier[0,%x] + %plugin.static.bezierctrl[0,0];
            %plugin.static.bezier[1,%z] = %plugin.static.bezier[3,%z] + %plugin.static.bezierctrl[0,1];

         } else
         {
            %plugin.static.bezier[1,%x] = %plugin.static.bezier[0,%x] + %plugin.static.bezierctrl[0,0];
            %plugin.static.bezier[1,%z] = %plugin.static.bezier[0,%z] + %plugin.static.bezierctrl[0,1];
         }

      } else if(%bezierindex == 2)
      {
         if(%plugin.static.cubic)
         {
            %plugin.static.bezier[2,%x] = %plugin.static.bezier[3,%x] + %plugin.static.bezierctrl[1,0];
            %plugin.static.bezier[2,%z] = %plugin.static.bezier[3,%z] + %plugin.static.bezierctrl[1,1];

         } else
         {
            for(%i=0; %i<3; %i++)
            {
               %plugin.static.bezier[2,%i] = %plugin.static.bezier[1,%i];
            }
         }

      } else if(%bezierindex == 3)
      {
         %plugin.static.bezier[3,%x] = %plugin.static.center[%x];
         %plugin.static.bezier[3,%z] = %plugin.static.center[%z] + %plugin.static.size[%z]*0.5 - %plugin.static.endwidth;
      }
   }

   //*** Rebuilds the values displayed to the user based on the
   //*** Bezier handles being moved
   function PrimitiveAddArch_RebuildHandleGUIFields(%plugin)
   {
      %static = %plugin.static;

      %x = getWord(%static.xaxis, %static.axis);
      %y = getWord(%static.yaxis, %static.axis);
      %z = getWord(%static.zaxis, %static.axis);

      %static.startwidth = mAbs(%static.center[%x] - %static.size[%x]*0.5 - %static.bezier[0,%x]);
      %static.endwidth = mAbs(%static.center[%z] + %static.size[%z]*0.5 - %static.bezier[3,%z]);

      if(!%static.cubic)
      {
         %static.bezierctrl[0,0] = %static.bezier[1,%x] - %static.bezier[0,%x];
         %static.bezierctrl[0,1] = %static.bezier[1,%z] - %static.bezier[3,%z];

         %static.bezierctrl[1,0] = %static.bezier[1,%x] - %static.bezier[0,%x];
         %static.bezierctrl[1,1] = %static.bezier[1,%z] - %static.bezier[3,%z];

      } else
      {
         %static.bezierctrl[0,0] = %static.bezier[1,%x] - %static.bezier[0,%x];
         %static.bezierctrl[0,1] = %static.bezier[1,%z] - %static.bezier[0,%z];

         %static.bezierctrl[1,0] = %static.bezier[2,%x] - %static.bezier[3,%x];
         %static.bezierctrl[1,1] = %static.bezier[2,%z] - %static.bezier[3,%z];
      }
   }

   //*** Checks the bounds on all Bezier curve handles
   function PrimitiveAddArch_HandleCenterHandleChange(%plugin, %prevcenterX, %prevcenterY, %prevcenterZ)
   {
         //*** Center handle was moved so update the Bezier handles
         %dx = %plugin.static.center[0] - %prevcenterX;
         %dy = %plugin.static.center[1] - %prevcenterY;
         %dz = %plugin.static.center[2] - %prevcenterZ;
         for(%i=0; %i<4; %i++)
         {
            %plugin.static.bezier[%i,0] += %dx;
            %plugin.static.bezier[%i,1] += %dy;
            %plugin.static.bezier[%i,2] += %dz;
         }

         //*** And the leg height handle
         %plugin.static.height[0] += %dx;
         %plugin.static.height[1] += %dy;
         %plugin.static.height[2] += %dz;
   }

   //*** Checks the bounds on all Bezier curve handles
   function PrimitiveAddArch_BezierBoundsCheckAll(%plugin)
   {
      PrimitiveAddArch_BezierBoundsCheck(%plugin, 0);
      PrimitiveAddArch_BezierBoundsCheck(%plugin, 3);
      PrimitiveAddArch_BezierBoundsCheck(%plugin, 1);
      PrimitiveAddArch_BezierBoundsCheck(%plugin, 2);
   }

   //*** Checks the bounds on the given Bezier curve handle
   function PrimitiveAddArch_BezierBoundsCheck(%plugin, %bezierindex)
   {
      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      %halfSizeX = %plugin.static.size[%x] * 0.5;
      %halfSizeZ = %plugin.static.size[%z] * 0.5;

      %plugin.static.bezier[%bezierindex,%y] = %plugin.static.center[%y];

      if(%bezierindex == 0)
      {
         //%plugin.static.bezier[0,%z] = %plugin.static.center[%z] - %halfSizeZ;
         %plugin.static.bezier[0,%z] = %plugin.static.height[%z];

         //*** Don't go out of bounds
         if(%plugin.static.bezier[0,%x] > %plugin.static.center[%x] + %halfSizeX)
         {
            %plugin.static.bezier[0,%x] = %plugin.static.center[%x] + %halfSizeX;
         }
         if(%plugin.static.bezier[0,%x] < (%plugin.static.center[%x] - %halfSizeX) )
         {
            %plugin.static.bezier[0,%x] = %plugin.static.center[%x] - %halfSizeX;
         }

         //*** Make sure the other control handles follow along
         if(%plugin.static.bezier[1,%x] < %plugin.static.bezier[0,%x])
         {
            %plugin.static.bezier[1,%x] = %plugin.static.bezier[0,%x];
         }
         if(%plugin.static.bezier[2,%x] < %plugin.static.bezier[0,%x])
         {
            %plugin.static.bezier[2,%x] = %plugin.static.bezier[0,%x];
         }

      } else if(%bezierindex == 1)
      {
         //*** First control handle: Must +ve 'x' from start endpoint and
         //*** -ve 'z' from end endpoint.  Must also be contained within BB.
         if(%plugin.static.bezier[1,%x] < %plugin.static.bezier[0,%x])
         {
            %plugin.static.bezier[1,%x] = %plugin.static.bezier[0,%x];
         }
         if(%plugin.static.bezier[1,%x] > %plugin.static.bezier[3,%x])
         {
            %plugin.static.bezier[1,%x] = %plugin.static.bezier[3,%x];
         }

         if(%plugin.static.bezier[1,%z] > %plugin.static.bezier[3,%z])
         {
            %plugin.static.bezier[1,%z] = %plugin.static.bezier[3,%z];
         }
         if(%plugin.static.bezier[1,%z] < %plugin.static.bezier[0,%z])
         {
            %plugin.static.bezier[1,%z] = %plugin.static.bezier[0,%z];
         }

      } else if(%bezierindex == 2)
      {
         //*** Second control handle: Must -ve 'x' from end endpoint and
         //*** +ve 'z' from start endpoint.  Must also be contained within BB.
         if(%plugin.static.bezier[2,%x] < %plugin.static.bezier[0,%x])
         {
            %plugin.static.bezier[2,%x] = %plugin.static.bezier[0,%x];
         }
         if(%plugin.static.bezier[2,%x] > %plugin.static.bezier[3,%x])
         {
            %plugin.static.bezier[2,%x] = %plugin.static.bezier[3,%x];
         }

         if(%plugin.static.bezier[2,%z] > %plugin.static.bezier[3,%z])
         {
            %plugin.static.bezier[2,%z] = %plugin.static.bezier[3,%z];
         }
         if(%plugin.static.bezier[2,%z] < %plugin.static.bezier[0,%z])
         {
            %plugin.static.bezier[2,%z] = %plugin.static.bezier[0,%z];
         }

      } else if(%bezierindex == 3)
      {
         %plugin.static.bezier[3,%x] = %plugin.static.center[%x];

         //*** Don't go out of bounds
         if(%plugin.static.bezier[3,%z] < %plugin.static.center[%z] - %halfSizeZ)
         {
            %plugin.static.bezier[3,%z] = %plugin.static.center[%z] - %halfSizeZ;
         }
         if(%plugin.static.bezier[3,%z] < %plugin.static.height[%z])
         {
            %plugin.static.bezier[3,%z] = %plugin.static.height[%z];
         }
         if(%plugin.static.bezier[3,%z] > (%plugin.static.center[%z] + %halfSizeZ) )
         {
            %plugin.static.bezier[3,%z] = %plugin.static.center[%z] + %halfSizeZ;
         }

         //*** Make sure the other control handles follow along
         if(%plugin.static.bezier[1,%z] > %plugin.static.bezier[3,%z])
         {
            %plugin.static.bezier[1,%z] = %plugin.static.bezier[3,%z];
         }
         if(%plugin.static.bezier[2,%z] > %plugin.static.bezier[3,%z])
         {
            %plugin.static.bezier[2,%z] = %plugin.static.bezier[3,%z];
         }
      }
   }

   //*** Build/modify the actual geometry
   function PrimitiveAddArch_PointOnBezier(%plugin, %t, %side)
   {
      if(%side)
      {
         //*** Mirror the controls points
         %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
         for(%j = 0; %j<3; %j++)
         {
            if(%j == %x)
            {
               %ctrl[0,%j] = %plugin.static.center[%j] - %plugin.static.bezier[0,%j] + %plugin.static.center[%j];
               %ctrl[1,%j] = %plugin.static.center[%j] - %plugin.static.bezier[1,%j] + %plugin.static.center[%j];
               %ctrl[2,%j] = %plugin.static.center[%j] - %plugin.static.bezier[2,%j] + %plugin.static.center[%j];
               %ctrl[3,%j] = %plugin.static.center[%j] - %plugin.static.bezier[3,%j] + %plugin.static.center[%j];
            } else
            {
               %ctrl[0,%j] = %plugin.static.bezier[0,%j];
               %ctrl[1,%j] = %plugin.static.bezier[1,%j];
               %ctrl[2,%j] = %plugin.static.bezier[2,%j];
               %ctrl[3,%j] = %plugin.static.bezier[3,%j];
            }
         }

      } else
      {
         for(%j = 0; %j<3; %j++)
         {
            %ctrl[0,%j] = %plugin.static.bezier[0,%j];
            %ctrl[1,%j] = %plugin.static.bezier[1,%j];
            %ctrl[2,%j] = %plugin.static.bezier[2,%j];
            %ctrl[3,%j] = %plugin.static.bezier[3,%j];
         }
      }

      if(!%plugin.static.cubic)
      {
         %a = (1.0 - %t);
         %a = %a * %a;
         %b = 2 * %t * (1.0 - %t);
         %c = %t * %t;

         for(%j = 0; %j<3; %j++)
         {
            %p[%j] = (%a * %ctrl[0,%j]) + (%b * %ctrl[1,%j]) + (%c * %ctrl[3,%j]);
         }

      } else
      {
         %invt = 1.0 - %t;
         %a = %invt * %invt * %invt;
         %b = %t * %invt * %invt;
         %c = %t * %t * %invt;
         %d = %t * %t * %t;

         for(%j = 0; %j<3; %j++)
         {
            %p[%j] = (%a * %ctrl[0,%j]) + (3.0 * %b * %ctrl[1,%j]) + (3.0 * %c * %ctrl[2,%j]) + (%d * %ctrl[3,%j]);
         }
      }

      return %p[0] SPC %p[1] SPC %p[2];
   }

   //*** Build/modify the actual geometry
   function PrimitiveAddArch_MakeGeometry(%plugin, %edit)
   {
      //*** If we're not active, don't create geometry
      if(!%plugin.active)
         return;

      %x = getWord(%plugin.static.xaxis, %plugin.static.axis);
      %y = getWord(%plugin.static.yaxis, %plugin.static.axis);
      %z = getWord(%plugin.static.zaxis, %plugin.static.axis);

      for(%i=0; %i<3; %i++)
      {
         %hs[%i] = %plugin.static.size[%i] * 0.5;
         %center[%i] = %plugin.static.center[%i];
      }

      %bezierstep = 1.0 / %plugin.static.segments;

      //*** Build the points that will be used for the brush planes.
      //*** 'front' set is index 0 and the 'back' set is index 1.
      //*** 'front mirror' is 2 and 'back mirror' is 3.
      //*** The 'bottom' point is first, followed by the corresponding
      //*** 'top' point.

      %p[%x] = (-%hs[%x]);
      %p[%y] = (-%hs[%y]);
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[0,0] = %p[0] SPC %p[1] SPC %p[2]; // Front bottom left

      %p[%z] = %hs[%z];
      %pnts[0,1] = %p[0] SPC %p[1] SPC %p[2];  // Front top left

      %p[%x] = 0.0;
      %boundstopright = %p[0] SPC %p[1] SPC %p[2];  // Front top right

      %p[%x] = %plugin.static.bezier[0,%x] - %center[%x];
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[0,2] = %p[0] SPC %p[1] SPC %p[2]; // Front start point bottom

      %p[%x] = (-%hs[%x]);
      %p[%y] = %hs[%y];
      %pnts[1,0] = %p[0] SPC %p[1] SPC %p[2]; // Back bottom left

      %p[%z] = %hs[%z];
      %pnts[1,1] = %p[0] SPC %p[1] SPC %p[2];  // Back top left

      %p[%x] = %plugin.static.bezier[0,%x] - %center[%x];
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[1,2] = %p[0] SPC %p[1] SPC %p[2]; // Back start point

      %p[%x] = %hs[%x];
      %p[%y] = (-%hs[%y]);
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[2,0] = %p[0] SPC %p[1] SPC %p[2]; // Mirror Front bottom right

      %p[%z] = %hs[%z];
      %pnts[2,1] = %p[0] SPC %p[1] SPC %p[2];  // Mirror Front top right

      %p[%x] = 0.0;
      %boundstopleft = %p[0] SPC %p[1] SPC %p[2];  // Mirror Front top left

      %p[%x] = mAbs(%plugin.static.bezier[0,%x] - %center[%x]);
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[2,2] = %p[0] SPC %p[1] SPC %p[2]; // Mirror Front start point bottom

      %p[%x] = %hs[%x];
      %p[%y] = %hs[%y];
      %pnts[3,0] = %p[0] SPC %p[1] SPC %p[2]; // Mirror Back bottom right

      %p[%z] = %hs[%z];
      %pnts[3,1] = %p[0] SPC %p[1] SPC %p[2];  // Mirror Back top right

      %p[%x] = mAbs(%plugin.static.bezier[0,%x] - %center[%x]);
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %pnts[3,2] = %p[0] SPC %p[1] SPC %p[2]; // Mirror Back start point

      //*** Calculate each of the Bezier curve points
      %texlen[2] = 0.0;
      %index = 3;
      for(%i=1; %i<=%plugin.static.segments; %i++)
      {
         %t = %i * %bezierstep;
         for(%j=0; %j<2; %j++)
         {
            %localindex = %index;
            %pnt = PrimitiveAddArch_PointOnBezier(%plugin, %t, %j); // Calculate first point after start
            if(%j == 0)
            {
               %side = 0;
               %pntXCalc = getWord(%pnt, %x) - %center[%x];

            } else
            {
               %side = 2;
               %pntXCalc = mAbs(getWord(%pnt, %x) - %center[%x]);
            }
            %pntZCalc = getWord(%pnt, %z) - %center[%z];

            %p[%x] = %pntXCalc;
            %p[%y] = (-%hs[%y]);
            %p[%z] = %pntZCalc;
            %pnts[%side,%localindex] = %p[0] SPC %p[1] SPC %p[2]; // Front bottom

            %p[%y] = %hs[%y];
            %pnts[%side+1,%localindex] = %p[0] SPC %p[1] SPC %p[2]; // Back bottom

            if(%index != 3)
            {
               %texlen[%localindex] = %texlen[%localindex-2] + VectorDist(%pnts[%side,%localindex], %pnts[%side,%localindex-2]) * %plugin.static.vscale;

            } else
            {
               %texlen[%localindex] = %texlen[%localindex-1] + VectorDist(%pnts[%side,%localindex], %pnts[%side,%localindex-1]) * %plugin.static.vscale;
            }

            %localindex++;

            %p[%x] = %pntXCalc;
            %p[%y] = (-%hs[%y]);
            %p[%z] = %hs[%z];
            %pnts[%side,%localindex] = %p[0] SPC %p[1] SPC %p[2]; // Front top

            %p[%y] = %hs[%y];
            %pnts[%side+1,%localindex] = %p[0] SPC %p[1] SPC %p[2]; // Back top

         }
         %index += 2;
      }

      %requiredbrushcount = %plugin.static.segments * 2;

      //*** Check if the slope of the Bezier curve for the first brush warrants a
      //*** horizontal brush.
      %firsthorizontal = false;
      %firstfiller = false;
      %slope = (getWord(%pnts[0,3], %z) - getWord(%pnts[0,2], %z)) / (getWord(%pnts[0,3], %x) - getWord(%pnts[0,2], %x));
      if(%slope > %plugin.static.changeoverslope)
      {
         %firsthorizontal = true;
         %firstfiller = true;
         %requiredbrushcount += 2;
      }

      //*** Should the first brush be a wedge?
      %firstwedge = false;
      if(VectorDist(%pnts[0,0], %pnts[0,2]) < %plugin.static.widthtolerance)
      {
         %firstwedge = true;
      }

      //*** Check if the slope of the Bezier curve for the last brush warrants a
      //*** horizontal brush.
      %lasthorizontal = false;
      %slope = (getWord(%pnts[0,(%index-2)], %z) - getWord(%pnts[0,(%index-4)], %z)) / (getWord(%pnts[0,(%index-2)], %x) - getWord(%pnts[0,(%index-4)], %x));
      if(%slope > %plugin.static.changeoverslope)
      {
         %lasthorizontal = true;
      }

      //*** Should the last brush be a wedge?
      %lastwedge = false;
      %lastfiller = false;
      if(VectorDist(%boundstopright, %pnts[0,(%index-2)]) < %plugin.static.widthtolerance)
      {
         %lastwedge = true;

      } else if(%lasthorizontal)
      {
         //*** Not a wedge but is horizontal, so we need a filler
         %lastfiller = true;
         %requiredbrushcount += 2;
      }

      //*** Should a leg brush be built?
      %buildleg = false;
      if(!%firstwedge && %plugin.static.heightctrl > 0.0)
      {
         %buildleg = true;
         %requiredbrushcount += 2;
      }

      //*** If the number of buffered brushes doesn't match our number
      //*** then clear it out and begin anew.
      %count = %edit.getEditBrushCount();
      if(%count != %requiredbrushcount)
      {
         %edit.clearAllNewBrushes();
      }

      //*** Points used by texturing
      %p[%x] = 0.0;
      %p[%y] = %hs[%y];
      %p[%z] = %lasthorizontal ? getWord(%pnts[0,(%index-4)], %z) : %hs[%z];
      %backtopright = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = (-%hs[%x]);
      %backtopleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = 0.0;
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %backbottomright = %p[0] SPC %p[1] SPC %p[2];
      %p[%y] = (-%hs[%y]);
      %p[%z] = %lasthorizontal ? getWord(%pnts[0,(%index-4)], %z) : %hs[%z];
      %fronttopright = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = (-%hs[%x]);
      %fronttopleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = 0.0;
      %p[%z] = (-%hs[%z]) + %plugin.static.heightctrl;
      %frontbottomright = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = (-%hs[%x]);
      %p[%y] = (-%hs[%y]);
      %p[%z] = getWord(%pnts[0,3], %z);
      %frontmidleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%z] = getWord(%pnts[0,(%index-2)], %z);
      %frontmid2left = %p[0] SPC %p[1] SPC %p[2];

      %p[%x] = (-%hs[%x]);
      %p[%y] = (-%hs[%y]);
      %p[%z] = (-%hs[%z]);
      %legfrontbottomleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = %plugin.static.bezier[0,%x] - %center[%x];
      %legfrontbottomright = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = (-%hs[%x]);
      %p[%y] = %hs[%y];
      %legbackbottomleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%x] = %plugin.static.bezier[0,%x] - %center[%x];
      %legbackbottomright = %p[0] SPC %p[1] SPC %p[2];

      %p[%x] = mAbs(%plugin.static.bezier[0,%x] - %center[%x]);
      %p[%y] = (-%hs[%y]);
      %p[%z] = (-%hs[%z]);
      %mlegfrontbottomleft = %p[0] SPC %p[1] SPC %p[2];
      %p[%y] = %hs[%y];
      %mlegbackbottomleft = %p[0] SPC %p[1] SPC %p[2];

      //**************************************
      //*** Build the first brush
      %nextBrushIndex = 0;
      %brush = %edit.getEditBrush(%nextBrushIndex);
      if(%brush == -1)
      {
         %brush = new MapBrush();
      }
      %nextBrushIndex++;

      %brush2 = %edit.getEditBrush(%nextBrushIndex);
      if(%brush2 == -1)
      {
         %brush2 = new MapBrush();
      }
      %nextBrushIndex++;

      %brush.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
      %brush.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
      %brush.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
      %brush.clearAllPlanes();

      %brush2.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
      %brush2.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
      %brush2.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
      %brush2.clearAllPlanes();

      %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
      %brush.addPlaneByPoints(%pnts[0,0], %pnts[1,0], %pnts[1,1]); // Brush 1 Left
      %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
      %brush2.addPlaneByPoints(%pnts[3,0], %pnts[2,0], %pnts[2,1]); // Brush 2 Right

      %brush.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
      %brush.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 1 Front
      %brush2.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
      %brush2.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 2 Front

      %brush.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
      %brush.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 1 Back
      %brush2.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
      %brush2.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 2 Back

      //*** Horizontal brush?
      if(!%firsthorizontal)
      {
         %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,3], %pnts[0,3], %pnts[0,4]); // Brush 1 Right
         %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[2,3], %pnts[3,3], %pnts[3,4]); // Brush 2 left

         %brush.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 1 Top
         %brush2.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush2.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 2 Top

      } else
      {
         %brush.setTexturePlanesByPoints2(%fronttopright, %pnts[0,1], %pnts[1,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,3], %pnts[0,3], %frontmidleft); // Brush 1 Top
         %brush2.setTexturePlanesByPoints2(%fronttopright, %pnts[0,1], %pnts[1,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,3], %pnts[0,3], %frontmidleft); // Brush 2 Top
      }

      //*** Check if the bottom would be less than the width tolerance.  If so then
      //*** build a wedge.
      if(!%firstwedge)
      {
         //*** Regular shape
         %brush.setTexturePlanesByPoints2(%pnts[1,2], %pnts[0,2], %pnts[0,3], 0.0, -%texlen[2]);
         %brush.addPlaneByPoints(%pnts[1,2], %pnts[0,2], %pnts[0,3]); // Brush 1 Start to 1st step
         %brush2.setTexturePlanesByPoints2(%pnts[2,2], %pnts[3,2], %pnts[3,3], 0.0, -%texlen[2]);
         %brush2.addPlaneByPoints(%pnts[2,2], %pnts[3,2], %pnts[3,3]); // Brush 2 Start to 1st step

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[0,2], %pnts[1,2], %pnts[1,0]); // Brush 1 Bottom
         %brush2.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[0,2], %pnts[1,2], %pnts[1,0]); // Brush 2 Bottom

      } else
      {
         //*** Wedge shape
         %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,3], 0.0, -%texlen[2]);
         %brush.addPlaneByPoints(%pnts[1,0], %pnts[0,0], %pnts[0,3]); // Brush 1 left to 1st step
         %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,3], 0.0, -%texlen[2]);
         %brush2.addPlaneByPoints(%pnts[2,0], %pnts[3,0], %pnts[3,3]); // Brush 2 left to 1st step
      }

      %edit.updateBrush(%brush);
      %edit.updateBrush(%brush2);

      //**************************************
      //*** All other brushes
      %topfillerneeded = false;
      %lastbrush = %plugin.static.segments-2;
      for(%i=0; %i<=%lastbrush; %i++)
      {
         %brush = %edit.getEditBrush(%nextBrushIndex);
         if(%brush == -1)
         {
            %brush = new MapBrush();
         }
         %nextBrushIndex++;

         %brush2 = %edit.getEditBrush(%nextBrushIndex);
         if(%brush2 == -1)
         {
            %brush2 = new MapBrush();
         }
         %nextBrushIndex++;

         %brush.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush.clearAllPlanes();

         %brush2.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush2.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush2.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush2.clearAllPlanes();

         %bindex = %i*2+5;
         %tindex = %i*2+6;
         %pbindex = %bindex-2;
         %ptindex = %tindex-2;

         %brush.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 1 Front
         %brush2.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 2 Front

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,0], %backbottomright, %backtopright);  // Brush 1 Back
         %brush2.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,0], %backbottomright, %backtopright);  // Brush 2 Back

         if(%i == %lastbrush && %lasthorizontal)
         {
            %brush.setTexturePlanesByPoints2(%pnts[1,%bindex],  %pnts[0,%bindex], %frontmid2left, 0.0, 0.0);
            %brush.addPlaneByPoints(%pnts[1,%bindex],  %pnts[0,%bindex], %frontmid2left); // Brush 1 Top
            %brush2.setTexturePlanesByPoints2(%pnts[1,%bindex],  %pnts[0,%bindex], %frontmid2left, 0.0, 0.0);
            %brush2.addPlaneByPoints(%pnts[1,%bindex],  %pnts[0,%bindex], %frontmid2left); // Brush 1 Top

            %brush.setTexturePlanesByPoints2(%backtopleft, %fronttopleft, %fronttopright, 0.0, 0.0);
            %brush.addPlaneByPoints(%backtopleft, %fronttopleft, %fronttopright); // Brush 1 Bottom
            %brush2.setTexturePlanesByPoints2(%backtopleft, %fronttopleft, %fronttopright, 0.0, 0.0);
            %brush2.addPlaneByPoints(%backtopleft, %fronttopleft, %fronttopright); // Brush 2 Bottom

            %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
            %brush.addPlaneByPoints(%pnts[0,0], %pnts[1,0], %pnts[1,1]); // Brush 1 Left
            %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
            %brush2.addPlaneByPoints(%pnts[3,0], %pnts[2,0], %pnts[2,1]); // Brush 2 Right

         } else
         {
            %brush.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
            %brush.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 1 Top
            %brush2.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
            %brush2.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 2 Top

            %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
            %brush.addPlaneByPoints(%pnts[0,%pbindex], %pnts[1,%pbindex], %pnts[1,%ptindex]); // Brush 1 Left
            %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
            %brush2.addPlaneByPoints(%pnts[3,%pbindex], %pnts[2,%pbindex], %pnts[2,%ptindex]); // Brush 2 Right
         }

         //*** If this is the last brush then check if a wedge should be produced
         if(%i == %lastbrush && %lastwedge)
         {
            //*** Build a wedge
            %brush.setTexturePlanesByPoints2(%pnts[1,%pbindex], %pnts[0,%pbindex], %pnts[0,%bindex], 0.0, -%texlen[%pbindex]);
            %brush.addPlaneByPoints(%pnts[0,%tindex],  %pnts[1,%tindex],  %pnts[1,%pbindex]); // Brush 1 Angled Bottom
            %brush2.setTexturePlanesByPoints2(%pnts[2,%pbindex], %pnts[3,%pbindex], %pnts[3,%bindex], 0.0, -%texlen[%pbindex]);
            %brush2.addPlaneByPoints(%pnts[3,%tindex],  %pnts[2,%tindex],  %pnts[2,%pbindex]); // Brush 2 Angled Bottom

         } else
         {
            //*** Build a regular brush
            if(!(%i == %lastbrush && %lasthorizontal))
            {
               %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,1], 0.0, 0.0);
               %brush.addPlaneByPoints(%pnts[1,%bindex],  %pnts[0,%bindex],  %pnts[0,%tindex]);  // Brush 1 Right
               %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,1], 0.0, 0.0);
               %brush2.addPlaneByPoints(%pnts[2,%bindex],  %pnts[3,%bindex],  %pnts[3,%tindex]);  // Brush 2 Left
            }

            %brush.setTexturePlanesByPoints2(%pnts[1,%pbindex], %pnts[0,%pbindex], %pnts[0,%bindex], 0.0, -%texlen[%pbindex]);
            %brush.addPlaneByPoints(%pnts[0,%bindex],  %pnts[1,%bindex],  %pnts[1,%pbindex]); // Brush 1 Angled Bottom
            %brush2.setTexturePlanesByPoints2(%pnts[2,%pbindex], %pnts[3,%pbindex], %pnts[3,%bindex], 0.0, -%texlen[%pbindex]);
            %brush2.addPlaneByPoints(%pnts[3,%bindex],  %pnts[2,%bindex],  %pnts[2,%pbindex]); // Brush 2 Angled Bottom
         }

         %edit.updateBrush(%brush);
         %edit.updateBrush(%brush2);
      }

      //*** Do we need to add a filler brush to the left?
      if(%firstfiller)
      {
         %brush = %edit.getEditBrush(%nextBrushIndex);
         if(%brush == -1)
         {
            %brush = new MapBrush();
         }
         %nextBrushIndex++;

         %brush2 = %edit.getEditBrush(%nextBrushIndex);
         if(%brush2 == -1)
         {
            %brush2 = new MapBrush();
         }
         %nextBrushIndex++;

         %brush.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush.clearAllPlanes();

         %brush2.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush2.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush2.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush2.clearAllPlanes();

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[0,0], %pnts[1,0], %pnts[1,1]); // Brush 1 Left
         %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[3,0], %pnts[2,0], %pnts[2,1]); // Brush 2 Right

         %brush.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 1 Front
         %brush2.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 2 Front

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 1 Back
         %brush2.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 2 Back

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,3], %pnts[0,3], %pnts[0,4]); // Brush 1 Right
         %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[2,3], %pnts[3,3], %pnts[3,4]); // Brush 2 Left

         %brush.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 1 Top
         %brush2.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush2.addPlaneByPoints(%fronttopright, %fronttopleft, %backtopleft); // Brush 2 Top

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush.addPlaneByPoints(%frontmidleft, %pnts[0,3], %pnts[1,3]); // Brush 1 Bottom
         %brush2.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontmidleft, %pnts[0,3], %pnts[1,3]); // Brush 2 Bottom

         %edit.updateBrush(%brush);
         %edit.updateBrush(%brush2);
      }

      //*** Do we need to add a filler brush to the top?
      if(%lastfiller)
      {
         %brush = %edit.getEditBrush(%nextBrushIndex);
         if(%brush == -1)
         {
            %brush = new MapBrush();
         }
         %nextBrushIndex++;

         %brush2 = %edit.getEditBrush(%nextBrushIndex);
         if(%brush2 == -1)
         {
            %brush2 = new MapBrush();
         }
         %nextBrushIndex++;

         %brush.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush.clearAllPlanes();

         %brush2.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush2.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush2.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush2.clearAllPlanes();

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[0,0], %pnts[1,0], %pnts[1,1]); // Brush 1 Left
         %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[3,0], %pnts[2,0], %pnts[2,1]); // Brush 2 Right

         %brush.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 1 Front
         %brush2.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 2 Front

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 1 Back
         %brush2.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 2 Back

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%backbottomright, %frontbottomright, %fronttopright); // Brush 1 Right
         %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%fronttopright, %frontbottomright, %backbottomright); // Brush 2 Left

         %brush.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush.addPlaneByPoints(%boundstopright, %pnts[0,1], %pnts[1,1]); // Brush 1 Top
         %brush2.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush2.addPlaneByPoints(%boundstopright, %pnts[0,1], %pnts[1,1]); // Brush 2 Top

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush.addPlaneByPoints(%frontmid2left,  %pnts[0,%index-2], %pnts[1,%index-2]); // Brush 1 Bottom
         %brush2.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontmid2left,  %pnts[0,%index-2], %pnts[1,%index-2]); // Brush 2 Bottom

         %edit.updateBrush(%brush);
         %edit.updateBrush(%brush2);
      }

      //*** Do we need to add a leg?
      if(%buildleg)
      {
         //*** First the left side
         %brush = %edit.getEditBrush(%nextBrushIndex);
         if(%brush == -1)
         {
            %brush = new MapBrush();
         }
         %nextBrushIndex++;

         %brush2 = %edit.getEditBrush(%nextBrushIndex);
         if(%brush2 == -1)
         {
            %brush2 = new MapBrush();
         }
         %nextBrushIndex++;

         %brush.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush.clearAllPlanes();

         %brush2.setOrigin(%center[0] SPC %center[1] SPC %center[2]);
         %brush2.setTextureOffset(%plugin.static.uoffset, %plugin.static.voffset);
         %brush2.setTextureScale(%plugin.static.uscale, %plugin.static.vscale);
         %brush2.clearAllPlanes();

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %pnts[1,0], %pnts[1,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[0,0], %pnts[1,0], %pnts[1,1]); // Brush 1 Left
         %brush2.setTexturePlanesByPoints2(%pnts[3,0], %pnts[2,0], %pnts[2,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[3,0], %pnts[2,0], %pnts[2,1]); // Brush 2 Right

         %brush.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 1 Front
         %brush2.setTexturePlanesByPoints2(%frontbottomright, %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%frontbottomright, %pnts[0,0], %pnts[0,1]); // Brush 2 Front

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 1 Back
         %brush2.setTexturePlanesByPoints2(%pnts[1,0], %backbottomright, %backtopright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,1], %pnts[1,0], %backbottomright); // Brush 2 Back

         %brush.setTexturePlanesByPoints2(%pnts[1,0], %pnts[0,0], %pnts[0,1], 0.0, 0.0);
         %brush.addPlaneByPoints(%legbackbottomright, %legfrontbottomright, %pnts[0,2]); // Brush 1 Right
         %brush2.setTexturePlanesByPoints2(%pnts[2,0], %pnts[3,0], %pnts[3,1], 0.0, 0.0);
         %brush2.addPlaneByPoints(%mlegfrontbottomleft, %mlegbackbottomleft, %pnts[3,2]); // Brush 2 Left

         %brush.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush.addPlaneByPoints(%pnts[1,2], %pnts[0,2], %pnts[0,0]); // Brush 1 Top
         %brush2.setTexturePlanesByPoints2(%fronttopright, %fronttopleft, %backtopleft, 0.0, 0.0);
         %brush2.addPlaneByPoints(%pnts[1,2], %pnts[0,2], %pnts[0,0]); // Brush 2 Top

         %brush.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush.addPlaneByPoints(%legbackbottomleft,  %legfrontbottomleft, %legfrontbottomright); // Brush 1 Bottom
         %brush2.setTexturePlanesByPoints2(%pnts[0,0], %frontbottomright, %backbottomright, 0.0, 0.0);
         %brush2.addPlaneByPoints(%legbackbottomleft,  %legfrontbottomleft, %legfrontbottomright); // Brush 2 Bottom

         %edit.updateBrush(%brush);
         %edit.updateBrush(%brush2);
      }
   }
   
};

tool.register("PrimitiveAddArch", tool.typeInteractive(), tool.RFLAG_NONE(), "Build Bezier Arch" );
