//
// csghollow.cs
//
// Defines the CSGHollow tool plug-in that will carve out selected brushes.
//
// The static ScriptObject makes use of the following dynamic fields:
// init      - If 'true' indicates that the static has been initialized (needs to be done once)
// thickness - The wall thickness to leave behind after a carve
//
// The plug-in instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
//
// Revision History:
// August 2, 2005		David Wyand		Created script file based on the work done by Brandon Maness
//

package CSGHollow
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("CSGHollow: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Check that we're the Brush selection mode is current -- the only one
      //*** that is supported at this time.
      if(select.getSelectionType() !$= "SelectPrimitives")
      {
         tool.activateErrorMsg = "This tool only works in the 'Brushes' selection mode.";
         return tool.FUNC_BADSELECTMODE();
      }
      
      //*** Make sure that at least one brush is selected
      %map = scene.getCurrentMap();
      if(%map.getNumSelectedBrushes() == 0)
      {
         tool.activateErrorMsg = "Please select one or more brushes for this tool to operate on.";
         return tool.FUNC_BADGENERAL();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         CSGHollow_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties.
      %plugin.static = %static;

      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = false;
      %plugin.update = tool.EDIT_DONOTHING();
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_NONE();
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("CSGHollow: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("CSGHollow: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("CSGHollow: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%plugin.active)
         return;

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("CSGHollow: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("CSGHollow: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      //*** If we're to keep the operation, this is where we'll do the work
      if(%keep)
      {
         //*** Perform the hollow by forcing an update
         %plugin.update = tool.EDIT_UPDATE();
         %plugin.dirty = tool.DIRTY_APPEARANCE();
         tool.refresh();
         
         scene.notifyBrushRefresh();
      }
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.active = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("CSGHollow: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;

      %map = scene.getCurrentMap();
      %sel = %map.getSelectedBrushes();
      %count = getWordCount(%sel);

      %doublethick = %plugin.static.thickness * 2.0; // Doubled for a wall per side
      
      for(%i=0; %i<%count; %i++)
      {
         %brush = getWord(%sel, %i); // Original
         %dup = %map.duplicateBrush(%brush); // Cutter
         
         //*** Calculate scale for the cutter
         %bounds = %map.getBrushBounds(%brush);
         for(%j=0; %j<3; %j++)
         {
            %b[%j] = getWord(%bounds, %j+3) - getWord(%bounds, %j);
            %s[%j] = (%b[%j] - %doublethick) / %b[%j];
         }
         %map.scaleBrush(%dup, %s[0] SPC %s[1] SPC %s[2]);

         %map.performCSGSubtraction(%dup, %brush);
         
         %map.removeBrush(%dup);
      }
            
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("CSGHollow: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new object.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            CSGHollow_DefaultValues(%plugin.static);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("CSGHollow: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Hollow Brush");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Wall Thickness", "distance");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("CSGHollow: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the thickness field
         case 0:
            %value = %plugin.static.thickness;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("CSGHollow: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the thickness field
         case 0:
            %plugin.static.thickness = %value;
      }

      //*** If we're not yet active, make it as if an EVENT_ACTIVATE has been received
      if(!%plugin.active)
      {
         %plugin.active = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      //%plugin.update = tool.EDIT_UPDATE();
      //%plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function CSGHollow_DefaultValues(%obj)
   {
      %obj.thickness = 0.5;
   }

   //*** Init the static object
   function CSGHollow_InitStatic(%static)
   {
      //*** Setup default values
      CSGHollow_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the plug-in instance to default values
   function CSGHollow_Reset(%plugin)
   {
      //*** Retrieve a pointer to the static data
      %static = %plugin.static;
      
      //*** Setup default values
      CSGHollow_DefaultValues(%static);
   }
   
};

tool.register("CSGHollow", tool.typeDialog(), tool.RFLAG_NONE(), "Hollow Brush" );
