//
// csgclip.cs
//
// Defines the CSGClip tool plug-in to slice up brushes using a 3D plane.
//
// The static ScriptObject makes use of the following dynamic fields:
// init         - If 'true' indicates that the static has been initialized (needs to be done once)
// handlepos[4] - Defines the three plane handles and center handle, each as "x y z".  0-2=plane handle, 3=center handle
//
// The plug-in instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// handleinit   - Indicates that the user is currently drawing out the tool for the first time.  Used to have the handles orient themselves.
// changeParam  - Flag to indicate that a tool's parameter has changed ie: plane handle
// changeCenter - Flag to indicate that the tool's center handle has been modified by the user
// handlecenter - Stores the latest center handle position as set when changeCenter == true
//
// Revision History:
// August 3, 2006		David Wyand		Created script file
//

package CSGClip
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("CSGClip: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         CSGClip_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;
      
      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = false;
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.handleinit = false;
      %plugin.changeCenter = false;
      %plugin.changeParam = false;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLSAME(); // Set up the tool with the standard geometry creation GUI and all views to use the same drawing buffer.  Our Draw() will only be called once.
      %inst.flagsApply = tool.AFLAG_NONE();                                            // No special actions when tool is applied
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("CSGClip: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("CSGClip: MouseDown(" @ %inst @ "," @ %event @ ")");

      return false;
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("CSGClip: HandleCount(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may interact with.
      return %plugin.active ? 4 : 0; // When active we have three handles
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse button has just been
   // depressed.  %event will contain the particulars of where the mouse button was pressed
   // to allow for the tool to set itself up for the first time.  The value returned is the
   // (zero-based) index of the handle that is now active and will be dragged by the user
   // until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("CSGClip: HandleInit(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** Make the tool active
      if(!%plugin.active)
      {
         %plugin.active = true;
      }
      
      //*** Set the tool's start, center and end based on where the user clicked.
      for(%i=0; %i<=3; %i++)
      {
         %plugin.static.handlepos[%i] = %event.pos;
      }
      
      //*** Record that the handles have changed, and we need to update our display
      %plugin.handleinit = true;
      %plugin.changeParam = true;
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      //*** Return the 3rd handle index
      return 2;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine which handle
   // has been selected.  In both cases, %info is a ScriptObject that contains the
   // .pos[3] fields that are to be filled in with the requested handle's position.
   // This function returns the priority of the handle, the higher the number the higher
   // the priority.  This is used to determine which handle should be selected when two
   // or more handles overlap on the screen.  If -1 is returned, then the handle is
   // considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("CSGClip: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      if(%hindex >= 0 && %hindex <= 3)
      {
         %info.pos[0] = getWord(%plugin.static.handlepos[%hindex], 0);
         %info.pos[1] = getWord(%plugin.static.handlepos[%hindex], 1);
         %info.pos[2] = getWord(%plugin.static.handlepos[%hindex], 2);
      
         //*** Return the handle's priority
         return %hindex; // set the handle's priority the same as its index
      }
      
      //*** The given handle index is not one of ours
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("CSGClip: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;
      
      if(%hindex >= 0 && %hindex <= 3)
      {
         //*** Store the handle's current values in case we're constrained
         %currentpos = %plugin.static.handlepos[%hindex];
         
         if(%hindex == 3)
         {
            //*** Store the previous center handle for later
            %plugin.handlecenter = %currentpos;
         }

         //*** If the move is being performed in the perspective view, we need
         //*** to calculate the movement based on a plane parallel to the grid
         //*** but going through the handle.
         if(%event.view == 3)
         {
            %point = %event.getMousePointOnWorkplaneThroughPoint(%plugin.static.handlepos[%hindex]);
            %plugin.static.handlepos[%hindex] = %point;
         
         } else
         {
            %plugin.static.handlepos[%hindex] = %event.pos;
         }

         //*** Adjust for constrained movement
         if(%event.FLAG_CONSTRAINED() && !%plugin.handleinit)
         {
            if(%event.FLAG_CONSTRAINEDX())
            {
               %plugin.static.handlepos[%hindex] = getWord(%plugin.static.handlepos[%hindex],0) SPC getWord(%currentpos, 1) SPC getWord(%currentpos, 2);
            
            } else if(%event.FLAG_CONSTRAINEDY())
            {
               %plugin.static.handlepos[%hindex] = getWord(%currentpos, 0) SPC getWord(%plugin.static.handlepos[%hindex],1) SPC getWord(%currentpos, 2);
            
            } else if(%event.FLAG_CONSTRAINEDZ())
            {
               %plugin.static.handlepos[%hindex] = getWord(%currentpos, 0) SPC getWord(%currentpos, 1) SPC getWord(%plugin.static.handlepos[%hindex],2);
            }
         }
         
         //*** If the CTRL key is held then we want to also move the other corner handles
         //*** by the same amount.
         if(%event.MOD_CTRL() && %hindex != 3 && !%plugin.handleinit)
         {
            for(%i=0; %i<3; %i++)
            {
               if(%i == %hindex)
                  continue;
               
               //*** Build the amount to move by
               for(%j=0; %j<3; %j++)
               {
                  %p[%j] = getWord(%plugin.static.handlepos[%hindex],%j) - getWord(%currentpos, %j) + getWord(%plugin.static.handlepos[%i],%j);
               }
               
               %plugin.static.handlepos[%i] = %p[0] SPC %p[1] SPC %p[2];
            }
         }
         
         //*** Deal with the initial handle drag out
         if(%plugin.handleinit)
         {
            //*** Handle 0 stays fixed while Handle 2 is moved by the user.  Handle 1 is
            //*** calculated here based on the view.
            if(%event.view == 0 || (%event.view == 3 && %event.gridBuildAxis == 0) )
            {
               //*** X
               %plugin.static.handlepos[1] = getWord(%plugin.static.handlepos[2],0) SPC getWord(%plugin.static.handlepos[2],1) SPC getWord(%plugin.static.handlepos[0],2);
               
            } else if(%event.view == 1 || (%event.view == 3 && %event.gridBuildAxis == 1) )
            {
               //*** Y
               %plugin.static.handlepos[1] = getWord(%plugin.static.handlepos[2],0) SPC getWord(%plugin.static.handlepos[2],1) SPC getWord(%plugin.static.handlepos[0],2);
               
            } else //if(%event.view == 2 || (%event.view == 3 && %event.gridBuildAxis == 2) )
            {
               //*** Z
               %plugin.static.handlepos[1] = getWord(%plugin.static.handlepos[2],0) SPC getWord(%plugin.static.handlepos[0],1) SPC getWord(%plugin.static.handlepos[2],2);
            }
         }
         
         //*** Record that a handle is being changed
         if(%hindex == 3)
         {
            %plugin.changeCenter = true;
            
         } else
         {
            %plugin.changeParam = true;
         }
         
      }
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      return %hindex;
   }

   //************************************************************************************
   // HandleDone()
   //
   // This function is called when the mouse button is released while manipulating a handle.
   // The %hindex is the (zero-based) index of the handle that is being adjusted.  This
   // function does not return a value.
   function Plugin::HandleDone(%this, %inst, %event, %hindex)
   {
      //error("TransformSize: MouseUp(" @ %inst @ "," @ %event @ ")");
      
      %plugin = %inst.instance;
      
      //*** Set that we're no longer in a handle init state
      %plugin.handleinit = false;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("CSGClip: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      if(%plugin.dirty)
      {
         //*** Update the handles.  The actual slice is done within EndEditAction().
         if(%plugin.changeParam)
         {
            //*** Recalculate the center handle based on the start and end ones.
            %diff = VectorSub(%plugin.static.handlepos[0], %plugin.static.handlepos[2]);
            %x = ( getWord(%plugin.static.handlepos[0],0) + getWord(%plugin.static.handlepos[1],0) + getWord(%plugin.static.handlepos[2],0) ) / 3.0;
            %y = ( getWord(%plugin.static.handlepos[0],1) + getWord(%plugin.static.handlepos[1],1) + getWord(%plugin.static.handlepos[2],1) ) / 3.0;
            %z = ( getWord(%plugin.static.handlepos[0],2) + getWord(%plugin.static.handlepos[1],2) + getWord(%plugin.static.handlepos[2],2) ) / 3.0;
         
            %plugin.static.handlepos[3] = %x SPC %y SPC %z;
         
            %plugin.changeParam = false;
         
         } else if(%plugin.changeCenter)
         {
            //*** Move the start and end handles by the same amount that the center has been moved
            %diff = VectorSub(%plugin.static.handlepos[3], %plugin.handlecenter);
         
            %plugin.static.handlepos[0] = VectorAdd(%diff, %plugin.static.handlepos[0]);
            %plugin.static.handlepos[1] = VectorAdd(%diff, %plugin.static.handlepos[1]);
            %plugin.static.handlepos[2] = VectorAdd(%diff, %plugin.static.handlepos[2]);
         
            %plugin.changeCenter = false;
         }
      }
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class and is
   // used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("CSGClip: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%plugin.active)
         return;

      //*** Draw the four handles
      %draw.handle(%draw.HANDLE_CUBE(), "255 0 0",     getWord(%plugin.static.handlepos[0],0), getWord(%plugin.static.handlepos[0],1), getWord(%plugin.static.handlepos[0],2));
      %draw.handle(%draw.HANDLE_CUBE(), "0 255 0",     getWord(%plugin.static.handlepos[1],0), getWord(%plugin.static.handlepos[1],1), getWord(%plugin.static.handlepos[1],2));
      %draw.handle(%draw.HANDLE_CUBE(), "0 0 255",     getWord(%plugin.static.handlepos[2],0), getWord(%plugin.static.handlepos[2],1), getWord(%plugin.static.handlepos[2],2));
      %draw.handle(%draw.HANDLE_CUBE(), "255 255 255", getWord(%plugin.static.handlepos[3],0), getWord(%plugin.static.handlepos[3],1), getWord(%plugin.static.handlepos[3],2));
         
      //*** Draw an outline for the triangle
      %draw.changeColorAlpha(255, 128, 0, 255);
      %draw.buildTriangle(%draw.POLYGON_OUTLINE(), %plugin.static.handlepos[0], %plugin.static.handlepos[1], %plugin.static.handlepos[2]);

      //*** Draw the triangle
      %draw.changeColorAlpha(255, 128, 0, 128);
      %draw.buildTriangle(%draw.DRAW_NONOCCLUDED_ONLY(), %plugin.static.handlepos[0], %plugin.static.handlepos[1], %plugin.static.handlepos[2]);
      
      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("CSGClip: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("CSGClip: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      //*** If we're to keep the operation, this is where we'll do the slice
      if(%keep && %plugin.active)
      {
         //*** If there are selected brushes on the map then we'll slice those.  Otherwise we
         //*** slice the whole map.
         %scene = scene.getCurrent();
         %map = scene.getCurrentMap();
         if(%scene >= 0 && %map >= 0)
         {
            warn("!!! Performing the slice");

            //*** Convert the three points from workplane space, if any
            %point1 = %scene.translatePointToWorkplane(%scene.getCurrentDetailLevel(), %plugin.static.handlepos[0]);
            %point2 = %scene.translatePointToWorkplane(%scene.getCurrentDetailLevel(), %plugin.static.handlepos[1]);
            %point3 = %scene.translatePointToWorkplane(%scene.getCurrentDetailLevel(), %plugin.static.handlepos[2]);
            
            %count = %map.getNumSelectedBrushes();
            if(%count > 0)
            {
               //*** Slice the selected brushes
               %ret = %map.performCSGSliceWithPoints(%point1, %point2, %point3, %map.getSelectedBrushes());
               
            } else
            {
               //*** Slice the whole map
               %ret = %map.performCSGSliceWithPoints(%point1, %point2, %point3);
            }
            
            //*** Tell everyone to update their brush list
            scene.notifyBrushRefresh();
            
            warn("!!! " @ %ret);
         }
      }
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.active = false;
      %plugin.changeParam = false;
      %plugin.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("CSGClip: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** The actual slice is done within EndEditAction().  Do nothing here.
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("CSGClip: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeParam = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            CSGClip_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeParam = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("CSGClip: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Clip Plane");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Point 1"   ,"distance3");
      %form.addField( 1, "Point 2"   ,"distance3");
      %form.addField( 2, "Point 3"   ,"distance3");
      %form.addField( 3, "Center"    ,"distance3");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("CSGClip: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Point 1' field
         case 0:
            %value = %plugin.static.handlepos[0];
            return %value;
            
         //*** Handle the 'Point 2' field
         case 1:
            %value = %plugin.static.handlepos[1];
            return %value;
            
         //*** Handle the 'Point 3' field
         case 2:
            %value = %plugin.static.handlepos[2];
            return %value;
      
         //*** Handle the 'Center' field
         case 3:
            %value = %plugin.static.handlepos[3];
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("CSGClip: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Point 1' field
         case 0:
            %plugin.static.handlepos[0] = %value;
            %plugin.changeParam = true;
            
         //*** Handle the 'Point 2' field
         case 1:
            %plugin.static.handlepos[1] = %value;
            %plugin.changeParam = true;
            
         //*** Handle the 'Point 3' field
         case 2:
            %plugin.static.handlepos[2] = %value;
            %plugin.changeParam = true;
      
         //*** Handle the 'Center' field
         case 3:
            %plugin.handlecenter = %plugin.static.handlepos[3]; //*** Store the previous center handle for later
            %plugin.static.handlepos[3] = %value;               //*** And store the new value
            %plugin.changeCenter = true;
      }

      //*** Make sure we're active
      %plugin.active = true;
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function CSGClip_DefaultValues(%static)
   {
      %static.handlepos[0] = "1.0 0.0 0.0"; // plane handle 1
      %static.handlepos[1] = "0.0 1.0 0.0"; // plane handle 2
      %static.handlepos[2] = "0.0 0.0 1.0"; // plane handle 3
      %static.handlepos[3] = "0.0 0.0 0.0"; // center handle
   }

   //*** Init the static object
   function CSGClip_InitStatic(%static)
   {
      //*** Setup default values
      CSGClip_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the tube instance to default values
   function CSGClip_Reset(%inst)
   {
      //*** Setup default values
      CSGClip_DefaultValues(%inst.static);
   }
};

tool.register("CSGClip", tool.typeInteractive(), tool.RFLAG_NONE(), "Clip" );
