//
// clonelinear.cs
//
// Defines the CloneLinear tool plug-in to duplicate selected objects.
//
// The static ScriptObject makes use of the following dynamic fields:
// numclones  - The number of clones to make
// offset     - The XYZ distance offset between each clone
// scale      - The XYZ amount to scale each clone from the previous
// rotation   - The HPB amount to rotate each clone from the previous
//
// The plug-in instance makes use of the following dynamic fields:
// duplist      - List of duplicated brush ID's.  The format is "brushID cloneNum".
// shapeDupList - List of duplicated shape ID's which includes point entities.  The format is "shapeID cloneNum".
// oldNumClones - Stores the number of clones prior to a change in numclones.
// selcenter    - Stores the selection center at tool activation time
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeNum    - Flag to indicate that the number of clones has changed
// changeTrans  - Flag to indicate that a transform parameter (ie: rotation) has changed
//
// Revision History:
// August 8, 2006		David Wyand		Created script file
//

package CloneLinear
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("CloneLinear: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Check that we're the Brush selection mode is current -- the only one
      //*** that is supported at this time.
      if(select.getSelectionType() !$= "SelectPrimitives")
      {
         tool.activateErrorMsg = "This tool only works in the 'Brushes' selection mode.";
         return tool.FUNC_BADSELECTMODE();
      }

      //*** With the appropriate preference set, make sure there is a selected brush or shape
      if(pref.getBool("ToolOperateSelectionOnly"))
      {
         if(select.count() == 0)
         {
            tool.activateErrorMsg = "Please select a brush or shape for this tool to operate on.";
            return tool.FUNC_BADSELECTMODE();
         }
      }
      
      //*** Set up the static data.
      if(!%static.init)
      {
         CloneLinear_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;
      
      //*** Setup some additional attributes for the instance
      %plugin.duplist = "";
      %plugin.shapeDupList = "";
      %plugin.oldNumClones = 0;
      %plugin.selcenter = select.getSelectionCenter();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.changeNum = true; // Set to true at start to generate our first batch of duplicated brushes
      %plugin.changeTrans = false;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_RESETBUTTON() + tool.IFLAG_MAKEBUTTON() + tool.IFLAG_DRAWALLSAME();
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("CloneLinear: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("CloneLinear: MouseDown(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      return true;
   }
   
   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("CloneLinear: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty;
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class and is
   // used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("CloneLinear: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** Draw our interface
      %center = %plugin.selcenter;
      %draw.changeColor(0,255,255);
      %draw.moveTo(%draw.COORD_ABSOLUTE(), %center);
      for(%i=0; %i<%plugin.static.numclones; %i++)
      {
         %draw.lineTo(%draw.COORD_RELATIVE(), %plugin.static.offset);
         %draw.plus(%draw.LINE_SOLID(), 0.1);
      }

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("CloneLinear: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      error("CloneLinear: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      if(%keep)
      {
         %plugin.duplist = "";
         %plugin.shapeDupList = "";
         %plugin.oldNumClones = 0;
         %plugin.update = tool.EDIT_DONOTHING();
         
      } else
      {
         %scene = scene.getCurrent();
         %map = scene.getCurrentMap();

         //*** Clean up the cloned brushes here
         %count = getWordCount(%plugin.duplist);
         for(%i=0; %i<%count; %i=%i+2)
         {
            %map.removeBrush(getWord(%plugin.duplist, %i));
         }

         //*** Clean up the cloned shapes (and point entities) here
         %count = getWordCount(%plugin.shapeDupList);
         for(%i=0; %i<%count; %i=%i+2)
         {
            %scene.removeShape(getWord(%plugin.shapeDupList, %i));
         }
         
         %plugin.duplist = "";
         %plugin.shapeDupList = "";
         %plugin.oldNumClones = 0;
         %plugin.update = tool.EDIT_DONOTHING();
      }
      
      %plugin.changeNum = false;
      %plugin.changeTrans = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("CloneLinear: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      if(%plugin.changeNum)
      {
         //*** The number of clones has changed.  If it is more then we just create new clones
         //*** and move them.  If it is less then remove the old clones and reset our internal list.
         if(%plugin.static.numclones > %plugin.oldNumClones)
         {
            %diff = %plugin.static.numclones - %plugin.oldNumClones;

            //*** Work with brushes using the helper function
            %plugin.duplist = CloneHelperAddBrushes(%plugin.duplist, %plugin.oldNumClones, %diff);

            //*** Work with shapes using the helper function
            %plugin.shapeDuplist = CloneHelperAddShapes(%plugin.shapeDuplist, %plugin.oldNumClones, %diff);
            
         } else if(%plugin.static.numclones < %plugin.oldNumClones)
         {
            //*** Work with brushes using the clone helper function
            %plugin.duplist = CloneHelperRemoveBrushes(%plugin.duplist, %plugin.static.numclones);

            //*** Work with shapes and point entities using the clone helper function
            %plugin.shapeDuplist = CloneHelperRemoveShapes(%plugin.shapeDuplist, %plugin.static.numclones);
         }
         
         %plugin.oldNumClones = %plugin.static.numclones;
         
         %plugin.changeTrans = true;
      }
      
      if(%plugin.changeTrans)
      {
         //*** Work with brushes
         %count = getWordCount(%plugin.duplist);
         for(%i=0; %i<%count; %i=%i+2)
         {
            %dupID = getWord(%plugin.duplist, %i);
            %cloneNum = getWord(%plugin.duplist, %i+1);

            %center = %plugin.selcenter;

            //*** Transform the brush
            %offsetNum = %cloneNum + 1;
            
            for(%s=0; %s<3; %s++)
            {
               %scale[%s] = getWord(%plugin.static.scale,%s);
               if(%scale[%s] > 1.0)
               {
                  %scale[%s] -= 1.0;
                  %scale[%s] *= %offsetNum;
                  %scale[%s] += 1.0;
                  
               } else if(%scale[%s] < 1.0)
               {
                  %mul = %scale[%s];
                  for(%t=1; %t<%offsetNum; %t++)
                  {
                     %scale[%s] *= %mul;
                  }
               }
            }
            %scaleoff = %scale[0] SPC %scale[1] SPC %scale[2];
            %edit.scaleBrushCenterRelBuf(%dupID, %scaleoff, %center);
            
            %rotoff = getWord(%plugin.static.rotation,2) * %offsetNum SPC getWord(%plugin.static.rotation,1) * %offsetNum SPC getWord(%plugin.static.rotation,0) * %offsetNum;
            %edit.rotateBrushCenterRel(%dupID, %rotoff, %center);
            
            %offset = getWord(%plugin.static.offset,0) * %offsetNum SPC getWord(%plugin.static.offset,1) * %offsetNum SPC getWord(%plugin.static.offset,2) * %offsetNum;
            %edit.moveBrushRel(%dupID, %offset);
         }

         //*** Work with shapes and point entities
         %count = getWordCount(%plugin.shapeDuplist);
         for(%i=0; %i<%count; %i=%i+2)
         {
            %dupID = getWord(%plugin.shapeDuplist, %i);
            %cloneNum = getWord(%plugin.shapeDuplist, %i+1);

            %center = %plugin.selcenter;

            //*** Transform the shape
            %offsetNum = %cloneNum + 1;
            
            for(%s=0; %s<3; %s++)
            {
               %scale[%s] = getWord(%plugin.static.scale,%s);
               if(%scale[%s] > 1.0)
               {
                  %scale[%s] -= 1.0;
                  %scale[%s] *= %offsetNum;
                  %scale[%s] += 1.0;
                  
               } else if(%scale[%s] < 1.0)
               {
                  %mul = %scale[%s];
                  for(%t=1; %t<%offsetNum; %t++)
                  {
                     %scale[%s] *= %mul;
                  }
               }
            }
            %scaleoff = %scale[0] SPC %scale[1] SPC %scale[2];
            %edit.scaleShapeCenterRelBuf(%dupID, %scaleoff, %center);
            
            %rotoff = getWord(%plugin.static.rotation,2) * %offsetNum SPC getWord(%plugin.static.rotation,1) * %offsetNum SPC getWord(%plugin.static.rotation,0) * %offsetNum;
            %edit.rotateShapeCenterRel(%dupID, %rotoff, %center);
            
            %offset = getWord(%plugin.static.offset,0) * %offsetNum SPC getWord(%plugin.static.offset,1) * %offsetNum SPC getWord(%plugin.static.offset,2) * %offsetNum;
            %edit.moveShapeRel(%dupID, %offset);
         }
      }
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeNum = false;
      %plugin.changeTrans = false;
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("CloneLinear: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeNum = true;
            %plugin.changeTrans = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            CloneLinear_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeNum = true;
            %plugin.changeTrans = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            %plugin.update = tool.EDIT_REJECT();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("CloneLinear: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Clone: Linear");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Clones"     ,"numericinteger");
      %form.addField( 1, "Offset"     ,"distance3");
      %form.addField( 2, "Scale"      ,"distance3");
      %form.addField( 3, "Rotation"   ,"angle3");
      
      //*** Set limits on some controls
      %form.setFieldMinLimit(0, 1);
      %form.setFieldMinLimit(2, 0);
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("CloneLinear: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Clones' field
         case 0:
            %value = %plugin.static.numclones;
            return %value;
            
         //*** Handle the 'Offset' field
         case 1:
            %value = %plugin.static.offset;
            return %value;
            
         //*** Handle the 'Scale' field
         case 2:
            %value = %plugin.static.scale;
            return %value;
            
         //*** Handle the 'Rotation' field
         case 3:
            %value = %plugin.static.rotation;
            return %value;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("CloneLinear: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Clones' field
         case 0:
            %plugin.static.numclones = %value;
            %plugin.changeNum = true;
            
         //*** Handle the 'Offset' field
         case 1:
            %plugin.static.offset = %value;
            %plugin.changeTrans = true;
            
         //*** Handle the 'Scale' field
         case 2:
            %plugin.static.scale = %value;
            %plugin.changeTrans = true;
            
         //*** Handle the 'Rotation' field
         case 3:
            %plugin.static.rotation = %value;
            %plugin.changeTrans = true;
      }
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function CloneLinear_DefaultValues(%static)
   {
      %static.numclones = 1;
      %static.offset = "2 0 0";
      %static.scale = "1 1 1";
      %static.rotation = "0 0 0";
   }

   //*** Init the static object
   function CloneLinear_InitStatic(%static)
   {
      //*** Setup default values
      CloneLinear_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the tube instance to default values
   function CloneLinear_Reset(%inst)
   {
      //*** Setup default values
      CloneLinear_DefaultValues(%inst.static);
   }
};

tool.register("CloneLinear", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS()+tool.RFLAG_NOACTIVATION(), "Linear Clone" );
