//
// buttontest.cs
//
// Tests the button GUI.
//
// The static ScriptObject makes use of the following dynamic fields:
// init         - If 'true' indicates that the static has been initialized (needs to be done once)
//
// The plug-in instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
//
// Revision History:
// July 7, 2006			David Wyand		Created script file
//

package ButtonTest
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("ButtonTest: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         ButtonTest_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;
      
      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = false;
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeCenter = false;
      %plugin.changeParam = false;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_DRAWALLSAME(); // Set up  all views to use the same drawing buffer.  Our Draw() will only be called once.
      %inst.flagsApply = tool.AFLAG_NONE();                                            // No special actions when tool is applied
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("ButtonTest: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("ButtonTest: MouseDown(" @ %inst @ "," @ %event @ ")");

      return true;
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("ButtonTest: MouseMove(" @ %inst @ "," @ %event @ ")");
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("ButtonTest: MouseUp(" @ %inst @ "," @ %event @ ")");
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("ButtonTest: HandleCount(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** If we're not yet active, return 0 to have the handles initialized by
      //*** HandleInit().  Otherwise, return the number of handles the user may interact with.
      return 0; // Not used for button test
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse button has just been
   // depressed.  %event will contain the particulars of where the mouse button was pressed
   // to allow for the tool to set itself up for the first time.  The value returned is the
   // (zero-based) index of the handle that is now active and will be dragged by the user
   // until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("ButtonTest: HandleInit(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;
      
      //*** Return the end handle index
      return 0;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine which handle
   // has been selected.  In both cases, %info is a ScriptObject that contains the
   // .pos[3] fields that are to be filled in with the requested handle's position.
   // This function returns the priority of the handle, the higher the number the higher
   // the priority.  This is used to determine which handle should be selected when two
   // or more handles overlap on the screen.  If -1 is returned, then the handle is
   // considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("ButtonTest: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** The given handle index is not one of ours
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("ButtonTest: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;
            
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      return %hindex;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("ButtonTest: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class and is
   // used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("ButtonTest: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%plugin.active)
         return;
      
      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("ButtonTest: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("ButtonTest: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.active = false;
      %plugin.changeParam = false;
      %plugin.changeCenter = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("ButtonTest: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("ButtonTest: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeParam = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            ButtonTest_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeParam = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("ButtonTest: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Button Test");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Test", "button");
      %form.addField( 1, "Script Test", "scriptbutton");

      //*** Add a script command to the "Script Test" button
      %form.addFieldListItem(1, "echo(\"This has been called from the script button\");");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("ButtonTest: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Test' button
         case 0:
            return 0;

         //*** Handle the 'Script Test' button
         case 1:
            return 0;
      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("ButtonTest: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'Test' button
         case 0:
            warn("!!! Button test!");

         //*** Handle the 'Script Test' button
         case 1:
            //*** NOTE: We should never get this notification as script buttons don't call home -- they're all grown up.
            warn("!!! Script Button test!");
      }

      //*** Make sure we're active
      %plugin.active = true;
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function ButtonTest_DefaultValues(%static)
   {
   }

   //*** Init the static object
   function ButtonTest_InitStatic(%static)
   {
      //*** Setup default values
      ButtonTest_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the tube instance to default values
   function ButtonTest_Reset(%inst)
   {
      //*** Setup default values
      ButtonTest_DefaultValues(%inst.static);
   }
};

tool.register("ButtonTest", tool.typeInteractive(), tool.RFLAG_NONE(), "ButtonTest" );
