//
// backdrop.cs
//
// Defines the Backdrop tool plug-in to modify the viewport backdrops of the current scene.
//
// The static ScriptObject makes use of the following dynamic fields:
// init     - If 'true' indicates that the static has been initialized (needs to be done once)
// lastview - Integer representing the last viewport (display view) that was modified.
//
// The plug-in instance makes use of the following dynamic fields:
// static       - Points to the static ScriptObject
// handlepos    - Defines the xyz center point of the user controlable handle as a 3 element array
// dirty        - Flag to indicate that the tool needs to be refreshed on screen
// active       - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update       - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.
// changeCenter - Flag to indicate that the tool's center (origin) position has changed
// changeView   - Flag to indicate that the tool's entity type has changed
// lastimage    - Integer representing the last image that was selected
//
// Revision History:
// February 1, 2006			David Wyand		Created script file
//

package Backdrop
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("Backdrop: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }
      
      //*** Has the static been set up?
      if(!%static.init)
      {
         Backdrop_InitStatic(%static);
      }
      
      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** Attach the static object to the instance.  This is to share some properties
      //*** such as the instance's centre and size
      %plugin.static = %static;

      //*** Setup the handles
      Backdrop_RebuildHandles(%plugin);
      
      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeCenter = false;
      %plugin.changeView = false;
      %plugin.lastimage = -1;
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_NOTOOLCURSOR(); // Set up the tool to not change the mouse cursor and not have any Tool Properties Form buttons
      %inst.flagsApply = tool.AFLAG_NONE(); // Geometry selections are not modified by this tool
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("Backdrop: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // MouseDown()
   //
   // This function is called allow the tool to process a mouse down event.  Returning
   // false indicates that the handles should be used rather than raw mouse handling.
   function Plugin::MouseDown(%this, %inst, %event)
   {
      //error("Backdrop: MouseDown(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;

      //*** Change to the view that the user has clicked in
      switch(%event.view)
      {
         case 0:
            if(%event.viewDir)
            {
               %plugin.static.lastview = 5; // Right
               
            } else
            {
               %plugin.static.lastview = 4; // Left
            }
         
         case 1:
            if(%event.viewDir)
            {
               %plugin.static.lastview = 2; // Front
               
            } else
            {
               %plugin.static.lastview = 3; // Back
            }
         
         case 2:
            if(%event.viewDir)
            {
               %plugin.static.lastview = 0; // Top
               
            } else
            {
               %plugin.static.lastview = 1; // Bottom
            }
         
         case 3:
            %plugin.static.lastview = 6; // Prespective
      }
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      //*** Find out if the mouse is over a handle
      %result = tool.isMouseOverHandle(%event);
      if(%result == -1)
      {
         //*** A mouse click in the view but not on a handle
         return true;
      
      } else
      {
         //*** The mouse is over a handle so let the handle system do its thing.
         return false;
      }
   }

   //************************************************************************************
   // MouseDrag()
   //
   // This function is called allow the tool to process a mouse drag event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseDrag(%this, %inst, %event)
   {
      //error("Backdrop: MouseMove(" @ %inst @ "," @ %event @ ")");
   }

   //************************************************************************************
   // MouseUp()
   //
   // This function is called allow the tool to process a mouse up event.  It will only
   // be called if MouseDown() doesn't return false.  This function doesn't return a value.
   function Plugin::MouseUp(%this, %inst, %event)
   {
      //error("Backdrop: MouseUp(" @ %inst @ "," @ %event @ ")");
   }
   
   //************************************************************************************
   // HandleCount()
   //
   // Returns the number of user controlable handles.  These allow the user to
   // graphically interact with the tool.  If this function returns 0, then
   // the HandleInit() function will be called to set the initial
   // handle points.
   function Plugin::HandleCount(%this, %inst, %event)
   {
      //error("Backdrop: HandleCount(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;
      
      //*** We always have 5 handles for the current view.  One for the center
      //*** and four for the edges.
      return 5;
   }
   
   //************************************************************************************
   // HandleInit()
   //
   // Called when the HandleCount() function returns 0 and the mouse
   // button has just been depressed.  %event will contain the particulars of where
   // the mouse button was pressed to allow for the tool to set itself up for the
   // first time.  The value returned is the (zero-based) index of the handle that is now
   // active and will be dragged by the user until the mouse button is released.
   function Plugin::HandleInit(%this, %inst, %event)
   {
      //error("Backdrop: HandleInit(" @ %inst @ "," @ %event @ ")");

      %plugin = %inst.instance;
      
      //*** There are always active handles so no need to initialize.  In fact,
      //*** this method should never be called.
      
      return -1;
   }
   
   //************************************************************************************
   // Handle()
   //
   // This function is called under a couple of different circumstances.  The first is
   // when the mouse button is held down and the mouse dragged.  In this case %hindex
   // contains the index to the handle that is being manipulated by the user.  This
   // function is also called right after the mouse button has been pressed for all of
   // the handles (as defined in HandleCount()) for the system to determine which handle
   // has been selected.  In both cases, %info is a ScriptObject that contains the
   // .pos[3] fields that are to be filled in with the requested handle's position.
   // This function returns the priority of the handle, the higher the number the higher
   // the priority.  This is used to determine which handle should be selected when two
   // or more handles overlap on the screen.  If -1 is returned, then the handle is
   // considered disabled and will not take part in user selections (and %info.pos[3]
   // need not be filled in).
   function Plugin::Handle(%this, %inst, %event, %hindex, %info)
   {
      //error("Backdrop: Handle(" @ %inst @ "," @ %event @ "," @ %hindex @ "," @ %info @ ")");

      %plugin = %inst.instance;
      
      //*** Fill in the handle's position and return its priority
      if(%hindex == 0)
      {
         //*** Center
         %scene = scene.getCurrent();
         %center = %scene.getVBCenter(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
         
         if(%event.view != 3)
         {
            //*** Calcualte the handle position for everything but perspective
            %pos = Backdrop_CalculateVBCenter(%view, %center);
         
         } else
         {
            //*** Handle the perspective view based on the backdrop projection
            %proj = %scene.getVBProjection(7) - 1; // The scene functions start at 1 rather than 0
            %pos = Backdrop_CalculateVBCenter(%proj, %center);
         }
         
         %info.pos[0] = getWord(%pos,0);
         %info.pos[1] = getWord(%pos,1);
         %info.pos[2] = getWord(%pos,2);

         //*** Return the handle's priority
         return %hindex; // set the handle's priority the same as its index
         
      } else if(%hindex == 1)
      {
         //*** Left
//         %scene = scene.getCurrent();
//         %size = %scene.getVBSize(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
//         %center = %scene.getVBCenter(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
         
      }
      
      //*** The given handle index is not one of ours
      return -1;
   }

   //************************************************************************************
   // HandleMoved()
   //
   // This function is called when the mouse moves and a handle is being dragged.  The
   // %hindex is the (zero-based) index of the handle that is being adjusted.  The value
   // returned is the index of the handle that should continue being moved -- usually this
   // is the same as %hindex.
   function Plugin::HandleMoved(%this, %inst, %event, %hindex)
   {
      //error("Backdrop: HandleMoved(" @ %inst @ "," @ %event @ "," @ %hindex @ ")");

      %plugin = %inst.instance;
      
      //*** Move the appropriate handle.
      if(%hindex == 0) // Center handle
      {
         %scene = scene.getCurrent();
         if(%event.view == 3)
         {
            //*** Perspective view.  What we do depends on the current projection for the view
            %view = %scene.getVBProjection(%plugin.static.lastview+1) - 1; // The scene functions start at 1 rather than 0
            if(%view == 0 || %view == 1)
            {
               //*** Top or bottom
               %point = %event.getMousePointOnWorldZPlane();
               %pos = getWord(%point,0) SPC getWord(%point, 1);
            
            } else if(%view == 2 || %view == 3)
            {
               //*** Front or back
               %point = %event.getMousePointOnWorldYPlane();
               %pos = getWord(%point,0) SPC getWord(%point, 2);
            
            } else if(%view == 4 || %view == 5)
            {
               //*** Left or right
               %point = %event.getMousePointOnWorldXPlane();
               %pos = getWord(%point,1) SPC getWord(%point, 2);
            }
            
            %scene.setVBCenter((%plugin.static.lastview+1), %pos); // The scene functions start at 1 rather than 0
         
         } else if(%event.view == 0)
         {
            %pos = getWord(%event.pos, 1) SPC getWord(%event.pos,2);
            %scene.setVBCenter((%plugin.static.lastview+1), %pos); // The scene functions start at 1 rather than 0
            
         } else if(%event.view == 1)
         {
            %pos = getWord(%event.pos, 0) SPC getWord(%event.pos,2);
            %scene.setVBCenter((%plugin.static.lastview+1), %pos); // The scene functions start at 1 rather than 0
            
         } else if(%event.view == 2)
         {
            %pos = getWord(%event.pos, 0) SPC getWord(%event.pos,1);
            %scene.setVBCenter((%plugin.static.lastview+1), %pos); // The scene functions start at 1 rather than 0
         }
         
         
         //*** Record that the center is being changed
         %plugin.changeCenter = true;
      }
      
      //*** Rebuild all handles
      Backdrop_RebuildHandles(%plugin);
      
      //*** Notify that we need to redraw the plugin as well as geometry
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      %plugin.update = tool.EDIT_UPDATE();
      
      return %hindex;
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("Backdrop: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class and is
   // used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("Backdrop: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;

      //*** Draw the center handle
      %view = %draw.getView();
      %scene = scene.getCurrent();
      %center = %scene.getVBCenter(%view+1); // The scene functions start at 1 rather than 0
      
      if(%view <= 5)
      {
         //*** Calcualte the image position for everything but perspective
         %pos = Backdrop_CalculateVBCenter(%view, %center);
         
      } else
      {
         //*** Handle the perspective view based on the backdrop projection
         %proj = %scene.getVBProjection(7) - 1; // The scene functions start at 1 rather than 0
         %pos = Backdrop_CalculateVBCenter(%proj, %center);
      }
         
      %draw.handle(%draw.HANDLE_CUBE(), "0 255 255", %pos);

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("Backdrop: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("Backdrop: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.changeCenter = false;
      %plugin.changeView = false;
   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("Backdrop: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("Backdrop: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new primitive.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeCenter = true;
            %plugin.changeView = true;
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            Backdrop_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            %plugin.changeCenter = true;
            %plugin.changeView = true;
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has change the currently active texture.  If the tool is active, then
         //*** we tell Constructor to update our geometry.
         //case tool.EVENT_TEXTURECHANGE(): //*** NOTE: We don't need to handle a texture change.
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("Backdrop: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("View Backdrop");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "View", "popup");
      %form.addField( 3, "Texture", "popup");
      %form.addField( 1, "Center", "distanceHV");
      %form.addField( 2, "Size", "distanceHV");
      %form.addField( 10, "Angle", "angle");
      %form.addField( 9, "Transparent", "numeric");
//      %form.addField( -1, "","divider");
      %form.addField( 7, "Flip H", "checkbox");
      %form.addField( 8, "Flip V", "checkbox");
      %form.addField( 4, "Keep Aspect", "checkbox");
      %form.addField( 6, "Overlay", "checkbox");
      %form.addField( 5, "Projection", "popup");

      //*** Add items to the 'View' popup
      %form.addFieldListItem(0, "Top");
      %form.addFieldListItem(0, "Bottom");
      %form.addFieldListItem(0, "Front");
      %form.addFieldListItem(0, "Back");
      %form.addFieldListItem(0, "Left");
      %form.addFieldListItem(0, "Right");
      %form.addFieldListItem(0, "Perspective");

      //*** Add items to the 'Texture' popup
      Backdrop_BuildTextureList(3);

      //*** Add items to the 'Projection' popup
      %form.addFieldListItem(5, "Top");
      %form.addFieldListItem(5, "Bottom");
      %form.addFieldListItem(5, "Front");
      %form.addFieldListItem(5, "Back");
      %form.addFieldListItem(5, "Left");
      %form.addFieldListItem(5, "Right");
      
      //*** Set limits on some controls
      %form.setFieldMinLimit(9, 0);
      %form.setFieldMaxLimit(9, 1);

   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("Backdrop: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'View' field
         case 0:
            return %plugin.static.lastview;
            
         //*** Handle the 'Center' field
         case 1:
            %scene = scene.getCurrent();
            return %scene.getVBCenter(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Size' field
         case 2:
            %scene = scene.getCurrent();
            return %scene.getVBSize(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0

         //*** Handle the 'Texture' field
         case 3:
            %value = 0;
            %scene = scene.getCurrent();
            %index = %scene.getVBImageListIndex(%plugin.static.lastview + 1); // The scene functions start at 1 rather than 0
            if(%index >= 0)
            {
               %value = %index+2;
            }
            return %value;
            
         //*** Handle the 'Keep Aspect' field
         case 4:
            %scene = scene.getCurrent();
            return %scene.getVBKeepAspect(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0

         //*** Handle the 'Projection' field
         case 5:
            %value = 0;
            %scene = scene.getCurrent();
            return (%scene.getVBProjection(%plugin.static.lastview+1)-1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Overlay' field
         case 6:
            %scene = scene.getCurrent();
            return %scene.getVBOverlay(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Flip H' field
         case 7:
            %scene = scene.getCurrent();
            return %scene.getVBFlipH(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Flip V' field
         case 8:
            %scene = scene.getCurrent();
            return %scene.getVBFlipV(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Transparent' field
         case 9:
            %scene = scene.getCurrent();
            return %scene.getVBTransparency(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0
            
         //*** Handle the 'Angle' field
         case 10:
            %scene = scene.getCurrent();
            return %scene.getVBAngle(%plugin.static.lastview+1); // The scene functions start at 1 rather than 0

      }
      
      return 0;
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      //error("Backdrop: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      switch(%id)
      {
         //*** Handle the 'View' field
         case 0:
            %plugin.static.lastview = %value;
            %plugin.changeView = true;

         //*** Handle the 'Center' field
         case 1:
            %scene = scene.getCurrent();
            %scene.setVBCenter((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0
            
            %plugin.changeCenter = true;

         //*** Handle the 'Size' field
         case 2:
            %scene = scene.getCurrent();
            %scene.setVBSize((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Texture' field
         case 3:
            %scene = scene.getCurrent();
            if(%value == 0)
            {
               //*** [none]
               Backdrop_ChangeTexture(%plugin, -1);
               
            } else if(%value == 1)
            {
               //*** [load]
               cmdOpenLoadImageDialog("", Backdrop_LoadTextureOKCallback, Backdrop_LoadTextureCancelCallback, %plugin);
               
            } else if(%value >=2)
            {
               //*** Texture has been selected
               Backdrop_ChangeTexture(%plugin, (%value-2));
            }

         //*** Handle the 'Keep Aspect' field
         case 4:
            %scene = scene.getCurrent();
            %scene.setVBKeepAspect((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Projection' field
         case 5:
            %scene = scene.getCurrent();
            %scene.setVBProjection((%plugin.static.lastview+1), (%value+1)); // The scene functions start at 1 rather than 0

         //*** Handle the 'Overlay' field
         case 6:
            %scene = scene.getCurrent();
            %scene.setVBOverlay((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Flip H' field
         case 7:
            %scene = scene.getCurrent();
            %scene.setVBFlipH((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Flip V' field
         case 8:
            %scene = scene.getCurrent();
            %scene.setVBFlipV((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Transparent' field
         case 9:
            %scene = scene.getCurrent();
            %scene.setVBTransparency((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

         //*** Handle the 'Angle' field
         case 10:
            %scene = scene.getCurrent();
            %scene.setVBAngle((%plugin.static.lastview+1), %value); // The scene functions start at 1 rather than 0

      }
      
      //*** Rebuild all handles if required
      if(%plugin.changeView || %plugin.changeCenter)
      {
         Backdrop_RebuildHandles(%plugin);
      }
      
      //*** Indicate that everything needs to be redrawn
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the given object to default values
   function Backdrop_DefaultValues(%static)
   {
      %static.center[0] = 0.0; // h
      %static.center[1] = 0.0; // v
      
      %static.lastview = 0;
   }

   //*** Init the static object
   function Backdrop_InitStatic(%static)
   {
      //*** Setup default values
      Backdrop_DefaultValues(%static);
      
      //*** Signal we're all setup.
      %static.init = true;
   }
   
   //*** Reset the tube instance to default values
   function Backdrop_Reset(%inst)
   {
      //*** Setup default values
      Backdrop_DefaultValues(%inst.static);
      Backdrop_RebuildHandles(%inst);
   }
   
   //*** Rebuild the handles
   function Backdrop_RebuildHandles(%inst)
   {
   }

   //*** Build the texture popup
   function Backdrop_BuildTextureList(%fieldID)
   {
      tool.clearFieldListItems(%fieldID);
      tool.addFieldListItem(%fieldID, "[none]");
      tool.addFieldListItem(%fieldID, "[load]");
      %scene = scene.getCurrent();
      %count = %scene.getILCount();
      for(%i=0; %i<%count; %i++)
      {
         %ilname = %scene.getILTextureFileName(%i);
         tool.addFieldListItem(%fieldID, %ilname);
      }
   }
   
   //*** Change the texture used for the last selected view.
   //*** If %imagenum == -1 then the displayed texture is cleared.
   function Backdrop_ChangeTexture(%plugin, %imagenum)
   {
      %plugin.lastimage = %imagenum;

      %scene = scene.getCurrent();
      if(%imagenum >= 0)
      {
         //*** Set the displayed texture
         %ilname = %scene.getILTextureFileName(%imagenum);
         %scene.setVBTexture((%plugin.static.lastview+1), %ilname);
         
      } else
      {
         //*** Clear the displayed texture
         %scene.setVBTexture((%plugin.static.lastview+1), "");
      }
   }

   //*** 'OK' Callback from the load texture dialog   
   function Backdrop_LoadTextureOKCallback(%filename, %plugin)
   {
      //*** Attempt to change the texture to the given one.  This also adds
      //*** the texture to the scene's image list.
      %scene = scene.getCurrent();
      %index = %scene.addILTexture(%filename); // Add to scene's image list and get its index
      
      //*** Rebuild the texture popup
      Backdrop_BuildTextureList(3);
      
      //*** Change the texture
      Backdrop_ChangeTexture(%plugin, %index);
      
      //*** Force a GUI refresh
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      tool.refresh();
   }

   //*** 'Cancel' Callback from the load texture dialog   
   function Backdrop_LoadTextureCancelCallback(%filename, %plugin)
   {
      //*** Go back to the previously selected texture
      Backdrop_ChangeTexture(%plugin, %plugin.lastimage);
      
      //*** Force a GUI refresh
      %plugin.update = tool.EDIT_UPDATE();
      %plugin.dirty = tool.DIRTY_APPEARANCE();
      tool.refresh();
   }
   
   //*** This helper function will return a 3D position for a
   //*** view's backdrop image based on the 2D center for the
   //*** given view.
   function Backdrop_CalculateVBCenter(%view, %center)
   {
      %pos = "0 0 0";
      
      if(%view == 0 || %view == 1)
      {
         //*** Top or bottom
         %pos = %center SPC "0";
            
      } else if(%view == 2 || %view == 3)
      {
         //*** Front or back
         %pos = getWord(%center,0) SPC "0" SPC getWord(%center,1);
            
      } else if(%view == 4 || %view == 5)
      {
         //*** Left or right
         %pos = "0" SPC %center;
            
      }
      
      return %pos;
   }
};

tool.register("Backdrop", tool.typeInteractive(), tool.RFLAG_NONE(), "Modify Backdrop" );
