//-----------------------------------------------------------------------------
// Torque Forest Kit
// Copyright (C) Sickhead Games, LLC
//-----------------------------------------------------------------------------


// ForestEditorGui Script Methods

function ForestEditorGui::setActiveTool( %this, %tool )
{
   if ( %tool == ForestTools->BrushTool )
      ForestEditTabBook.selectPage(0);
      
   Parent::setActiveTool( %this, %tool );
}

/// This is called by the editor when the active forest has
/// changed giving us a chance to update the GUI.
function ForestEditorGui::onActiveForestUpdated( %this, %forest, %createNew )
{
   %gotForest = isObject( %forest );

   // Give the user a chance to add a forest.
   if ( !%gotForest && %createNew )
   {
      MessageBoxYesNo(  "Forest", 
                        "There is not a Forest in this mission.  Do you want to add one?",
                        %this @ ".createForest();", "" );
      return;                                                         
   }  
}

/// Called from a message box when a forest is not found.
function ForestEditorGui::createForest( %this )
{
   if ( isObject( theForest ) )
   {
      error( "Cannot create a second 'theForest' Forest!" );
      return;
   }
   
   // Allocate the Forest and make it undoable.
   new Forest( theForest )
   {
      dataFile = "";
      parentGroup = "MissionGroup";
   };
   
   MECreateUndoAction::submit( theForest );
   
   ForestEditorInspector.inspect( theForest );
   
   EWorldEditor.isDirty = true;
}

function ForestEditorGui::newBrush( %this )
{   
   %internalName = getUniqueInternalName( "Brush", ForestBrushGroup, true );
         
   %brush = new ForestBrush()
   {
      internalName = %internalName;
      parentGroup = ForestBrushGroup; 
   };   
   
   MECreateUndoAction::submit( %brush );
   
   ForestEditBrushTree.open( ForestBrushGroup );
   ForestEditBrushTree.buildVisibleTree(true);
   %item = ForestEditBrushTree.findItemByObjectId( %brush );
   ForestEditBrushTree.clearSelection();
   ForestEditBrushTree.addSelection( %item );
   ForestEditBrushTree.scrollVisible( %item );   
   
   ForestEditorPlugin.dirty = true;
}

function ForestEditorGui::newElement( %this )
{
   %sel = ForestEditBrushTree.getSelectedObject();
   
   if ( !isObject( %sel ) )
      %parentGroup = ForestBrushGroup;
   else
   {
      if ( %sel.getClassName() $= "ForestBrushElement" )  
         %parentGroup = %sel.parentGroup;
      else
         %parentGroup = %sel;
   }
      
   %internalName = getUniqueInternalName( "Element", ForestBrushGroup, true );   
   
   %element = new ForestBrushElement()
   {
      internalName = %internalName;
      parentGroup =  %parentGroup;
   };
   
   MECreateUndoAction::submit( %element );
   
   ForestEditBrushTree.clearSelection();      
   ForestEditBrushTree.buildVisibleTree( true );
   %item = ForestEditBrushTree.findItemByObjectId( %element.getId() );
   ForestEditBrushTree.scrollVisible( %item );
   ForestEditBrushTree.addSelection( %item );  
   
   ForestEditorPlugin.dirty = true;
}

function ForestEditorGui::deleteBrushOrElement( %this )
{
   ForestEditBrushTree.deleteSelection();
   ForestEditorPlugin.dirty = true;
}

function ForestEditorGui::newMesh( %this )
{
   %spec = "All Mesh Files|*.dts;*.dae|DTS|*.dts|DAE|*.dae";
   
   %dlg = new OpenFileDialog()
   {
      Filters        = %spec;
      DefaultPath    = $Pref::WorldEditor::LastPath;
      DefaultFile    = "";
      ChangePath     = true;
   };
         
   %ret = %dlg.Execute();
   
   if ( %ret )
   {
      $Pref::WorldEditor::LastPath = filePath( %dlg.FileName );
      %fullPath = %dlg.FileName;
      %file = fileBase( %fullPath );
   }   
   
   %dlg.delete();
   
   if ( !%ret )
      return;
         
   %name = getUniqueName( %file );
      
   %str = "datablock TSForestItemData( " @ %name @ " ) { shapeFile = \"" @ %fullPath @ "\"; };";            
   eval( %str );
   
   if ( isObject( %name ) )
   {
      ForestEditMeshTree.clearSelection();
      ForestEditMeshTree.buildVisibleTree( true );
      %item = ForestEditMeshTree.findItemByObjectId( %name.getId() );
      ForestEditMeshTree.scrollVisible( %item );
      ForestEditMeshTree.addSelection( %item );
      
      ForestDataManager.setDirty( %name, "art/forest/managedItemData.cs" );  
      
      %element = new ForestBrushElement()
      {
         internalName = %name;
         forestItemData = %name;  
         parentGroup = ForestBrushGroup;
      };                       

      ForestEditBrushTree.clearSelection();      
      ForestEditBrushTree.buildVisibleTree( true );
      %item = ForestEditBrushTree.findItemByObjectId( %element.getId() );
      ForestEditBrushTree.scrollVisible( %item );
      ForestEditBrushTree.addSelection( %item );    
            
      pushInstantGroup();      
      %action = new MECreateUndoAction()
      {
         actionName = "Create TSForestItemData";
      };      
      popInstantGroup();
            
      %action.addObject( %name );
      %action.addObject( %element );            
      %action.addToManager( Editor.getUndoManager() );   
      
      ForestEditorPlugin.dirty = true;   
   }         
}

function ForestEditorGui::deleteMesh( %this )
{
   %obj = ForestEditMeshTree.getSelectedObject();   
   
   // Can't delete itemData's that are in use without
   // crashing at the moment...
      
   if ( isObject( %obj ) )
   {
      MessageBoxOKCancel( "Warning", 
                          "Deleting this mesh will also delete BrushesElements and ForestItems referencing it.", 
                          "ForestEditorGui.okDeleteMesh(" @ %obj @ ");",
                          "" );      
   }   
}

function ForestEditorGui::okDeleteMesh( %this, %mesh )
{
   // Submitting undo actions is handled in code.
   %this.deleteMeshSafe( %mesh );   
   
   // Update TreeViews.
   ForestEditBrushTree.buildVisibleTree( true );
   ForestEditMeshTree.buildVisibleTree( true );
   
   ForestEditorPlugin.dirty = true;
}



// Child-control Script Methods


function ForestEditMeshTree::onSelect( %this, %obj )
{
   ForestEditorInspector.inspect( %obj );
}

function ForestEditBrushTree::onRemoveSelection( %this, %obj )
{   
   %this.buildVisibleTree( true );
   ForestTools->BrushTool.collectElements();
   
   if ( %this.getSelectedItemsCount() == 1 )
      ForestEditorInspector.inspect( %obj );
   else
      ForestEditorInspector.inspect( "" );
}

function ForestEditBrushTree::onAddSelection( %this, %obj )
{
   %this.buildVisibleTree( true );
   ForestTools->BrushTool.collectElements();
   
   if ( %this.getSelectedItemsCount() == 1 )
      ForestEditorInspector.inspect( %obj );
   else
      ForestEditorInspector.inspect( "" );
}

function ForestEditTabBook::onTabSelected( %this, %text, %idx )
{
   %bbg = ForestEditorPalleteWindow.findObjectByInternalName("BrushButtonGroup");
   %mbg = ForestEditorPalleteWindow.findObjectByInternalName("MeshButtonGroup");
   
   %bbg.setVisible( false );
   %mbg.setVisible( false );
      
   if ( %text $= "Brushes" )
   {   
      %bbg.setVisible( true );
      %obj = ForestEditBrushTree.getSelectedObject();      
      ForestEditorInspector.inspect( %obj );
   }
   else if ( %text $= "Meshes" )
   {
      %mbg.setVisible( true );
      %obj = ForestEditMeshTree.getSelectedObject();
      ForestEditorInspector.inspect( %obj );      
   }
}

function ForestEditBrushTree::onDeleteSelection( %this )
{
   %list = ForestEditBrushTree.getSelectedObjectList();               
   
   MEDeleteUndoAction::submit( %list, true );
   
   ForestEditorPlugin.dirty = true;
} 

function ForestEditBrushTree::onDragDropped( %this )
{
   ForestEditorPlugin.dirty = true;
}

function ForestEditMeshTree::onDragDropped( %this )
{
   ForestEditorPlugin.dirty = true;
}

function ForestEditMeshTree::onDeleteObject( %this, %obj )
{
   // return true - skip delete.
   return true;
}

function ForestEditMeshTree::onDoubleClick( %this )
{
   %obj = %this.getSelectedObject();     
   
   %name = getUniqueInternalName( %obj.getName(), ForestBrushGroup, true );
   
   %element = new ForestBrushElement()
   {
      internalName = %name;
      forestItemData = %obj.getName();  
      parentGroup = ForestBrushGroup;
   };          
   
   //ForestDataManager.setDirty( %element, "art/forest/brushes.cs" );                 

   ForestEditBrushTree.clearSelection();      
   ForestEditBrushTree.buildVisibleTree( true );
   %item = ForestEditBrushTree.findItemByObjectId( %element );
   ForestEditBrushTree.scrollVisible( %item );
   ForestEditBrushTree.addSelection( %item );  
   
   ForestEditorPlugin.dirty = true;
}

function ForestEditBrushTree::handleRenameObject( %this, %name, %obj )
{   
   if ( %name !$= "" )
   {       
      %found = ForestBrushGroup.findObjectByInternalName( %name );
      if ( isObject( %found ) && %found.getId() != %obj.getId() )
      {
         MessageBoxOK( "Error", "Brush or Element with that name already exists.", "" );   
         
         // true as in, we handled it, don't rename the object.      
         return true;   
      }
   }      
   
   // Since we aren't showing any groups whens inspecting a ForestBrushGroup
   // we can't push this event off to the inspector to handle.

   //return GuiTreeViewCtrl::handleRenameObject( %this, %name, %obj );      
   
   
   // The instant group will try to add our
   // UndoAction if we don't disable it.   
   pushInstantGroup();

   %nameOrClass = %obj.getName();
   if ( %nameOrClass $= "" )
      %nameOrClass = %obj.getClassname();

   %action = new InspectorFieldUndoAction()
   {
      actionName = %nameOrClass @ "." @ "internalName" @ " Change";
      
      objectId = %obj.getId();
      fieldName = "internalName";
      fieldValue = %obj.internalName;
      arrayIndex = 0;
                  
      inspectorGui = "";
   };
   
   // Restore the instant group.
   popInstantGroup();
         
   %action.addToManager( Editor.getUndoManager() );
   EWorldEditor.isDirty = true;   
   
   return false;   
}

function ForestEditorInspector::inspect( %this, %obj )
{
   if ( isObject( %obj ) )
      %class = %obj.getClassName();
   
   %this.showObjectName = false;
   %this.showCustomFields = false;
   
   switch$ ( %class )
   {
      case "ForestBrush":   
         %this.groupFilters = "+NOTHING,-Ungrouped";      

      case "ForestBrushElement":
         %this.groupFilters = "+ForestBrushElement,-Ungrouped";

      case "TSForestItemData":   
         %this.groupFilters = "+Media,+Wind";

      default:
         %this.groupFilters = "";
   }
   
   Parent::inspect( %this, %obj );  
}

function ForestEditorInspector::onInspectorFieldModified( %this, %object, %fieldName, %oldValue, %newValue )
{
   // The instant group will try to add our
   // UndoAction if we don't disable it.   
   %instantGroup = $InstantGroup;
   $InstantGroup = 0;

   %nameOrClass = %object.getName();
   if ( %nameOrClass $= "" )
      %nameOrClass = %object.getClassname();

   %action = new InspectorFieldUndoAction()
   {
      actionName = %nameOrClass @ "." @ %fieldName @ " Change";
      
      objectId = %object.getId();
      fieldName = %fieldName;
      fieldValue = %oldValue;
                  
      inspectorGui = %this;
   };
   
   // Restore the instant group.
   $InstantGroup = %instantGroup; 
         
   %action.addToManager( Editor.getUndoManager() );
   
   if ( %object.getClassName() $= "TSForestItemData" )
      ForestDataManager.setDirty( %object );
      
   ForestEditorPlugin.dirty = true;
}

function ForestEditorInspector::onFieldSelected( %this, %fieldName, %fieldTypeStr, %fieldDoc )
{
   //FieldInfoControl.setText( "<font:ArialBold:14>" @ %fieldName @ "<font:ArialItalic:14> (" @ %fieldTypeStr @ ") " NL "<font:Arial:14>" @ %fieldDoc );
}
