
function getGem(%gemDB)
{
   if (!isObject(%gemDB))
   {
      // ugh
      error("error, invalid gem db passed to getGem()");
      return 0;
   }
   
   %gemDB = %gemDB.getId();
   
   %freeGroup = nameToId("FreeGems" @ %gemDB);
   if (isObject(%freeGroup) && %freeGroup.getCount() > 0)
   {
      //echo("allocating gem from pool");
      %gem = %freeGroup.getObject(0);
      %freeGroup.remove(%gem);
      %gem.setHidden(false);
      return %gem;
   } 
   
   //echo("allocating new gem");
   // spawn new Gem   
   %gem = new Item() {
      dataBlock = %gemDB;
      collideable = "0";
      static = "1";
      rotate = "1";
      permanent = "0";
      deleted = "0";
   };
   return %gem;
}

function freeGem(%gem)
{
   if (!isObject(%gem))
      return;
      
   if (!isObject(FreeGemGroups))
      new SimGroup(FreeGemGroups);
      
   %gemDB = %gem.getDatablock();
   %gemDB = %gemDB.getId();
   
   %freeGroupName = "FreeGems" @ %gemDB;
   %freeGroup = nameToId(%freeGroupName);
   if (!isObject(%freeGroup))
   {
      %freeGroup = new SimGroup(%freeGroupName);
      FreeGemGroups.add(%freeGroup);
   }
   
   //echo("freeing gem to pool");
   %freeGroup.add(%gem);
   %gem.setHidden(true);
}

function removeGem(%gem)
{
   freeGem(%gem);
   
   // refill gem groups if necessary
   if ($Server::ServerType $= "MultiPlayer")
      refillGemGroups();
}

function spawnGemAt(%spawnPoint, %includeLight)
{
   if (!isObject(%spawnPoint))
   {
      error("Unable to spawn gem at specified point: spawn point does not exist" SPC %spawnPoint);
      return 0;
   }
   
   // if it has a gem on it, that is not good
   if (isObject(%spawnPoint.gem) && !%spawnPoint.gem.isHidden())
   {
      error("Gem spawn point already has an active gem on it");
      return %spawnPoint.gem;
   }
     
   // see if the spawn point has a custom gem datablock
   %gemDB = %spawnPoint.getGemDataBlock();
   if (!%gemDB)
      %gemDB = "GemItem";
      
   %gem = getGem(%gemDB);
   %gem.setBuddy(%includeLight);
   
   %gem.setTransform(%spawnPoint.getTransform());
   
   // point the gem on the spawn point
   %spawnPoint.gem = %gem;
      
   return %gem;
}

function getRandomObject(%groupName)
{
   %group = nameToID(%groupName);

   %object = 0;
   
   if (%group != -1)
   {
      %count = %group.getCount();

      if (%count > 0)
      {
         %index = getRandom(%count - 1);
         %object = %group.getObject(%index);
      }
   }
   return %object;
}

// returns a random gem spawn point
function findGemSpawn()
{
   return getRandomObject("MissionGroup/GemSpawns");
}

// returns a random gem spawn group
function findGemSpawnGroup()
{
   return getRandomObject("GemSpawnGroups");
}

// test function
function gemSpawnReport()
{
   %group = nameToId("MissionGroup/GemSpawns");
   for (%i = 0; %i < %group.getCount(); %i++)
   {
      %spawn = %group.getObject(%i);
      if (isObject(%spawn.gem))
         echo(%spawn.gem.getDatablock().getName() SPC "object found on gem spawn point");
   }
}

// test function
function deleteSomeGemSpawnGroups()
{
   for (%i = GemSpawnGroups.getCount() - 1; %i > 0; %i--)
   {
      %x = GemSpawnGroups.getObject(%i);
      GemSpawnGroups.remove(%x);
      %x.delete();
   }
}

// returns a random gem spawn group that is sufficiently far from other active gem groups that still
// have gems in them.
function pickGemSpawnGroup()
{
   if (!isObject(ActiveGemGroups))
   {
      error("ActiveGemGroups is not an object");
      return findGemSpawnGroup();
   }
   
   %gemGroupRadius = $Server::GemGroupRadius;   
   // allow mission to override radius
   if (MissionInfo.gemGroupRadius > 0)
      %gemGroupRadius = MissionInfo.gemGroupRadius;
   echo("PickGemSpawnGroup: using radius" SPC %gemGroupRadius);
   
   // double the radius to make it that much more unlikely that we'll pick a group that is too close
   // to another active group
   %gemGroupRadius *= 2;
   
   // we'll make 6 attempts to find a group that is sufficiently far from other groups
   // after that we give up and use a random group
   %group = 0;
   for (%attempt = 0; %attempt < 6; %attempt++)
   {
      %group = findGemSpawnGroup();
      if (!isObject(%group))
      {
         error("findGemSpawnGroup returned non-object");
         return 0;
      }
      if (%group.getCount() == 0)
      {
         error("gem spawn group contains no spawn points");
         continue;
      }

      %groupSpawn = %group.getObject(0);
            
      // see if the spawn group is far enough from the primary spawn point of point of the active
      // gem groups
      %ok = true;
      for (%i = 0; %i < ActiveGemGroups.getCount(); %i++)
      {
         %active = ActiveGemGroups.getObject(%i);
         if (%active.getCount() == 0)
            continue;
         %gem = %active.getObject(0);
         if (VectorDist(%gem.getTransform(), %groupSpawn.getTransform()) < %gemGroupRadius)
         {
            %ok = false;
            break;
         }
      }

      if (%ok) 
         return %group;
   }
   
   // if we get here we did not find an appropriate group, just use a random one
   error("unable to find a spawn group that works with active gem groups, using random");
   return findGemSpawnGroup();
}

// re-fill any gem groups that are empty
function refillGemGroups()
{  
   if (!isObject(ActiveGemGroups))
   {
      error("ActiveGemGroups does not exist, can't refill gem groups");
      return;
   }
     
   for (%i = 0; %i < ActiveGemGroups.getCount(); %i++)
   {
      %gemGroup = ActiveGemGroups.getObject(%i);
      
      if (%gemGroup.getCount() == 0)
      {
         // pick a new spawnGroup for the group
         %spawnGroup = pickGemSpawnGroup();
         if (!isObject(%spawnGroup))
         {
            error("Unable to locate gem spawn group, aborting");
            break;
         }
         %gemGroup.spawnGroup = %spawnGroup;
         fillGemGroup(%gemGroup);
      }
   }
}

// fill the specified gem group with gems using spawn points from its spawnGroup
function fillGemGroup(%gemGroup)
{
   %start = getRealTime();
   
   if (!isObject(%gemGroup))
   {
      error("Can't fill gem group, supplied group is not an object:" SPC %gemGroup);
      return;
   }

   %spawnGroup = %gemGroup.spawnGroup;
   if (!isObject(%spawnGroup))
   {
      error("Can't fill gem group, it does not contain a valid spawn group:" SPC %gemGroup);
      return;
   }
   
   // it should be empty already but clear it out anyway
   %gemGroup.clear();
   
   // refill it using spawn points from its spawn group
   for (%i = 0; %i < %spawnGroup.getCount(); %i++)
   {
      %spawn = %spawnGroup.getObject(%i);
      
      // don't spawn duplicate gems
      if (isObject(%spawn.gem) && !%spawn.gem.isHidden())
         continue;
      
      // spawn a gem and light at the spawn point
      %gem = spawnGemAt(%spawn,true);
            
      // add gem to gemGroup
      %gemGroup.add(%gem);
   }
   
   echo("Took" SPC (getRealTime() - %start) @ "ms to fill gem groups");
   
   $LastFilledGemSpawnGroup = %gemGroup;
}

// setup the gem groups
function SetupGems(%numGemGroups)
{
   %start = getRealTime();
   
   // delete any active gem groups and their associated gems
   if (isObject(ActiveGemGroups))
      ActiveGemGroups.delete();
      
   // get gem group configuration params
   %gemGroupRadius = $Server::GemGroupRadius;
   %maxGemsPerGroup = $Server::MaxGemsPerGroup;
      
   // allow mission to override
   if (MissionInfo.gemGroupRadius > 0)
      %gemGroupRadius = MissionInfo.gemGroupRadius;
   if (MissionInfo.maxGemsPerGroup > 0)
      %maxGemsPerGroup = MissionInfo.maxGemsPerGroup;
   echo("SetupGems: using radius" SPC %gemGroupRadius SPC "and max gems per group" SPC %maxGemsPerGroup);
   
   // clear the "gem" field on each spawn point.  these may contain bogus but valid object ids that were
   // persisted by the mission editor   
   %group = nameToId("MissionGroup/GemSpawns");
   for (%i = 0; %i < %group.getCount(); %i++)
      %group.getObject(%i).gem = 0;
   
   // set up the gem spawn groups (this is done in engine code to improve performance).
   // it will create a GemSpawnGroups object that contains groups of clustered spawn points
   SetupGemSpawnGroups(%gemGroupRadius, %maxGemsPerGroup);
   
   // ActiveGemGroups contains groups of spawned gems.  the groups are populated using the spawn point
   // information from the ActiveGemGroups object.  when a group is empty, a new gem spawn group is 
   // selected for it, and then it is refilled with gems
   new SimGroup(ActiveGemGroups);
   MissionCleanup.add(ActiveGemGroups);
   
   // create the active gem groups
   for (%i = 0; %i < %numGemGroups; %i++)
   {
      %gemGroup = new SimGroup();
      ActiveGemGroups.add(%gemGroup);
   }
   
   // fill them up
   refillGemGroups();
   
   echo("Took" SPC (getRealTime() - %start) @ "ms to set up gems for mission");
}

function faceGems( %player )
{
   // In multi-player, set the position, and face gems
   if( $Server::ServerType $= "MultiPlayer" &&
       isObject( $LastFilledGemSpawnGroup ) &&
       $LastFilledGemSpawnGroup.getCount() )
   {
      %avgGemPos = 0;
      // Ok, for right now, just use the first gem in the group. In final implementation
      // do an average of all the positions of the gems, and look at that.
      for( %i = 0; %i < $LastFilledGemSpawnGroup.getCount(); %i++ )
      {
         %gem = $LastFilledGemSpawnGroup.getObject( %i );
         
         %avgGemPos = VectorAdd( %avgGemPos, %gem.getPosition() );
      }
      %avgGemPos = VectorScale( %avgGemPos, 1 / $LastFilledGemSpawnGroup.getCount() );
      
      %player.cameraLookAtPt( %avgGemPos );
   }
}