//------------------------------------------------------------------------------
// Torque        Engine
// Copyright (C) GarageGames.com, Inc.
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// global vars
//------------------------------------------------------------------------------

$OPTIONS_MENU::ROW_LOOK_X     = 0;  ///< Row to invert X-look
$OPTIONS_MENU::ROW_LOOK_Y     = 1;  ///< Row to invert Y-look
$OPTIONS_MENU::ROW_VOLUME     = 2;  ///< Row to set audio volume

// HAX -- localization strings
$OPTIONS_MENU::STRINGS["CameraX"] = "Camera X";
$OPTIONS_MENU::STRINGS["CameraY"] = "Camera Y";
$OPTIONS_MENU::STRINGS["Volume"] = "Volume";

$OPTIONS_MENU::STRINGS["CameraOptions"] = "Invert" TAB "Normal";

//------------------------------------------------------------------------------
// OptionsMenu methods
//------------------------------------------------------------------------------

/// Callback when this gui is added to the sim.
function OptionsMenu::onAdd(%this)
{
   // Camera X 
   %this.addRow($OPTIONS_MENU::STRINGS["CameraX"], 
      $OPTIONS_MENU::STRINGS["CameraOptions"], false, "", 0, -15);
   
   // Camera Y
   %this.addRow($OPTIONS_MENU::STRINGS["CameraY"], 
      $OPTIONS_MENU::STRINGS["CameraOptions"], false, "", 0, -15);

   %volumeList = %this.getSfxVolumeList();
   %this.addRow("Volume", %volumeList, false, "onVolumeChange", 0, -15, false);
}

/// Callback when the control wakes up.
function OptionsMenu::onWake(%this)
{
   %this.loadPrefs();
}

/// Initializes each row on this control to reflect the state of saved prefs.
function OptionsMenu::loadPrefs(%this)
{
   // init the x/y invert options
   %invX = ($pref::invertXCamera) ? 
      getField($OPTIONS_MENU::STRINGS["CameraOptions"], 0) :
      getField($OPTIONS_MENU::STRINGS["CameraOptions"], 1);
   %this.selectOption($OPTIONS_MENU::ROW_LOOK_X , %invX);
   
   %invY = ($pref::invertYCamera) ? 
      getField($OPTIONS_MENU::STRINGS["CameraOptions"], 0) :
      getField($OPTIONS_MENU::STRINGS["CameraOptions"], 1);
   %this.selectOption($OPTIONS_MENU::ROW_LOOK_Y , %invY);

   // init the master volume
   %volume = mRoundByFive($pref::Audio::masterVolume * 100);
   %this.selectOption($OPTIONS_MENU::ROW_VOLUME, %volume);
}

/// Builds a list of volume increments suitable for listing on the volume
/// control. List will be from 0 to 100, incrementing by 5.
function OptionsMenu::getSfxVolumeList(%this)
{
   %start = 0;
   %end = 100;
   %inc = 5;

   %list = "";
   for (%i = %start; %i <= %end; %i += %inc)
   {
      %list = %list TAB %i;
   }

   return trim(%list);
}

//------------------------------------------------------------------------------
// OptionsButtonHolder methods
//------------------------------------------------------------------------------

function OptionsButtonHolder::onWake(%this)
{
   %this.add(GamepadButtonsGui);

   GamepadButtonsGui.setButton($BUTTON_A, "Apply Changes", OptionsMenu.CallbackOnA);
   GamepadButtonsGui.setButton($BUTTON_B, "Go Back", OptionsMenu.CallbackOnB);
   GamepadButtonsGui.setButton($BUTTON_Y, "Revert Options", OptionsMenu.CallbackOnY);
}

/// Callback when the volume setting is changed.
///
/// \param %direction (string) "LEFT" or "RIGHT" indicating the direction the
/// option changed.
function onVolumeChange(%direction)
{
   %volume = (OptionsMenu.getCurrentOption($OPTIONS_MENU::ROW_VOLUME) / 100);
   sfxSetMasterVolume(%volume);
   // TODO: play sample audio blip for user feedback
}

//------------------------------------------------------------------------------
// global methods
//------------------------------------------------------------------------------

/// Applies the options that have been set.
function applyOptions()
{
   // set camera inversion options
   $pref::invertXCamera = 
      (getField($OPTIONS_MENU::STRINGS["CameraOptions"], 0) $= OptionsMenu.getCurrentOption($OPTIONS_MENU::ROW_LOOK_X));
   $pref::invertYCamera = 
      (getField($OPTIONS_MENU::STRINGS["CameraOptions"], 0) $= OptionsMenu.getCurrentOption($OPTIONS_MENU::ROW_LOOK_Y));
      
   // set the audio options
   %rawVolume = OptionsMenu.getCurrentOption($OPTIONS_MENU::ROW_VOLUME);
   $pref::Audio::masterVolume = %rawVolume / 100;
}

/// Reverts all options to the saved preferences.
function revertOptions()
{
   OptionsMenu.loadPrefs();
}

/// Determines if the list contains the given element.
///
/// \return (bool) True if the element is found in the list, false if it is not.
function listHasElement(%list, %element)
{
   %count = getFieldCount(%list);
   for (%i = 0; %i < %count; %i++)
   {
      %word = getField(%list, %i);
      if (%word $= %element)
      {
         return true;
      }
   }

   return false;
}

/// Rounds %n to the nearest whole integer ending in a multiple of five.
///
/// \param %n (int or float) The number to round.
/// \return (int) %n rounded to the nearest multiple of five.
function mRoundByFive(%n)
{
   return (mFloor((%n + 2.5) / 5) * 5);
}
