//-----------------------------------------------------------------------------
// Torque 3D
// Copyright (C) GarageGames.com, Inc.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//    Source groups.
//-----------------------------------------------------------------------------

singleton SFXDescription( AudioMaster );
singleton SFXSource( AudioChannelMaster )
{
   description = AudioMaster;
};

singleton SFXDescription( AudioChannel )
{
   sourceGroup = AudioChannelMaster;
};

singleton SFXSource( AudioChannelDefault )
{
   description = AudioChannel;
};
singleton SFXSource( AudioChannelGui )
{
   description = AudioChannel;
};
singleton SFXSource( AudioChannelEffects )
{
   description = AudioChannel;
};
singleton SFXSource( AudioChannelMessages )
{
   description = AudioChannel;
};
singleton SFXSource( AudioChannelMusic )
{
   description = AudioChannel;
};

// Start playback of the master channel.
AudioChannelMaster.play();

//-----------------------------------------------------------------------------
//    Master SFXDescriptions.
//-----------------------------------------------------------------------------

// Master description for interface audio.
singleton SFXDescription( AudioGui )
{
   volume         = 1.0;
   sourceGroup    = AudioChannelGui;
};

// Master description for game effects audio.
singleton SFXDescription( AudioEffect )
{
   volume         = 1.0;
   sourceGroup    = AudioChannelEffects;
};

// Master description for audio in notifications.
singleton SFXDescription( AudioMessage )
{
   volume         = 1.0;
   sourceGroup    = AudioChannelMessages;
};

// Master description for music.
singleton SFXDescription( AudioMusic )
{
   volume         = 1.0;
   sourceGroup    = AudioChannelMusic;
};

//-----------------------------------------------------------------------------
//    SFX Functions.
//-----------------------------------------------------------------------------

/// This initializes the sound system device from
/// the defaults in the $pref::SFX:: globals.
function sfxStartup()
{
   sfxShutdown();
   
   // The console builds should re-detect, by default, so that it plays nicely 
   // along side a PC build in the same script directory.
   
   if ( $platform $= "xenon" )
   {
      // TODO: Change this to always use FMOD      
      if($pref::SFX::provider $= "DirectSound" || 
         $pref::SFX::provider $= "OpenAL" ||
         $pref::SFX::provider $= "XAudio")
      {
         $pref::SFX::provider = "";
      }
      
      if($pref::SFX::provider $= "")
      {
         $pref::SFX::autoDetect = 1;
         
         warn( "Xbox360 is auto-detecting available sound providers..." ); 
         warn( "   - You may wish to alter this functionality before release (core/scripts/client/audio.cs)" );
      }
   }

   echo( "sfxStartup..." );
   
   // If we have a provider set, try initialize a device now.
   
   if( $pref::SFX::provider !$= "" )
   {
      if( sfxInit() )
         return;
      else
      {
         // Force auto-detection.
         $pref::SFX::autoDetect = true;
      }
   }

   // If enabled autodetect a safe device.

   if(    ( !isDefined( "$pref::SFX::autoDetect" ) || $pref::SFX::autoDetect )
       && sfxAutodetect() )
      return;
   
   // Failure.

   error( "   Failed to initialize device!\n\n" );
   
   $pref::SFX::provider = "";
   $pref::SFX::device   = "";
   
   return;
}


/// This initializes the sound system device from
/// the defaults in the $pref::SFX:: globals.
function sfxInit()
{
   sfxShutdown();
      
   // Start it up!
   %maxBuffers = $pref::SFX::useHardware ? -1 : $pref::SFX::maxSoftwareBuffers;
   if ( !sfxCreateDevice( $pref::SFX::provider, $pref::SFX::device, $pref::SFX::useHardware, %maxBuffers ) )
      return false;

   // This returns a tab seperated string with
   // the initialized system info.
   %info = sfxGetDeviceInfo();
   $pref::SFX::provider       = getField( %info, 0 );
   $pref::SFX::device         = getField( %info, 1 );
   $pref::SFX::useHardware    = getField( %info, 2 );
   %useHardware               = $pref::SFX::useHardware ? "Yes" : "No";
   %maxBuffers                = getField( %info, 3 );
   
   echo( "   Provider: "    @ $pref::SFX::provider );
   echo( "   Device: "      @ $pref::SFX::device );
   echo( "   Hardware: "    @ %useHardware );
   echo( "   Buffers: "      @ %maxBuffers );

   sfxSetMasterVolume( $pref::SFX::masterVolume );
   
   if( isDefined( "$pref::SFX::distanceModel" ) )
      sfxSetDistanceModel( $pref::SFX::distanceModel );
   if( isDefined( "$pref::SFX::dopplerFactor" ) )
      sfxSetDopplerFactor( $pref::SFX::dopplerFactor );
   if( isDefined( "$pref::SFX::rolloffFactor" ) )
      sfxSetRolloffFactor( $pref::SFX::rolloffFactor );

   //RDTODO
   for ( %channel = 1; %channel <= 8; %channel++ ) 
      sfxSetChannelVolume( %channel, $pref::SFX::channelVolume[ %channel ] );
      
   return true;
}


/// Destroys the current sound system device.
function sfxShutdown()
{
   // We're assuming here that a null info 
   // string means that no device is loaded.
   if ( sfxGetDeviceInfo() $= "" )
      return;

   sfxDeleteDevice();
}


/// Determines which of the two SFX providers is preferable.
function sfxCompareProvider( %providerA, %providerB )
{
   if( %providerA $= %providerB )
      return 0;
      
   switch$( %providerA )
   {
      // Always prefer FMOD over anything else.
      case "FMOD":
         return 1;
         
      // Prefer OpenAL over anything but FMOD.
      case "OpenAL":
         if( %providerB $= "FMOD" )
            return -1;
         else
            return 1;
            
      // As long as the XAudio SFX provider still has issues,
      // choose stable DSound over it.
      case "DirectSound":
         if( %providerB $= "FMOD" || %providerB $= "OpenAL" )
            return -1;
         else
            return 0;
            
      case "XAudio":
         if( %providerB !$= "FMOD" && %providerB !$= "OpenAL" && %providerB !$= "DirectSound" )
            return 1;
         else
            return -1;
         
      default:
         return -1;
   }
}


/// Try to detect and initalize the best SFX device available.
function sfxAutodetect()
{
   // Get all the available devices.
   
   %devices = sfxGetAvailableDevices();

   // Collect and sort the devices by preferentiality.
   
   %deviceTrySequence = new ArrayObject();
   %bestMatch = -1;
   %count = getRecordCount( %devices );
   for( %i = 0; %i < %count; %i++ )
   {
      %info = getRecord( %devices, %i );

      // Skip the null provider.
      %provider = getField( %info, 0 );
      if( stricmp( %provider, "null" ) == 0 )
         continue;
         
      %deviceTrySequence.push_back( %provider, %info );
   }
   
   %deviceTrySequence.sortfk( "sfxCompareProvider" );
   
   // Try the devices in order.
   
   %count = %deviceTrySequence.count();
   for( %i = 0; %i < %count; %i ++ )
   {
      %provider = %deviceTrySequence.getKey( %i );
      %info = %deviceTrySequence.getValue( %i );
      
      $pref::SFX::provider       = %provider;
      $pref::SFX::device         = getField( %info, 1 );
      $pref::SFX::useHardware    = getField( %info, 2 );
      
      // By default we've decided to avoid hardware devices as
      // they are buggy and prone to problems.
      $pref::SFX::useHardware = false;

      if( sfxInit() )
      {
         $pref::SFX::autoDetect = false;
         %deviceTrySequence.delete();
         return true;
      }
   }
   
   // Found no suitable device.
   
   error( "sfxAutodetect - Could not initialize a valid SFX device." );
   
   $pref::SFX::provider = "";
   $pref::SFX::device = "";
   $pref::SFX::useHardware = "";
   
   %deviceTrySequence.delete();
   
   return false;
}


//-----------------------------------------------------------------------------
//    Backwards-compatibility with old channel system.
//-----------------------------------------------------------------------------

// Volume channel IDs for backwards-compatibility.

$GuiAudioType        = 1;  // Interface.
$SimAudioType        = 2;  // Game.
$MessageAudioType    = 3;  // Notifications.
$MusicAudioType      = 4;  // Music.

function sfxOldChannelToGroup( %channel )
{
   switch( %channel )
   {
      case 1:  return AudioChannelGui;
      case 2:  return AudioChannelEffects;
      case 3:  return AudioChannelMessages;
      case 4:  return AudioChannelMusic;
   }
   
   return AudioChannelDefault;
}

function sfxGroupToOldChannel( %group )
{
   %id = %group.getId();
   if( %id == AudioChannelGui.getId() )
      return $GuiAudioType;
   else if( %id == AudioChannelEffects.getId() )
      return $SimAudioType;
   else if( %id == AudioChannelMessages.getId() )
      return $MessageAudioType;
   else if( %id == AudioChannelMusic.getId() )
      return $MusicAudioType;
      
   return 0;
}

function sfxSetMasterVolume( %volume )
{
   AudioChannelMaster.setVolume( %volume );
}

function sfxGetMasterVolume( %volume )
{
   return AudioChannelMaster.getVolume();
}

function sfxStopAll( %channel )
{
   sfxStop( sfxOldChannelToGroup( %channel ) );
}

function sfxGetChannelVolume( %channel )
{
   return sfxOldChannelToGroup( %channel ).getVolume();
}

function sfxSetChannelVolume( %channel, %volume )
{
   sfxOldChannelToGroup( %channel ).setVolume( %volume );
}

singleton SimSet( SFXPausedSet );


/// Pauses the playback of active sound sources.
/// 
/// @param %channels    An optional word list of channel indices or an empty 
///                     string to pause sources on all channels.
/// @param %pauseSet    An optional SimSet which is filled with the paused 
///                     sources.  If not specified the global SfxSourceGroup 
///                     is used.
///
/// @deprecated
function sfxPause( %channels, %pauseSet )
{
   // Did we get a set to populate?
   if ( !isObject( %pauseSet ) )
      %pauseSet = SFXPausedSet;
      
   %count = SFXSourceSet.getCount();
   for ( %i = 0; %i < %count; %i++ )
   {
      %source = SFXSourceSet.getObject( %i );

      %channel = sfxGroupToOldChannel( %source.getGroup() );
      if( %channels $= "" || findWord( %channels, %channel ) != -1 )
      {
         %source.pause();
         %pauseSet.add( %source );
      }
   }
}


/// Resumes the playback of paused sound sources.
/// 
/// @param %pauseSet    An optional SimSet which contains the paused sound 
///                     sources to be resumed.  If not specified the global 
///                     SfxSourceGroup is used.
/// @deprecated
function sfxResume( %pauseSet )
{
   if ( !isObject( %pauseSet ) )
      %pauseSet = SFXPausedSet;
                  
   %count = %pauseSet.getCount();
   for ( %i = 0; %i < %count; %i++ )
   {
      %source = %pauseSet.getObject( %i );
      %source.play();
   }

   // Clear our pause set... the caller is left
   // to clear his own if he passed one.
   %pauseSet.clear();
}
